package ru.yandex.direct.core.entity.bids.utils.autoprice;

import java.math.BigDecimal;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.auction.container.bs.KeywordBidBsAuctionData;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidCalculationType;
import ru.yandex.direct.core.entity.keyword.model.Place;

import static com.google.common.base.Preconditions.checkArgument;

@ParametersAreNonnullByDefault
public class SearchPriceWizard implements PriceWizard<KeywordBidBsAuctionData> {

    private Integer increasePercent;
    private Place basePosition;
    private final PriceRule priceRules;
    private final PriceCalculationStrategy calculationStrategy;

    private SearchPriceWizard(Integer increasePercent, Place basePosition,
                              SetAutoBidCalculationType calculationBasement) {
        this.increasePercent = increasePercent;
        this.basePosition = basePosition;
        // При расчёте цен из web-интерфейса третья позиция СР не должна учитываться
        priceRules = PriceRule.forApi();
        calculationStrategy = getPriceCalculationStrategy(calculationBasement);
    }

    @SuppressWarnings("WeakerAccess")
    public static SearchPriceWizard byDiff(Integer increasePercent, Place basePosition) {
        return new SearchPriceWizard(increasePercent, basePosition, SetAutoBidCalculationType.DIFF);
    }

    @SuppressWarnings("WeakerAccess")
    public static SearchPriceWizard byValue(Integer increasePercent, Place basePosition) {
        return new SearchPriceWizard(increasePercent, basePosition, SetAutoBidCalculationType.VALUE);
    }

    @Override
    public BigDecimal calcPrice(KeywordBidBsAuctionData auctionData) {
        checkArgument(auctionData != null
                        && auctionData.getPremium() != null
                        && auctionData.getGuarantee() != null,
                "Complete auctionData is required for search price calculation. Given: %s", auctionData);

        PriceInPosition priceInPosition = priceRules.describePlace(basePosition, auctionData);

        return calculationStrategy.calcNewPrice(priceInPosition);
    }

    private PriceCalculationStrategy getPriceCalculationStrategy(SetAutoBidCalculationType calculationBasement) {
        switch (calculationBasement) {
            case VALUE:
                return PriceCalculationStrategy.byValue(increasePercent);
            case DIFF:
                return PriceCalculationStrategy.byDiff(increasePercent);
            default:
                throw new IllegalArgumentException("Not supported value of getCalculatedBy: " + calculationBasement);
        }
    }

}
