package ru.yandex.direct.core.entity.bids.validation;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

/**
 * В этом классе собраны проверки на ручную и автобюджетную стратегии.
 */
public class BidsConstraints {

    private static final Set<Integer> ALLOWED_AUTOBUDGET_PRIORITIES;

    static {
        ALLOWED_AUTOBUDGET_PRIORITIES = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(1, 3, 5)));
    }

    /**
     * Проверка, что стратегия установлена для компании
     */
    public static <T extends BiddingShowCondition> Constraint<T, Defect> strategyIsSet(DbStrategy strategy) {
        return bid -> strategy == null
                ? new Defect<>(BidsDefects.Ids.STRATEGY_IS_NOT_SET)
                : null;
    }

    /**
     * Проверка, что для ручной стратегии задан price
     */
    public static Constraint<BigDecimal, Defect> searchPriceNotNullForManualStrategy(DbStrategy strategy) {
        return price -> price == null && strategy.isSearchManualBidManaging()
                ? new Defect<>(BidsDefects.Ids.SEARCH_PRICE_IS_NOT_SET_FOR_MANUAL_STRATEGY)
                : null;
    }

    /**
     * Проверка, что для ручной стратегии задан priceContext
     */
    public static Constraint<BigDecimal, Defect> contextPriceNotNullForManualStrategy(DbStrategy strategy) {
        return price -> price == null && strategy.isContextManualBidManaging()
                ? new Defect<>(BidsDefects.Ids.CONTEXT_PRICE_IS_NOT_SET_FOR_MANUAL_STRATEGY)
                : null;
    }

    /**
     * Проверка, что задано обязательное поле autoBudgetPriority при автобюджетной стратегии
     */
    public static Constraint<Integer, Defect> autoBudgetPriorityIsNotNullForAutoStrategy(DbStrategy strategy) {
        return priority -> priority == null && strategy.isAutoBudget()
                ? new Defect<>(BidsDefects.Ids.PRIORITY_IS_NOT_SET_FOR_AUTO_STRATEGY)
                : null;
    }

    /**
     * Проверка, что autoBudgetPriority выставлен только для автобюджетной стратегии
     */
    public static Constraint<Integer, Defect> autoBudgetPriorityIsAcceptedForStrategy(DbStrategy strategy) {
        return priority -> priority != null && !strategy.isAutoBudget()
                ? new Defect<>(
                BidsDefects.Ids.PRIORITY_WONT_BE_ACCEPTED_IN_CASE_OF_NOT_AUTO_BUDGET_STRATEGY)
                : null;
    }

    /**
     * Проверка, что autoBudgetPriority входит в диапазон допустимых значений
     */
    public static Constraint<Integer, Defect> autoBudgetPriorityIsAccepted() {
        return priority -> priority != null && !ALLOWED_AUTOBUDGET_PRIORITIES.contains(priority)
                ? new Defect<>(BidsDefects.Ids.PRIORITY_HAS_WRONG_VALUE)
                : null;
    }

    public static Constraint<BigDecimal, Defect> priceSearchIsAcceptedForStrategy(DbStrategy strategy) {
        return price -> {
            if (price != null) {
                if (strategy.isSearchStop()) {
                    return new Defect<>(
                            BidsDefects.Ids.BID_FOR_SEARCH_WONT_BE_ACCEPTED_SEARCH_IS_SWITCHED_OFF);
                } else if (strategy.isAutoBudget()) {
                    return new Defect<>(
                            BidsDefects.Ids.BID_FOR_SEARCH_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY);
                }
            }
            return null;
        };
    }

    public static Constraint<BigDecimal, Defect> priceContextIsAcceptedForStrategy(DbStrategy strategy) {
        return price -> {
            if (price == null) {
                return null;
            }
            if (!strategy.isDifferentPlaces()) {
                if (strategy.isNetStop()) {
                    return new Defect<>(
                            BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NET_IS_SWITCHED_OFF);
                } else {
                    return new Defect<>(
                            BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NOT_DIFFERENT_PLACES);
                }
            } else if (strategy.isAutoBudget()) {
                return new Defect<>(
                        BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY);
            }
            return null;
        };
    }

    public static Constraint<BigDecimal, Defect> priceContextWontApplyNetIsSwitchedOff() {
        return price -> {
            if (price == null) {
                return null;
            } else {
                return new Defect<>(BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NET_IS_SWITCHED_OFF);
            }
        };
    }
}
