package ru.yandex.direct.core.entity.bids.validation;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.service.BidValidationContainer;
import ru.yandex.direct.core.entity.bids.utils.BidStorage;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.autoBudgetPriorityIsNotNullForAutoStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceContextIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceSearchIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.SetBidConstraints.requiredBidsFieldManualStrategyValidator;

@ParametersAreNonnullByDefault
public class BidsValidator implements Validator<List<SetBidItem>, Defect> {

    private final CommonBidsValidator<SetBidItem> commonBidsValidator;
    private final BidValidationContainer<? extends BiddingShowCondition> bidValidationContainer;
    private boolean warnOnStrategyMismatch;
    private final BidStorage<? extends BiddingShowCondition> existingBids;

    public BidsValidator(BidValidationContainer<? extends BiddingShowCondition> bidValidationContainer,
                         boolean warnOnStrategyMismatch) {
        this.bidValidationContainer = bidValidationContainer;
        this.commonBidsValidator = new CommonBidsValidator<>(SetBidItem.class, bidValidationContainer);
        this.warnOnStrategyMismatch = warnOnStrategyMismatch;
        this.existingBids = new BidStorage<>(bidValidationContainer.getExistingShowCondition());
    }

    @Override
    public ValidationResult<List<SetBidItem>, Defect> apply(List<SetBidItem> setBidItems) {
        ListValidationBuilder<SetBidItem, Defect> lvb =
                ListValidationBuilder.of(setBidItems, Defect.class)
                        .checkEachBy(this::validateBid);

        return lvb.getResult();
    }

    private ValidationResult<SetBidItem, Defect> validateBid(SetBidItem setBidItem) {
        ItemValidationBuilder<SetBidItem, Defect> v = ModelItemValidationBuilder.of(setBidItem);
        v.checkBy(SetBidConstraints.positiveIdsValidator());
        // хотя бы один из ID должен быть задан, если предыдущие проверки прошли
        v.checkBy(commonBidsValidator.rightsForCampaignValidator(), When.isValid());

        v.checkBy(commonBidsValidator.campaignNotArchivedValidator(), When.isValid());

        v.checkBy(bidsValidator(), When.isValid());
//        todo ssdmitriev: remove after checking in realworld
//        v.checkBy(SetBidConstraints.requiredBidsFieldValidator(bidValidationContainer), When.isValid());

        return v.getResult();
    }

    /**
     * Валидатор проверяет, переданные параметры ставок валидны
     */
    private Validator<SetBidItem, Defect> bidsValidator() {
        return bid -> {
            ModelItemValidationBuilder<SetBidItem> v = ModelItemValidationBuilder.of(bid);
            DbStrategy strategy = bidValidationContainer.getCampaignStrategy(bid);
            List<? extends BiddingShowCondition> bids = existingBids.getBySelection(bid);
            Long adGroupId = null;
            if (!bids.isEmpty()) {
                adGroupId = bids.get(0).getAdGroupId();
            }
            AdGroupType adGroupType = bidValidationContainer.getAdGroupTypesByIds().get(adGroupId);

            if (bid.getShowConditionType().shouldCheckBidValueFor(Bid.PRICE)) {
                v.item(SetBidItem.PRICE_SEARCH)
                        .checkBy(priceSearchValidator(strategy, adGroupType))
                        .weakCheck(priceSearchIsAcceptedForStrategy(strategy), When.isTrue(warnOnStrategyMismatch));
            }

            if (bid.getShowConditionType().shouldCheckBidValueFor(Bid.PRICE_CONTEXT)) {
                v.item(SetBidItem.PRICE_CONTEXT)
                        .checkBy(priceContextValidator(strategy, adGroupType))
                        .weakCheck(priceContextIsAcceptedForStrategy(strategy), When.isTrue(warnOnStrategyMismatch));
            }

            if (bid.getShowConditionType().shouldCheckBidValueFor(Bid.PRICE_CONTEXT) || bid.getShowConditionType()
                    .shouldCheckBidValueFor(Bid.PRICE)) {
                v.checkBy(requiredBidsFieldManualStrategyValidator(bidValidationContainer), When.isValid());
            }

            if (bid.getShowConditionType().shouldCheckBidValueFor(Bid.AUTOBUDGET_PRIORITY)) {
                v.item(SetBidItem.AUTOBUDGET_PRIORITY)
                        .checkBy(autoBudgetPriorityValidator(strategy))
                        .weakCheck(autoBudgetPriorityIsAcceptedForStrategy(strategy),
                                When.isTrue(warnOnStrategyMismatch));
            }

            return v.getResult();
        };
    }

    private Validator<BigDecimal, Defect> priceSearchValidator(DbStrategy strategy, AdGroupType adGroupType) {
        return price -> {
            ItemValidationBuilder<BigDecimal, Defect> v = ItemValidationBuilder.of(price);
            v
                    .checkBy(new PriceValidator(bidValidationContainer.getClientWorkCurrency(), adGroupType),
                            When.notNull())
                    .check(searchPriceNotNullForOnlySearchManualStrategy(strategy));
            return v.getResult();
        };
    }

    private Validator<BigDecimal, Defect> priceContextValidator(DbStrategy strategy, AdGroupType adGroupType) {
        return price -> {
            ItemValidationBuilder<BigDecimal, Defect> v = ItemValidationBuilder.of(price);
            v
                    .checkBy(new PriceValidator(bidValidationContainer.getClientWorkCurrency(), adGroupType),
                            When.notNull())
                    .check(contextPriceNotNullForOnlyContextManualStrategy(strategy));
            return v.getResult();
        };
    }

    private Validator<Integer, Defect> autoBudgetPriorityValidator(DbStrategy strategy) {
        return priority -> {
            ItemValidationBuilder<Integer, Defect> v = ItemValidationBuilder.of(priority);
            v
                    .checkBy(new AutobudgetValidator2(), When.notNull())
                    .check(autoBudgetPriorityIsNotNullForAutoStrategy(strategy),
                            BidsDefects.fieldDoesNotMatchStrategy(
                                    new BidsDefects.BidsParams().withField(SetBidItem.AUTOBUDGET_PRIORITY)
                                            .withId(null)));
            return v.getResult();
        };
    }

    /**
     * требуем поисковую ставку, если включено ручное управление ставками только на поиске
     */
    private Constraint<BigDecimal, Defect> searchPriceNotNullForOnlySearchManualStrategy(DbStrategy strategy) {
        return price -> {
            if (price == null && strategy.isSearchManualBidManaging() &&
                    (strategy.isNetStop() || !strategy.isDifferentPlaces())) {
                return new Defect<>(BidsDefects.Ids.SEARCH_PRICE_IS_NOT_SET_FOR_MANUAL_STRATEGY);
            }
            return null;
        };
    }

    /**
     * требуем ставку в сетях, если включено ручное управление ставками только в сетях
     */
    private Constraint<BigDecimal, Defect> contextPriceNotNullForOnlyContextManualStrategy(DbStrategy strategy) {
        return price -> {
            if (price == null && strategy.isContextManualBidManaging() && strategy.isSearchStop()) {
                return new Defect<>(BidsDefects.Ids.CONTEXT_PRICE_IS_NOT_SET_FOR_MANUAL_STRATEGY);
            }
            return null;
        };
    }

    public static Constraint<Integer, Defect> autoBudgetPriorityIsAcceptedForStrategy(DbStrategy strategy) {
        return priority -> {
            if (priority != null && !strategy.isAutoBudget()) {
                return new Defect<>(
                        BidsDefects.Ids.PRIORITY_WONT_BE_ACCEPTED_IN_CASE_OF_NOT_AUTO_BUDGET_STRATEGY);
            }
            return null;
        };
    }
}
