package ru.yandex.direct.core.entity.bids.validation;

import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.service.BidValidationContainer;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.notFoundShowConditionByParameters;
import static ru.yandex.direct.core.entity.bids.validation.SetBidConstraints.getBidParams;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRightsCantWrite;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

/**
 * Набор валидаторов, которые используются при проверках разных запросов на изменение ставок
 */
class CommonBidsValidator<T extends Model & BidSelectionCriteria> {

    private final BidValidationContainer<? extends BiddingShowCondition> bidValidationContainer;
    private final ModelProperty<?, Long> campaignIdProperty;

    CommonBidsValidator(Class<T> clazz,
                        BidValidationContainer<? extends BiddingShowCondition> bidValidationContainer) {
        this.bidValidationContainer = bidValidationContainer;
        campaignIdProperty = getCampaignIdProperty(clazz);
    }

    private ModelProperty<?, Long> getCampaignIdProperty(Class<T> clazz) {
        ModelProperty<?, Long> a;
        if (clazz == SetBidItem.class) {
            a = SetBidItem.CAMPAIGN_ID;
        } else if (clazz == SetAutoBidItem.class) {
            a = SetAutoBidItem.CAMPAIGN_ID;
        } else {
            throw new IllegalArgumentException("Unexpected class in CommonBidsValidator constructor: " + clazz);
        }
        return a;
    }

    /**
     * Валидатор проверяет, что у пользователя есть права на используемую кампанию
     */
    Validator<T, Defect> rightsForCampaignValidator() {
        return (T setBidItem) -> {
            ItemValidationBuilder<T, Defect> v = ModelItemValidationBuilder.of(setBidItem);
            v.check(fromPredicate(b -> bidValidationContainer.getCampaignIdFor(b) != null,
                    notFoundShowConditionByParameters(getBidParams(setBidItem))))
                    .check(fromPredicate(
                            b -> bidValidationContainer.getVisibleCampaignIds()
                                    .contains(bidValidationContainer.getCampaignIdFor(b)),
                            notFoundShowConditionByParameters(getBidParams(setBidItem))),
                            When.isValid())
                    .check(fromPredicate(
                            b -> bidValidationContainer.getWritableCampaignIds()
                                    .contains(bidValidationContainer.getCampaignIdFor(b)),
                            noRightsCantWrite()),
                            When.isValid());
            return v.getResult();
        };
    }

    /**
     * Валидатор проверяет, что кампания не архивирована.
     */
    Validator<T, Defect> campaignNotArchivedValidator() {
        return (T setBidItem) -> {
            ItemValidationBuilder<T, Defect> v = ItemValidationBuilder.of(setBidItem);
            // Так как проверки выше на существование campaign прошли успешно, тут должен быть не null
            Long campaignId = checkNotNull(bidValidationContainer.getCampaignIdFor(setBidItem));

            Constraint<T, Defect> constraint = fromPredicate(
                    (T b) -> !bidValidationContainer.getCampaignParam(b, CampaignSimple::getStatusArchived, true),
                    BidsDefects
                            .badStatusCampaignArchivedOnUpdateBids(campaignIdProperty, campaignId));

            v.check(constraint, When.isValid());
            return v.getResult();
        };
    }


}
