package ru.yandex.direct.core.entity.bids.validation;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.regions.Region;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoFrontpageNoBrowserNewTabImmersionsInRegions;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoFrontpageNoDesktopImmersionsInRegions;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoFrontpageNoMobileImmersionsInRegions;
import static ru.yandex.direct.core.validation.defects.MoneyDefects.invalidValueCpmNotGreaterThan;
import static ru.yandex.direct.core.validation.defects.MoneyDefects.invalidValueCpmNotLessThan;
import static ru.yandex.direct.core.validation.defects.MoneyDefects.invalidValueNotGreaterThan;
import static ru.yandex.direct.core.validation.defects.MoneyDefects.invalidValueNotLessThan;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;

/**
 * Валидирует, что цена в рамкам [min, max].
 */
public class PriceValidator implements Validator<BigDecimal, Defect> {

    private final Currency workCurrency;
    private final AdGroupType adGroupType;
    private final CpmYndxFrontpageAdGroupPriceRestrictions cpmYndxFrontpageCurrencyValidationData;

    public PriceValidator(Currency workCurrency,
                          AdGroupType type) {
        this(workCurrency, type,
                new CpmYndxFrontpageAdGroupPriceRestrictions(workCurrency.getMinCpmPrice(),
                        workCurrency.getMaxCpmPrice())
                        .withClientCurrency(workCurrency));
    }

    public PriceValidator(Currency workCurrency,
                          AdGroupType type,
                          CpmYndxFrontpageAdGroupPriceRestrictions cpmYndxFrontpageCurrencyValidationData) {
        checkState(cpmYndxFrontpageCurrencyValidationData != null || type != AdGroupType.CPM_YNDX_FRONTPAGE,
                "Frontpage data shouldn't be null in price validation");
        this.workCurrency = workCurrency;
        this.adGroupType = type;
        this.cpmYndxFrontpageCurrencyValidationData = cpmYndxFrontpageCurrencyValidationData;
    }

    @Override
    public ValidationResult<BigDecimal, Defect> apply(BigDecimal price) {
        ItemValidationBuilder<BigDecimal, Defect> validationBuilder = ItemValidationBuilder.of(price);

        if (price == null) {
            return validationBuilder.getResult();
        }

        if (adGroupType == null) {
            return validationBuilder.getResult();
        }

        CurrencyCode code = workCurrency.getCode();

        BigDecimal minPrice;
        BigDecimal maxPrice;
        Defect minPriceDefect;
        Defect maxPriceDefect;

        switch (adGroupType) {
            case CPM_BANNER:
            case CPM_VIDEO:
            case CPM_INDOOR:
            case CPM_OUTDOOR:
            case CPM_GEO_PIN:
            case CPM_GEOPRODUCT:
                minPrice = workCurrency.getMinCpmPrice();
                maxPrice = workCurrency.getMaxCpmPrice();
                minPriceDefect = invalidValueCpmNotLessThan(Money.valueOf(minPrice, code));
                maxPriceDefect = invalidValueCpmNotGreaterThan(Money.valueOf(maxPrice, code));
                break;

            case CPM_YNDX_FRONTPAGE:
                minPrice = workCurrency.getMinCpmPrice()
                        .max(cpmYndxFrontpageCurrencyValidationData.getCpmYndxFrontpageMinPrice());
                maxPrice = workCurrency.getMaxPrice()
                        .min(cpmYndxFrontpageCurrencyValidationData.getCpmYndxFrontpageMaxPrice());
                minPriceDefect = invalidValueCpmNotLessThan(Money.valueOf(minPrice, code));
                maxPriceDefect = invalidValueCpmNotGreaterThan(Money.valueOf(maxPrice, code));

                //в зависимости от того, где: на мобильной или десктопной версии не будет показов у кампании на главной,
                //выдаём различные ворнинги(могут быть обоих типов)
                Map<FrontpageCampaignShowType, Function<List<Region>, Defect>> defectSupplier = ImmutableMap.of(
                        FrontpageCampaignShowType.FRONTPAGE,
                        regions -> geoFrontpageNoDesktopImmersionsInRegions(regions),
                        FrontpageCampaignShowType.FRONTPAGE_MOBILE,
                        regions -> geoFrontpageNoMobileImmersionsInRegions(regions),
                        FrontpageCampaignShowType.BROWSER_NEW_TAB,
                        regions -> geoFrontpageNoBrowserNewTabImmersionsInRegions(regions)
                );
                for (FrontpageCampaignShowType campaignShowType : defectSupplier.keySet()) {
                    Map<Long, BigDecimal> minPriceByRegion =
                            nvl(cpmYndxFrontpageCurrencyValidationData.getMinPriceByRegion().get(campaignShowType),
                                    emptyMap());
                    Map<Long, BigDecimal> maxPriceByRegion =
                            nvl(cpmYndxFrontpageCurrencyValidationData.getMaxPriceByRegion().get(campaignShowType),
                                    emptyMap());
                    Set<Long> minPriceExceeded = EntryStream.of(minPriceByRegion)
                            .filterValues(v -> v.compareTo(price) > 0)
                            .keys()
                            .toSet();
                    Set<Long> maxPriceExceeded = EntryStream.of(maxPriceByRegion)
                            .filterValues(v -> v.compareTo(price) < 0)
                            .keys()
                            .toSet();

                    //регионы, по которым выдаём предупреждения
                    Map<Long, Region> regionsWithWarnings =
                            EntryStream.of(cpmYndxFrontpageCurrencyValidationData.getRegionsById())
                                    .filterKeys(t -> minPriceExceeded.contains(t) || maxPriceExceeded.contains(t))
                                    .toMap();
                    //найдем для каждого региона regionsWithWarnings первый родительский, отличный от данного
                    Map<Long, Region> upToParentRegions = EntryStream.of(regionsWithWarnings)
                            .mapValues(region -> {
                                do {
                                    region = region.getParent();
                                } while (region != null && region.getId() != GLOBAL_REGION_ID &&
                                        !regionsWithWarnings.containsKey(region.getId()));
                                return region;
                            })
                            .filterValues(Objects::nonNull)
                            .toMap();
                    //фильтруем регионы, чтобы предупреждение для региона верхнего уровня не дублировалось на нижнее
                    //Для этого берём те регионы из regionsWithWarnings, для которых не нашлось региона в
                    // upToParentRegions
                    //или для которых найденный равен GLOBAL_REGION_ID, и regionsWithWarnings не содержит
                    // GLOBAL_REGION_ID
                    List<Region> regionsWithWarningsFiltered = EntryStream.of(regionsWithWarnings)
                            .filterKeys(t -> !upToParentRegions.containsKey(t) ||
                                    (upToParentRegions.get(t).getId() == GLOBAL_REGION_ID &&
                                            !regionsWithWarnings.containsKey(GLOBAL_REGION_ID)))
                            .values()
                            .toList();

                    validationBuilder.weakCheck(fromPredicate(v -> regionsWithWarningsFiltered.isEmpty(),
                            defectSupplier.get(campaignShowType).apply(regionsWithWarningsFiltered)));
                }
                break;

            case PERFORMANCE:
                minPrice = workCurrency.getMinCpcCpaPerformance();
                maxPrice = workCurrency.getMaxPrice();
                minPriceDefect = invalidValueNotLessThan(Money.valueOf(minPrice, code));
                maxPriceDefect = invalidValueNotGreaterThan(Money.valueOf(maxPrice, code));
                break;

            default:
                minPrice = workCurrency.getMinPrice();
                maxPrice = workCurrency.getMaxPrice();
                minPriceDefect = invalidValueNotLessThan(Money.valueOf(minPrice, code));
                maxPriceDefect = invalidValueNotGreaterThan(Money.valueOf(maxPrice, code));
        }

        return validationBuilder
                .check(fromPredicate(v -> v.compareTo(minPrice) >= 0, minPriceDefect))
                .check(fromPredicate(v -> v.compareTo(maxPrice) <= 0, maxPriceDefect))
                .getResult();
    }
}
