package ru.yandex.direct.core.entity.bids.validation;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.BidTargetType;
import ru.yandex.direct.core.entity.bids.container.SetAutoBidItem;
import ru.yandex.direct.core.entity.bids.container.SetAutoNetworkByCoverage;
import ru.yandex.direct.core.entity.bids.container.SetAutoParameterWithIncreasePercent;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByPosition;
import ru.yandex.direct.core.entity.bids.container.SetAutoSearchByTrafficVolume;
import ru.yandex.direct.core.entity.bids.container.SetBidSelectionType;
import ru.yandex.direct.core.entity.bids.container.ShowConditionType;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.service.BidValidationContainer;
import ru.yandex.direct.core.entity.bids.utils.BidStorage;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupTypeNotSupported;
import static ru.yandex.direct.core.entity.bids.container.BidUtils.containsSearch;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.detectSelectionType;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.bidChangeNotAllowedForBsRarelyLoadedAdGroup;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.dependentFieldMissing;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.fieldRequiredForContextStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.fieldRequiredForSearchStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.notFoundShowConditionByParameters;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.relevanceMatchCantBeUsedInSetAuto;
import static ru.yandex.direct.core.entity.bids.validation.SetBidConstraints.getBidParams;
import static ru.yandex.direct.core.entity.bids.validation.SetBidConstraints.positiveIdsValidator;
import static ru.yandex.direct.core.entity.retargeting.service.RequestSetBidType.getByBidSelectionCriteria;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxEnumSetSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.absentRequiredField;

/**
 * {@link Validator} для списка {@link SetAutoBidItem}.
 * Проверяет, что элементы в списке заданы и имеют уникальные идентификаторы.
 * Каждый элемент проверяется методом {@link #validateBid(SetAutoBidItem)}.
 */
public class SetAutoBidValidator implements Validator<List<SetAutoBidItem>, Defect> {

    /**
     * Типы групп, для которых не поддерживается метод setAuto.
     */
    private static final Set<AdGroupType> FORBIDDEN_ADGROUP_TYPES = ImmutableSet.<AdGroupType>builder()
            .add(AdGroupType.CONTENT_PROMOTION)
            .build();

    public static final int MAX_TARGET_TRAFFIC_VOLUME = 100;
    public static final int MIN_TARGET_TRAFFIC_VOLUME = 5;
    public static final int MAX_CONTEXT_COVERAGE = 100;
    public static final int MIN_CONTEXT_COVERAGE = 1;
    public static final int MIN_INCREASE_PERCENT = 0;
    public static final int MAX_INCREASE_PERCENT = 1000;


    private final BidStorage<Bid> existingBids;
    private final CommonBidsValidator<SetAutoBidItem> commonBidsValidator;
    private final BidValidationContainer<Bid> bidValidationContainer;
    private final boolean isKeywordBidsService;

    public SetAutoBidValidator(BidValidationContainer<Bid> bidValidationContainer, boolean isKeywordBidsService) {
        this.existingBids = new BidStorage<>(bidValidationContainer.getExistingShowCondition());
        commonBidsValidator =
                new CommonBidsValidator<>(SetAutoBidItem.class, bidValidationContainer);
        this.bidValidationContainer = bidValidationContainer;
        this.isKeywordBidsService = isKeywordBidsService;
    }

    @Override
    public ValidationResult<List<SetAutoBidItem>, Defect> apply(List<SetAutoBidItem> setAutoBidItem) {
        ListValidationBuilder<SetAutoBidItem, Defect> lvb =
                ListValidationBuilder.of(setAutoBidItem, Defect.class)
                        .checkEach(notNull())
                        .checkEach(unique(item -> getByBidSelectionCriteria(item).getBidCriteriaSupplier().apply(item)),
                                duplicatedObject(), When.isValid())
                        .checkEachBy(this::validateBid, When.isValid());

        return lvb.getResult();
    }

    private ValidationResult<SetAutoBidItem, Defect> validateBid(SetAutoBidItem setAutoBidItem) {
        ItemValidationBuilder<SetAutoBidItem, Defect> validationBuilder =
                ItemValidationBuilder.of(setAutoBidItem);

        validationBuilder
                .checkBy(requiredFieldValidator())
                .checkBy(oneRuleValidator(), When.isTrue(isKeywordBidsService))
                .checkBy(positiveIdsValidator())
                .checkBy(increasePercentValidator())
                .checkBy(contextCoverageValidator())
                .checkBy(trafficVolumeValidator())
                .checkBy(maxBidValidator())
                .checkBy(commonBidsValidator.rightsForCampaignValidator(), When.isValid())
                .checkBy(prohibitRelevanceMatchChangeByKeywordId(), When.isValid())
                .checkBy(hasMatchedExistingBid(), When.isValid())
                .checkBy(commonBidsValidator.campaignNotArchivedValidator(), When.isValid())
                .checkBy(strategyNotAutobudgetValidator(), When.isValid())
                .checkBy(scopeValidator(), When.isValid())
                .checkBy(adGroupTypeValidator(), When.isValid())
                .checkBy(adGroupNotBsRarelyLoadedValidator(), When.isValid());

        return validationBuilder.getResult();
    }

    /**
     * Проверка наличия обязательных полей в зависимости от сценария вычисления ставок.
     */
    private Validator<SetAutoBidItem, Defect> requiredFieldValidator() {
        return setAutoBidItem -> {
            ModelItemValidationBuilder<SetAutoBidItem> vb = ModelItemValidationBuilder.of(setAutoBidItem);
            vb.item(SetAutoBidItem.SCOPE)
                    .check(notNull(), absentRequiredField());
            if (vb.getResult().hasAnyErrors()) {
                // Если не задан scope, дальше проверки не выполняем
                return vb.getResult();
            }

            EnumSet<BidTargetType> scopes = setAutoBidItem.getScope();
            boolean contextPriceCalc = scopes.contains(BidTargetType.CONTEXT);
            if (contextPriceCalc) {
                ModelItemValidationBuilder<SetAutoNetworkByCoverage> ivb =
                        vb.modelItem(SetAutoBidItem.NETWORK_BY_COVERAGE);
                ivb.item(SetAutoNetworkByCoverage.CONTEXT_COVERAGE)
                        .check(notNull(), fieldRequiredForContextStrategy(
                                new BidsDefects.BidsParams()
                                        .withField(SetAutoNetworkByCoverage.CONTEXT_COVERAGE)));
            }

            boolean searchPriceCalc = scopes.contains(BidTargetType.SEARCH);
            if (searchPriceCalc) {

                ModelItemValidationBuilder<SetAutoSearchByPosition> ivb =
                        vb.modelItem(SetAutoBidItem.SEARCH_BY_POSITION);
                ivb.item(SetAutoSearchByPosition.POSITION)
                        .check(notNull(), fieldRequiredForSearchStrategy(
                                new BidsDefects.BidsParams().withField(SetAutoSearchByPosition.POSITION)));

                SetAutoSearchByPosition searchByPosition = checkNotNull(setAutoBidItem.getSearchByPosition(),
                        "searchByPosition is expected to be non null");
                if (searchByPosition.getIncreasePercent() != null) {
                    // Если указано, что надо увеличить ставку, то требуется указать, на основе чего считается это значение
                    ivb.item(SetAutoSearchByPosition.CALCULATED_BY)
                            .check(notNull(), dependentFieldMissing(
                                    new BidsDefects.BidsParams2Fields()
                                            .withField2(SetAutoSearchByPosition.INCREASE_PERCENT)
                                            .withField(SetAutoSearchByPosition.CALCULATED_BY)
                            ));
                }
            }

            boolean searchPriceCalcByTrafficVolume = scopes.contains(BidTargetType.SEARCH_BY_TRAFFIC_VOLUME);
            if (searchPriceCalcByTrafficVolume) {
                ModelItemValidationBuilder<SetAutoSearchByTrafficVolume> ivb =
                        vb.modelItem(SetAutoBidItem.SEARCH_BY_TRAFFIC_VOLUME);
                ivb.item(SetAutoSearchByTrafficVolume.TARGET_TRAFFIC_VOLUME)
                        .check(notNull(), fieldRequiredForSearchStrategy(
                                new BidsDefects.BidsParams()
                                        .withField(SetAutoSearchByTrafficVolume.TARGET_TRAFFIC_VOLUME)));
            }
            return vb.getResult();
        };
    }

    /**
     * Недопустимо указывать в setAuto несколько rule'ов.
     */
    private Validator<SetAutoBidItem, Defect> oneRuleValidator() {
        return setAutoBidItem -> {
            ModelItemValidationBuilder<SetAutoBidItem> vb = ModelItemValidationBuilder.of(setAutoBidItem);

            vb.item(SetAutoBidItem.SCOPE)
                    .check(maxEnumSetSize(1));

            return vb.getResult();
        };
    }

    /**
     * Проверка корректности значения ContextCoverage.
     * Допустимые значения: [1, 100]
     */
    private Validator<SetAutoBidItem, Defect> contextCoverageValidator() {
        return setAutoBidItem -> {
            ModelItemValidationBuilder<SetAutoBidItem> ivb = ModelItemValidationBuilder.of(setAutoBidItem);
            ivb.modelItem(SetAutoBidItem.NETWORK_BY_COVERAGE)
                    .item(SetAutoNetworkByCoverage.CONTEXT_COVERAGE)
                    .check(inRange(MIN_CONTEXT_COVERAGE, MAX_CONTEXT_COVERAGE));
            return ivb.getResult();
        };
    }

    /**
     * Проверка корректности значения TargetTrafficVolume.
     * Допустимые значения: [5, 100]
     */
    private Validator<SetAutoBidItem, Defect> trafficVolumeValidator() {
        return setAutoBidItem -> {
            ModelItemValidationBuilder<SetAutoBidItem> ivb = ModelItemValidationBuilder.of(setAutoBidItem);
            ivb.modelItem(SetAutoBidItem.SEARCH_BY_TRAFFIC_VOLUME)
                    .item(SetAutoSearchByTrafficVolume.TARGET_TRAFFIC_VOLUME)
                    .check(inRange(MIN_TARGET_TRAFFIC_VOLUME, MAX_TARGET_TRAFFIC_VOLUME));
            return ivb.getResult();
        };
    }

    /**
     * Проверка необязательного параметра IncreasePercent.
     * Допустимые значения: [0, 1000]
     */
    private Validator<SetAutoBidItem, Defect> increasePercentValidator() {
        return setAutoBidItem -> {
            ModelItemValidationBuilder<SetAutoBidItem> ivb = ModelItemValidationBuilder.of(setAutoBidItem);
            ivb.modelItem(SetAutoBidItem.NETWORK_BY_COVERAGE)
                    .item(SetAutoNetworkByCoverage.INCREASE_PERCENT)
                    .check(inRange(MIN_INCREASE_PERCENT, MAX_INCREASE_PERCENT));
            ivb.modelItem(SetAutoBidItem.SEARCH_BY_POSITION)
                    .item(SetAutoParameterWithIncreasePercent.INCREASE_PERCENT)
                    .check(inRange(MIN_INCREASE_PERCENT, MAX_INCREASE_PERCENT));
            ivb.modelItem(SetAutoBidItem.SEARCH_BY_TRAFFIC_VOLUME)
                    .item(SetAutoSearchByTrafficVolume.INCREASE_PERCENT)
                    .check(inRange(MIN_INCREASE_PERCENT, MAX_INCREASE_PERCENT));
            return ivb.getResult();
        };
    }

    /**
     * Проверка поля с ограничением ставки
     */
    private Validator<SetAutoBidItem, Defect> maxBidValidator() {
        return bid -> {
            List<Bid> bids = existingBids.getBySelection(bid);
            Long adGroupId = null;
            if (!bids.isEmpty()) {
                adGroupId = bids.get(0).getAdGroupId();
            }
            AdGroupType adGroupType = bidValidationContainer.getAdGroupTypesByIds().get(adGroupId);
            ModelItemValidationBuilder<SetAutoBidItem> v = ModelItemValidationBuilder.of(bid);
            PriceValidator priceValidator =
                    new PriceValidator(bidValidationContainer.getClientWorkCurrency(), adGroupType
                    );
            v.modelItem(SetAutoBidItem.NETWORK_BY_COVERAGE)
                    .item(SetAutoNetworkByCoverage.MAX_BID)
                    .checkBy(priceValidator);
            v.modelItem(SetAutoBidItem.SEARCH_BY_POSITION)
                    .item(SetAutoSearchByPosition.MAX_BID)
                    .checkBy(priceValidator);
            v.modelItem(SetAutoBidItem.SEARCH_BY_TRAFFIC_VOLUME)
                    .item(SetAutoSearchByTrafficVolume.MAX_BID)
                    .checkBy(priceValidator);
            return v.getResult();
        };
    }

    /**
     * Не допускается изменять ставку автотаргетинга в setAuto при явном указании его ID.
     */
    private Validator<SetAutoBidItem, Defect> prohibitRelevanceMatchChangeByKeywordId() {
        return setAutoBidItem -> {
            ItemValidationBuilder<SetAutoBidItem, Defect> ivb = ItemValidationBuilder.of(setAutoBidItem);
            SetBidSelectionType selectionType = detectSelectionType(setAutoBidItem);

            boolean selectionByKeywordId = selectionType == SetBidSelectionType.KEYWORD_ID;
            boolean matchingBidIsRelevanceMatch =
                    existingBidPresent(setAutoBidItem, ShowConditionType.RELEVANCE_MATCH);

            ivb.check(unconditional(relevanceMatchCantBeUsedInSetAuto(getBidParams(setAutoBidItem))),
                    When.isTrue(selectionByKeywordId && matchingBidIsRelevanceMatch));

            return ivb.getResult();
        };
    }

    /**
     * Проверка, что хотя бы одна ставка находится по заданному условию.
     */
    private Validator<SetAutoBidItem, Defect> hasMatchedExistingBid() {
        return setAutoBidItem -> {
            ItemValidationBuilder<SetAutoBidItem, Defect> ivb = ItemValidationBuilder.of(setAutoBidItem);
            boolean noBidsPresent = !existingBidPresent(setAutoBidItem, ShowConditionType.KEYWORD);
            boolean noRelevanceMatchBidsPresent =
                    !existingBidPresent(setAutoBidItem, ShowConditionType.RELEVANCE_MATCH);
            ivb.check(unconditional(notFoundShowConditionByParameters(getBidParams(setAutoBidItem))),
                    When.isTrue(noBidsPresent && noRelevanceMatchBidsPresent));
            return ivb.getResult();
        };
    }

    /**
     * @return {@code true}, если для заданного {@code showCondition} существуют ставки.
     */
    private boolean existingBidPresent(BidSelectionCriteria showCondition, ShowConditionType bidType) {
        return StreamEx.of(existingBids.getBySelection(showCondition))
                .map(Bid::getType)
                .anyMatch(bidType::equals);
    }

    /**
     * Валидатор проверяет, что статегия изменяемой кампании не автобюджетная
     */
    private Validator<SetAutoBidItem, Defect> strategyNotAutobudgetValidator() {
        return bid -> {
            ModelItemValidationBuilder<SetAutoBidItem> v = ModelItemValidationBuilder.of(bid);
            DbStrategy strategy = bidValidationContainer.getCampaignStrategy(bid);

            v.check(unconditional(BidsDefects.bidChangeNotAllowedForAutobudgetStrategy()),
                    When.isTrue(strategy.isAutoBudget()));

            return v.getResult();
        };
    }

    /**
     * Проверка того, что ставка для целевой площадки может быть изменена.
     * Эта проверка должна должна идти после {@link #strategyNotAutobudgetValidator()}
     */
    private Validator<SetAutoBidItem, Defect> scopeValidator() {
        return bid -> {
            ModelItemValidationBuilder<SetAutoBidItem> v = ModelItemValidationBuilder.of(bid);
            DbStrategy strategy = bidValidationContainer.getCampaignStrategy(bid);
            v.item(SetAutoBidItem.SCOPE)
                    .check(notEmptyCollection())
                    .checkBy(searchScopeValidator(strategy))
                    .checkBy(contextScopeValidator(strategy));
            return v.getResult();
        };
    }

    private Validator<SetAutoBidItem, Defect> adGroupTypeValidator() {
        return bid -> {
            List<Bid> bids = existingBids.getBySelection(bid);
            Long adGroupId = null;
            if (!bids.isEmpty()) {
                adGroupId = bids.get(0).getAdGroupId();
            }
            AdGroupType adGroupType = bidValidationContainer.getAdGroupTypesByIds().get(adGroupId);

            ModelItemValidationBuilder<SetAutoBidItem> vb = ModelItemValidationBuilder.of(bid);

            vb.check(unconditional(adGroupTypeNotSupported()),
                    When.isTrue(FORBIDDEN_ADGROUP_TYPES.contains(adGroupType)));

            return vb.getResult();
        };
    }

    /**
     * Валидатор проверяет, что у группы нет статуса "Мало показов"
     */
    private Validator<SetAutoBidItem, Defect> adGroupNotBsRarelyLoadedValidator() {
        return setAutoBidItem -> {
            SetBidSelectionType selectionType = detectSelectionType(setAutoBidItem);
            List<Bid> bids = existingBids.getBySelection(setAutoBidItem);
            Long adGroupId = null;
            if (!bids.isEmpty()) {
                adGroupId = bids.get(0).getAdGroupId();
            }
            Set<Long> bsRarelyLoadedAdgroupIds = bidValidationContainer.getBsRarelyLoadedAdGroupIds();
            boolean isBidFromBsRarelyLoadedAdGroup = selectionType != SetBidSelectionType.CAMPAIGN_ID &&
                    adGroupId != null && bsRarelyLoadedAdgroupIds.contains(adGroupId);

            ModelItemValidationBuilder<SetAutoBidItem> vb = ModelItemValidationBuilder.of(setAutoBidItem);

            vb.weakCheck(unconditional(bidChangeNotAllowedForBsRarelyLoadedAdGroup()),
                    When.isTrue(isBidFromBsRarelyLoadedAdGroup));

            return vb.getResult();
        };
    }

    /**
     * Проверяет соответствие {@code scope} запроса текущей стратегии для {@link BidTargetType#SEARCH}.
     */
    private Validator<EnumSet<BidTargetType>, Defect> searchScopeValidator(DbStrategy strategy) {
        return scopes -> {
            ValidationResult<EnumSet<BidTargetType>, Defect> result = new ValidationResult<>(scopes);

            if (containsSearch(scopes) && strategy.isSearchStop()) {
                if (scopes.contains(BidTargetType.CONTEXT)) {
                    result.addWarning(new Defect<>(
                            BidsDefects.Ids.BID_FOR_SEARCH_WONT_BE_ACCEPTED_SEARCH_IS_SWITCHED_OFF));
                } else {
                    result.addError(new Defect<>(
                            BidsDefects.Ids.BID_FOR_SEARCH_NOT_ALLOWED_SEARCH_IS_SWITCHED_OFF));
                }
            }
            return result;
        };
    }

    /**
     * Проверяет соответствие {@code scope} запроса текущей стратегии для {@link BidTargetType#CONTEXT}.
     */
    private Validator<EnumSet<BidTargetType>, Defect> contextScopeValidator(DbStrategy strategy) {
        return scopes -> {
            ValidationResult<EnumSet<BidTargetType>, Defect> result = new ValidationResult<>(scopes);

            if (scopes.contains(BidTargetType.CONTEXT)) {
                if (strategy.isNetStop()) {
                    if (containsSearch(scopes)) {
                        result.addWarning(new Defect<>(
                                BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NET_IS_SWITCHED_OFF));
                    } else {
                        result.addError(new Defect<>(
                                BidsDefects.Ids.BID_FOR_CONTEXT_NOT_ALLOWED_NET_IS_SWITCHED_OFF));
                    }
                } else if (!strategy.isDifferentPlaces()) {
                    if (containsSearch(scopes)) {
                        result.addWarning(new Defect<>(
                                BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NOT_DIFFERENT_PLACES));
                    } else {
                        result.addError(new Defect<>(
                                BidsDefects.Ids.BID_FOR_CONTEXT_NOT_ALLOWED_NOT_DIFFERENT_PLACES));
                    }
                }
            }
            return result;
        };
    }

}
