package ru.yandex.direct.core.entity.bids.validation;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

import ru.yandex.direct.core.entity.bids.container.BidSelectionCriteria;
import ru.yandex.direct.core.entity.bids.container.BidUtils;
import ru.yandex.direct.core.entity.bids.container.BiddingShowCondition;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.model.Bid;
import ru.yandex.direct.core.entity.bids.service.BidValidationContainer;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Arrays.asList;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdateContextPrice;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdatePrice;
import static ru.yandex.direct.core.entity.bids.utils.BidSelectionUtils.canUpdatePriority;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.mixedTypes;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.oneOfFieldsShouldBeSpecified;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.possibleOnlyOneField;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.requiredAtLeastOneOfFields;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.requiredAtLeastOneOfFieldsForAutobudgetStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.requiredAtLeastOneOfFieldsForManualStrategy;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

/**
 * В этом классе собраны проверки, применяемые к запросам на изменение ставок.
 */
public class SetBidConstraints {
    private static final List<ModelProperty<?, ?>> BIDS_IDS_FIELDS =
            asList(SetBidItem.ID, SetBidItem.AD_GROUP_ID, SetBidItem.CAMPAIGN_ID);

    public static BidsDefects.BidsParams getBidParams(BidSelectionCriteria setBidItem) {
        BidsDefects.BidsParams params = new BidsDefects.BidsParams();
        if (setBidItem.getCampaignId() != null) {
            return params.withField(SetBidItem.CAMPAIGN_ID).withId(setBidItem.getCampaignId());
        } else if (setBidItem.getAdGroupId() != null) {
            return params.withField(SetBidItem.AD_GROUP_ID).withId(setBidItem.getAdGroupId());
        } else if (setBidItem.getId() != null) {
            return params.withField(SetBidItem.ID).withId(setBidItem.getId());
        } else {
            return params;
        }
    }

    /**
     * Валидатор проверяет, что у переданного Bid'а указан ровно один Id (Id/AdGroupId/CampaignId)
     */
    public static <T extends BidSelectionCriteria> Validator<T, Defect> selectionFieldCountValidator() {
        return (T bid) -> ModelItemValidationBuilder.<T, Defect>of(bid)
                .check(atLeastOneSelectionFieldRequired())
                .check(noMoreThanOneOfSelectionFieldsPresent())
                .getResult();
    }

    /**
     * Проверка, что хотя в запросе задано не более одного selection-поля
     */
    public static <T extends BidSelectionCriteria> Constraint<T, Defect> noMoreThanOneOfSelectionFieldsPresent() {
        return fromPredicate((T bid) -> (BidUtils.fieldCount(bid) <= 1), possibleOnlyOneField(BIDS_IDS_FIELDS));
    }

    /**
     * Проверка, что хотя бы одно из selection-полей задано в запросе
     */
    public static <T extends BidSelectionCriteria> Constraint<T, Defect> atLeastOneSelectionFieldRequired() {
        return fromPredicate((T bid) -> (BidUtils.fieldCount(bid) > 0), oneOfFieldsShouldBeSpecified(BIDS_IDS_FIELDS));
    }

    /**
     * Constraint, который проверяет, что все Bid'ы одного типа.
     * Bid'ы с незаполненными полями игнорируются при проверке
     */
    public static <T extends BidSelectionCriteria> Constraint<List<T>, Defect> selectorsAreOfSameType() {
        return fromPredicate(bids -> {
            Optional<T> firstBid = bids.stream().findFirst();

            //noinspection SimplifiableIfStatement,OptionalIsPresent
            if (!firstBid.isPresent()) {
                return true;
            }

            return bids.stream()
                    .allMatch(bid -> idFieldsOfTheSameType(bid, firstBid.get()));
        }, mixedTypes());
    }

    private static boolean idFieldsOfTheSameType(BidSelectionCriteria setBidItem1, BidSelectionCriteria setBidItem2) {
        return Objects.equals(getBidParams(setBidItem1).getField(), getBidParams(setBidItem2).getField());
    }

    public static Validator<SetBidItem, Defect> requiredBidsFieldManualStrategyValidator(
            BidValidationContainer<? extends BiddingShowCondition> bidsContainer) {
        return bid -> {
            if (bid == null) {
                return new ValidationResult<>(null);
            }
            ModelItemValidationBuilder<SetBidItem> v = ModelItemValidationBuilder.of(bid);
            int fieldsCount = 0;
            DbStrategy strategy = bidsContainer.getCampaignStrategyOptional(bid).get();
            List<ModelProperty<?, ?>> requiredFields = new ArrayList<>();
            if (canUpdatePrice(strategy, bid.getShowConditionType())) {
                requiredFields.add(SetBidItem.PRICE_SEARCH);
                if (bid.getPriceSearch() != null) {
                    fieldsCount++;
                }
            }

            if (canUpdateContextPrice(strategy, bid.getShowConditionType())) {
                requiredFields.add(SetBidItem.PRICE_CONTEXT);
                if (bid.getPriceContext() != null) {
                    fieldsCount++;
                }
            }
            if (fieldsCount == 0 && !requiredFields.isEmpty()) {
                v.getResult().addError(requiredAtLeastOneOfFieldsForManualStrategy(requiredFields));
            }
            return v.getResult();
        };
    }

    public static Validator<SetBidItem, Defect> requiredBidsFieldValidator(
            BidValidationContainer<? extends BiddingShowCondition> bidsContainer) {
        return bid -> {
            if (bid == null) {
                return new ValidationResult<>(null);
            }

            ModelItemValidationBuilder<SetBidItem> v = ModelItemValidationBuilder.of(bid);
            int fieldsCount = 0;

            Optional<DbStrategy> strategyOptional = bidsContainer.getCampaignStrategyOptional(bid);
            DbStrategy strategy = strategyOptional.get();
            List<ModelProperty<?, ?>> requiredFields = new ArrayList<>();
            if (canUpdatePrice(strategy, bid.getShowConditionType())) {
                requiredFields.add(SetBidItem.PRICE_SEARCH);
                if (bid.getPriceSearch() != null) {
                    fieldsCount++;
                }
            }

            if (canUpdateContextPrice(strategy, bid.getShowConditionType())) {
                requiredFields.add(SetBidItem.PRICE_CONTEXT);
                if (bid.getPriceContext() != null) {
                    fieldsCount++;
                }
            }

            if (canUpdatePriority(strategy, bid.getShowConditionType())) {
                requiredFields.add(SetBidItem.AUTOBUDGET_PRIORITY);
                if (bid.getAutobudgetPriority() != null) {
                    fieldsCount++;
                }
            }

            if (fieldsCount == 0 && !requiredFields.isEmpty()) {
                // текст ошибки зависит от типа стратегии
                if (strategy.isAutoBudget()) {
                    v.getResult().addError(requiredAtLeastOneOfFieldsForAutobudgetStrategy(requiredFields));
                } else {
                    v.getResult().addError(requiredAtLeastOneOfFieldsForManualStrategy(requiredFields));
                }
            }

            return v.getResult();
        };
    }

    /**
     * Валидатор проверяет, переданные параметры ставок валидны
     * Требуется, чтобы хотя бы одно из value-полей было задано для заявки на изменение ставок
     */
    public static Validator<SetBidItem, Defect> bidsFieldValidator() {
        return bid -> {
            ModelItemValidationBuilder<SetBidItem> v = ModelItemValidationBuilder.of(bid);
            int fieldsCount = 0;

            List<ModelProperty<?, ?>> requiredFields = new ArrayList<>();
            if (bid.getShowConditionType().shouldCheckBidValueFor(Bid.PRICE)) {
                requiredFields.add(SetBidItem.PRICE_SEARCH);
                if (bid.getPriceSearch() != null) {
                    fieldsCount++;
                }
            }

            if (bid.getShowConditionType().shouldCheckBidValueFor(Bid.PRICE_CONTEXT)) {
                requiredFields.add(SetBidItem.PRICE_CONTEXT);
                if (bid.getPriceContext() != null) {
                    fieldsCount++;
                }
            }

            if (bid.getShowConditionType().shouldCheckBidValueFor(Bid.AUTOBUDGET_PRIORITY)) {
                requiredFields.add(SetBidItem.AUTOBUDGET_PRIORITY);
                if (bid.getAutobudgetPriority() != null) {
                    fieldsCount++;
                }
            }
            if (fieldsCount == 0) {
                v.getResult().addError(requiredAtLeastOneOfFields(requiredFields));
            }

            return v.getResult();
        };
    }

    /**
     * Валидатор проверяет, что переданные с {@link BidSelectionCriteria} идентификаторы
     * являются целыми положительными числами
     */
    public static <T extends Model & BidSelectionCriteria> Validator<T, Defect> positiveIdsValidator() {
        return (T item) -> {
            ModelItemValidationBuilder<T> v = ModelItemValidationBuilder.of(item);
            if (item == null) {
                return v.getResult();
            }

            // тут нельзя использовать ModelProperty, поскольку типы моделей у нас разные: SetBidItem и SetAutoBidItem
            v.item(item.getId(), "id")
                    .check(validId(), When.notNull());
            v.item(item.getAdGroupId(), "adGroupId")
                    .check(validId(), When.notNull());
            v.item(item.getCampaignId(), "campaignId")
                    .check(validId(), When.notNull());
            return v.getResult();
        };
    }
}
