package ru.yandex.direct.core.entity.brandlift.repository;

import java.util.Collection;
import java.util.List;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.brandSurvey.BrandSurvey;
import ru.yandex.direct.dbschema.ppc.tables.records.BrandSurveyRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.tables.BrandSurvey.BRAND_SURVEY;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class BrandSurveyRepository {

    private static final Logger logger = LoggerFactory.getLogger(BrandSurveyRepository.class);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<BrandSurvey> brandSurveyMapper;

    public BrandSurveyRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        brandSurveyMapper = createMapper();
    }

    public void addBrandSurvey(int shard, BrandSurvey brandSurvey) {

        InsertHelper<BrandSurveyRecord> insertHelper = new InsertHelper<>(dslContextProvider.ppc(shard), BRAND_SURVEY);

        insertHelper.add(brandSurveyMapper, brandSurvey).newRecord();

        int rowsInserted = insertHelper.execute();
        logger.debug("Rows inserted: {}", rowsInserted);
    }

    public void renameBrandSurvey(int shard, String brandSurveyId, String newName) {
        var updatedRows = dslContextProvider.ppc(shard)
                .update(BRAND_SURVEY)
                .set(BRAND_SURVEY.NAME, newName)
                .where(BRAND_SURVEY.BRAND_SURVEY_ID.eq(brandSurveyId))
                .execute();
        if (updatedRows == 0) {
            logger.warn("No any updates for brand_survey with ID: {}", brandSurveyId);
        }
    }

    public List<BrandSurvey> getBrandSurvey(int shard, String brandSurveyId) {

        DSLContext dslContext = dslContextProvider.ppc(shard);
        return dslContext
                .select(brandSurveyMapper.getFieldsToRead())
                .from(BRAND_SURVEY)
                .where(BRAND_SURVEY.BRAND_SURVEY_ID.eq(brandSurveyId))
                .fetch(brandSurveyMapper::fromDb);
    }

    public List<BrandSurvey> getBrandSurveys(int shard, Collection<String> brandSurveyIds) {

        DSLContext dslContext = dslContextProvider.ppc(shard);
        return dslContext
                .select(brandSurveyMapper.getFieldsToRead())
                .from(BRAND_SURVEY)
                .where(BRAND_SURVEY.BRAND_SURVEY_ID.in(brandSurveyIds))
                .fetch(brandSurveyMapper::fromDb);
    }

    public List<BrandSurvey> getBrandSurveysWithEmptyFields(int shard) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return dslContext
                .select(brandSurveyMapper.getFieldsToRead())
                .from(BRAND_SURVEY)
                .where(BRAND_SURVEY.EXPERIMENT_ID.isNull()
                        .or(BRAND_SURVEY.SEGMENT_ID.isNull())
                        .or(BRAND_SURVEY.EXPERIMENT_ID.eq(0L))
                        .or(BRAND_SURVEY.SEGMENT_ID.eq(0L)))
                .fetch(brandSurveyMapper::fromDb);
    }

    public List<BrandSurvey> getClientBrandSurveys(int shard, Long clientId) {

        DSLContext dslContext = dslContextProvider.ppc(shard);
        return dslContext
                .select(brandSurveyMapper.getFieldsToRead())
                .from(BRAND_SURVEY)
                .where(BRAND_SURVEY.CLIENT_ID.eq(clientId))
                .orderBy(BRAND_SURVEY.BRAND_SURVEY_ID)
                .fetch(brandSurveyMapper::fromDb);
    }

    private JooqMapperWithSupplier<BrandSurvey> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(BrandSurvey::new)
                .map(property(BrandSurvey.CLIENT_ID, BRAND_SURVEY.CLIENT_ID))
                .map(property(BrandSurvey.BRAND_SURVEY_ID, BRAND_SURVEY.BRAND_SURVEY_ID))
                .map(property(BrandSurvey.NAME, BRAND_SURVEY.NAME))
                .map(property(BrandSurvey.RETARGETING_CONDITION_ID, BRAND_SURVEY.RET_COND_ID))
                .map(convertibleProperty(BrandSurvey.IS_BRAND_LIFT_HIDDEN, BRAND_SURVEY.IS_BRAND_LIFT_HIDDEN,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLong))
                .map(property(BrandSurvey.EXPERIMENT_ID, BRAND_SURVEY.EXPERIMENT_ID))
                .map(property(BrandSurvey.SEGMENT_ID, BRAND_SURVEY.SEGMENT_ID))
                .build();
    }
}
