package ru.yandex.direct.core.entity.brandlift.service.targetestimation;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.inventori.service.InventoriServiceCore;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.inventori.InventoriClient;
import ru.yandex.direct.inventori.model.request.CampaignPredictionRequest;
import ru.yandex.direct.inventori.model.response.CampaignPredictionResponse;
import ru.yandex.direct.inventori.model.response.GeneralCampaignPredictionResponse;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@ParametersAreNonnullByDefault
@Component
public class TargetEstimationsService {
    private static final Logger logger = LoggerFactory.getLogger(TargetEstimationsService.class);
    private static final int GREEN_TRAFFIC_LIGHT = 2;

    private final InventoriClient inventoriClient;
    private final UserService userService;
    private final InventoriServiceCore inventoriServiceCore;
    private final DirectConfig brandLiftConfig;

    public TargetEstimationsService(
            InventoriClient inventoriClient,
            UserService userService,
            InventoriServiceCore inventoriServiceCore,
            DirectConfig directConfig
    ) {
        this.inventoriClient = inventoriClient;
        this.userService = userService;
        this.inventoriServiceCore = inventoriServiceCore;
        this.brandLiftConfig = directConfig.getBranch("brand_lift");
    }

    public List<TargetEstimation> getTargetEstimations(List<Campaign> campaigns) {
        List<Long> campaignIds = campaigns.stream()
                .filter(c -> c.getStrategy().getStrategyName() != StrategyName.CPM_DEFAULT)
                .filter(c -> !c.getStatusArchived())
                .map(c -> c.getId())
                .collect(toList());
        ListValidationBuilder<Long, Defect> listValidationBuilder =
                ListValidationBuilder.of(campaignIds, Defect.class);
        List<CampaignPredictionRequest> campaignPredictionRequestForCampaigns =
                inventoriServiceCore.getCampaignPredictionRequestForCampaigns(
                        true,
                        null,
                        null,
                        campaignIds,
                        true,
                        true,
                        true,
                        true,
                        listValidationBuilder.getResult());

        // для ручных стратегий не собираем охват и сразу проставляем выше порога, см. DIRECT-108080
        var targetThreshold = brandLiftConfig.getBranch("thresholds").getLong("target");
        var targetEstimations = campaigns.stream()
                .filter(c -> c.getStrategy().getStrategyName() == StrategyName.CPM_DEFAULT)
                .map(c -> new TargetEstimation(c.getId(), targetThreshold + 1, GREEN_TRAFFIC_LIGHT))
                .collect(toList());

        var operatorLogin = brandLiftConfig.getBranch("inventori").getString("operator_login");
        Map<Long, String> chiefsLoginsByCampaignIds = getChiefsLoginsByCampaignIds(campaigns);

        for (var request : campaignPredictionRequestForCampaigns) {
            try {
                var clientLogin = chiefsLoginsByCampaignIds.get(request.getCampaignId());
                var requestId = UUID.randomUUID().toString();
                CampaignPredictionResponse response = null;
                try {
                    response =
                            inventoriClient.getGeneralCampaignPrediction(requestId, operatorLogin, clientLogin, request);
                } catch(Exception e){}

                TargetEstimation targetEstimation;
                if (response != null && response instanceof GeneralCampaignPredictionResponse) {
                    targetEstimation = new TargetEstimation(
                            request.getCampaignId(),
                            nvl(((GeneralCampaignPredictionResponse) response).getTargetReach(), 0L),
                            nvl(((GeneralCampaignPredictionResponse) response).getTrafficLightColour(), 0));
                } else {
                    logger.warn("Got nothing while requesting inventori " +
                            "for general campaign performance for campaign " +
                            request.getCampaignId());
                    targetEstimation = new TargetEstimation(request.getCampaignId(), 0L, 0);
                }

                targetEstimations.add(targetEstimation);
            } catch (RuntimeException ex) {
                logger.error("Got exception while requesting inventori " +
                        "for general campaign performance for campaign " +
                        request.getCampaignId(), ex);
            }
        }

        return targetEstimations;
    }

    private Map<Long, String> getChiefsLoginsByCampaignIds(List<Campaign> campaigns) {
        Map<Long, Long> clientIdsByCampaignIds = listToMap(campaigns, Campaign::getId, Campaign::getClientId);
        Set<ClientId> clientIds = listToSet(campaigns, campaign -> ClientId.fromLong(campaign.getClientId()));
        Map<ClientId, String> chiefsLoginsByClientIds = userService.getChiefsLoginsByClientIds(clientIds);

        return clientIdsByCampaignIds
                .entrySet()
                .stream()
                .collect(
                        toMap(Map.Entry::getKey,
                                clientIdByCampaign -> chiefsLoginsByClientIds.get(ClientId.fromLong(clientIdByCampaign.getValue()))));
    }
}
