package ru.yandex.direct.core.entity.bs;

import java.math.BigInteger;
import java.util.Collection;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.jooq.types.Unsigned;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.ytcomponents.service.PhraseIdByDataMd5DynContextProvider;
import ru.yandex.direct.ytwrapper.YtTableUtils;
import ru.yandex.yt.ytclient.proxy.LookupRowsRequest;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

import static ru.yandex.direct.grid.schema.yt.Tables.PHRASEIDBYDATAMD5_BS;

/**
 * Получение "исторически выданных" PhraseID по хешам от фраз (yabs_md5)
 */
@Repository
public class PhraseIdByDataMd5Repository {
    public static final TableSchema SCHEMA = new TableSchema.Builder()
            .addKey(PHRASEIDBYDATAMD5_BS.DATA_MD5.getName(), ColumnValueType.UINT64)
            .addValue(PHRASEIDBYDATAMD5_BS.PHRASE_ID.getName(), ColumnValueType.UINT64)
            .sortBy(PHRASEIDBYDATAMD5_BS.DATA_MD5.getName())
            .build();

    private final PhraseIdByDataMd5DynContextProvider dynContextProvider;

    @Autowired
    public PhraseIdByDataMd5Repository(PhraseIdByDataMd5DynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    /**
     * Получить маппинг md5 во PhraseID (если он существует)
     *
     * @param md5Hashes — хеши фраз, для которых нужно получить PhraseID
     * @return словарь соответствия md5->PhraseID, хеши без маппинга в нём отсутствуют!
     */
    public Map<BigInteger, BigInteger> getPhraseIdByMd5(Collection<BigInteger> md5Hashes) {
        String path = dynContextProvider.getContext().getPathToTable(PHRASEIDBYDATAMD5_BS);
        LookupRowsRequest request = new LookupRowsRequest(path, SCHEMA.toLookup());
        md5Hashes.forEach(h -> request.addFilter(h.longValue()));

        Function<UnversionedValue, BigInteger> convert = value -> Unsigned.ulong(value.longValue()).toBigInteger();

        try (var ignored = Trace.current().profile("getPhraseIdByMd5FromYt", "", md5Hashes.size())) {
            var result = dynContextProvider.getContext().executeLookup(request);
            TableSchema resultSchema = result.getSchema();
            int md5ColNum = YtTableUtils.findColumnOrThrow(resultSchema, PHRASEIDBYDATAMD5_BS.DATA_MD5.getName());
            int phraseIdColNum = YtTableUtils.findColumnOrThrow(resultSchema, PHRASEIDBYDATAMD5_BS.PHRASE_ID.getName());


            return StreamEx.of(result.getRows())
                    .map(UnversionedRow::getValues)
                    .mapToEntry(v -> v.get(md5ColNum), v -> v.get(phraseIdColNum))
                    .mapKeys(convert)
                    .mapValues(convert)
                    .toMap();
        }
    }
}
