package ru.yandex.direct.core.entity.bs.common.service;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
public class BsBannerIdCalculator {
    private BsBannerIdCalculator() {
    }

    // Старший байт BannerID, используется при вычислении BannerID
    // Первоисточник констант - маппинг "источник данных" - "номер источника":
    // https://a.yandex-team.ru/arc/trunk/arcadia/direct/libs/bstransport/proto/integration.proto
    private final static byte BANNER_ID_SIGNIFICANT_BYTE = 1;

    /**
     * Вычисляет идентификатор баннера в БК на основе переданного ID баннера в Директе
     * <p>
     * Идентификатор баннера в БК (ID) вычисляется из идентификатора баннера в Директе (EID) по схеме
     * SIGNIFICANT_BYTE в старшем байте + EID.
     * Для Директа SIGNIFICANT_BYTE = 1:
     * 0x00FFFFFFFFFFFFFF & directBannerId + 0x0100000000000000
     * <p>
     * Соотвествующий код в перле (_calculate_banner_id):
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/BS/ExportQuery.pm#L7070
     */
    @Nonnull
    public static Long calculateBsBannerId(Long directBannerId) {
        if ((directBannerId & ((1L << 56) - 1L)) != directBannerId) {
            throw new IllegalArgumentException("BannerId must be positive and no more than 72057594037927935");
        }

        return (directBannerId & ((1L << 56) - 1L)) + ((long) BANNER_ID_SIGNIFICANT_BYTE << 56);
    }
}
