package ru.yandex.direct.core.entity.bs.common.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bs.common.repository.BsCommonRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.toMap;

@Service
@ParametersAreNonnullByDefault
public class BsOrderIdCalculator {

    private final BsCommonRepository bsCommonRepository;
    private final DslContextProvider dslContextProvider;

    // константа аналогичная перлу
    // todo добавить ссылку после https://st.yandex-team.ru/DIRECT-125679
    public static final int ORDER_ID_OFFSET = 100_000_000;

    public BsOrderIdCalculator(BsCommonRepository bsCommonRepository,
                               DslContextProvider dslContextProvider) {
        this.bsCommonRepository = bsCommonRepository;
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получаем для каждого переданного campaignId кампании соответсвующий OrderID для БК, сгенерированный в Директе.
     * <p>
     * Если OrderID или уже записан в ppc.campaigns, то он и вернется
     * Если нигде не записан, тогда он генерируется прибавлением {@link BsOrderIdCalculator#ORDER_ID_OFFSET}
     * Если кампании нет в таблице campaigns, то ее не будет в результате
     *
     * @param campaignIds набор ID кампаний
     * @return отображение campaignId в OrderID
     */
    @Nonnull
    public Map<Long, Long> calculateOrderIdIfNotExist(int shard, Collection<Long> campaignIds) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        Map<Long, Long> orderIdForCampaigns =
                bsCommonRepository.getOrderIdForCampaigns(dslContext, campaignIds);

        Map<Long, Long> allocatedOrderIdForCampaigns = EntryStream.of(orderIdForCampaigns)
                .filterValues(v -> v != 0L)
                .toMap();

        List<Long> withoutOrderIds = EntryStream.of(orderIdForCampaigns)
                .filterValues(v -> v == 0L)
                .keys()
                .toList();

        if (withoutOrderIds.isEmpty()) {
            return allocatedOrderIdForCampaigns;
        }

        Map<Long, Long> calculatedOrderIds = calculateOrderId(withoutOrderIds);

        return EntryStream.of(allocatedOrderIdForCampaigns).append(calculatedOrderIds).toMap();
    }

    /**
     * Получаем для каждого переданного campaignId кампании соответсвующий OrderID для БК, сгенерированный в Директе.
     * <p>
     * OrderID генерируется прибавлением {@link BsOrderIdCalculator#ORDER_ID_OFFSET}
     *
     * @param campaignIds набор ID кампаний
     * @return отображение campaignId в OrderID
     */
    @Nonnull
    private Map<Long, Long> calculateOrderId(Collection<Long> campaignIds) {
        return campaignIds.stream()
                .collect(toMap(cid -> cid, cid -> calculateOrderId(cid)));
    }

    public long calculateOrderId(long campaignId) {
        return campaignId + ORDER_ID_OFFSET;
    }
}
