package ru.yandex.direct.core.entity.bs.export;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyName;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.bs.export.model.WorkerSpec;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static ru.yandex.direct.common.db.PpcPropertyNames.BSEXPORT_WORKERS_NUM_CONTROLLED_MANUALLY;

/**
 * Сервис по получению/заданию различных настроек экспорта в БК и ре-экспорта в контент-систему БК
 */
@Service
@ParametersAreNonnullByDefault
public class BsExportParametersService {
    /**
     * Количество воркеров, возвращаемое при отсутствии в базе значения проперти.
     */
    public static final int DEFAULT_WORKERS_NUM = 1;

    private static final Map<WorkerType, Integer> WORKERS_NUM_BY_TYPE = StreamEx.of(WorkerSpec.values())
            .mapToEntry(WorkerSpec::getWorkerType)
            .invert()
            .sortedBy(Map.Entry::getKey)
            .collapseKeys()
            .mapValues(Collection::size)
            .toImmutableMap();

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final ShardHelper shardHelper;

    private final PpcProperty<Boolean> fullExportRollingWorkAllowed;
    private final PpcProperty<Boolean> isManualModeProperty;
    private final PpcProperty<Integer> fullExportChunkPerWorker;
    private final PpcProperty<Integer> fullExportMaxCampaignsInQueue;
    private final PpcProperty<Integer> fullExportMaxChunkPerIteration;

    @Autowired
    public BsExportParametersService(PpcPropertiesSupport ppcPropertiesSupport, ShardHelper shardHelper) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.shardHelper = shardHelper;

        this.isManualModeProperty = ppcPropertiesSupport.get(BSEXPORT_WORKERS_NUM_CONTROLLED_MANUALLY);
        this.fullExportChunkPerWorker = ppcPropertiesSupport.get(PpcPropertyNames.FULL_LB_EXPORT_MAX_CHUNK_PER_WORKER);
        this.fullExportMaxCampaignsInQueue =
                ppcPropertiesSupport.get(PpcPropertyNames.FULL_LB_EXPORT_MAX_CAMPAIGNS_IN_QUEUE);
        this.fullExportMaxChunkPerIteration =
                ppcPropertiesSupport.get(PpcPropertyNames.FULL_LB_EXPORT_MAX_CHUNK_PER_ITERATION);
        this.fullExportRollingWorkAllowed =
                ppcPropertiesSupport.get(PpcPropertyNames.FULL_LB_EXPORT_ALLOW_ROLLING_WORK);
    }

    /**
     * Получить список пар номер шарда и количество воркеров указанного типа
     *
     * @param workerType тип воркера
     * @param shards     список шардов
     * @return список из пар номер шарда/количество воркеров
     */
    private Map<Integer, Integer> getWorkersNumByShard(WorkerType workerType, Collection<Integer> shards) {
        Function<Integer, PpcPropertyName<Integer>> propertyFactory = getWorkersNumPropertyNameFactory(workerType);
        Map<String, Integer> property2shard = StreamEx.of(shards)
                .mapToEntry(propertyFactory)
                .mapValues(PpcPropertyName::getName)
                .invert()
                .toMap();

        return EntryStream.of(ppcPropertiesSupport.getByNames(property2shard.keySet()))
                .mapKeys(property2shard::get)
                .mapValues(v -> v == null ? DEFAULT_WORKERS_NUM : Integer.parseInt(v))
                .toMap();
    }

    /**
     * Получить количество воркеров указанного типа для всех шардов
     *
     * @param workerType тип воркера
     * @return текущие значения из property или {@link #DEFAULT_WORKERS_NUM} если оно не задано, для всех шардов
     * @throws IllegalArgumentException в случае, если для типа воркера не опеределено свойство, хранящее число воркеров
     * @throws NumberFormatException    если не удается распарсить значение property как число
     */
    public Map<Integer, Integer> getWorkersNumForAllShards(WorkerType workerType) {
        return getWorkersNumByShard(workerType, shardHelper.dbShards());
    }

    /**
     * Получить настройку "количество воркеров указанного типа в шарде"
     *
     * @param workerType тип воркера
     * @param shard      номер шарда
     * @return текущее значение из property или {@link #DEFAULT_WORKERS_NUM} если оно не задано
     * @throws IllegalArgumentException в случае, если для типа воркера не опеределено свойство, хранящее число воркеров
     * @throws NumberFormatException    если не удается распарсить значение property как число
     */
    public int getWorkersNum(WorkerType workerType, int shard) {
        return getWorkersNumByShard(workerType, Collections.singletonList(shard)).get(shard);
    }

    private static Function<Integer, PpcPropertyName<Integer>> getWorkersNumPropertyNameFactory(WorkerType workerType) {
        switch (workerType) {
            case FULL_LB_EXPORT:
                return PpcPropertyNames::bsExportFullLbExportWorkersNum;
            case STD:
                return PpcPropertyNames::bsExportStdWorkersNum;
            case HEAVY:
                return PpcPropertyNames::bsExportHeavyWorkersNum;
            case BUGGY:
                return PpcPropertyNames::bsExportBuggyWorkersNum;
            default:
                throw new IllegalArgumentException(workerType + " doesn't have property for WorkersNum");
        }
    }

    /**
     * Установить настройку "количество воркеров указанного типа в шарде"
     *
     * @param workerType тип воркера
     * @param shard      номер шарда
     * @param value      новое значение количества воркеров данного типа
     */
    public void setWorkersNumInShard(WorkerType workerType, int shard, int value) {
        PpcPropertyName<Integer> ppcPropertyName = getWorkersNumPropertyNameFactory(workerType).apply(shard);
        ppcPropertiesSupport.get(ppcPropertyName).set(value);
    }

    /**
     * Установить настройку "количество воркеров указанного типа для всех шардов"
     *
     * @param workerType тип воркера
     * @param value      новое значение количества воркеров данного типа
     */
    public void setWorkersNumForAllShards(WorkerType workerType, int value) {
        for (Integer shard : shardHelper.dbShards()) {
            setWorkersNumInShard(workerType, shard, value);
        }
    }

    /**
     * Включить режим ручного управления количеством воркеров экпорта в БК
     */
    public void enableManualControllingMode() {
        isManualModeProperty.set(Boolean.TRUE);
    }

    /**
     * Отключить режим ручного управления количеством воркеров экспорта в БК
     */
    public void disableManualControllingMode() {
        isManualModeProperty.set(Boolean.FALSE);
    }

    /**
     * Проверка возможности ручного управления количеством воркеров экспорта в БК
     *
     * @return возвращает истину, если включен режим ручного управления
     */
    public boolean isManualMode() {
        return isManualModeProperty.getOrDefault(Boolean.FALSE);
    }

    /**
     * Проверить, можно ли продолжить работу ре-экспорта с начала после завершения цикла по шарду
     */
    public boolean canFullExportRollingWork() {
        return fullExportRollingWorkAllowed.getOrDefault(false);
    }

    /**
     * Установить значение свойства, отвечающего за возможность повторения "с начала" по завершению переотправки
     */
    public void setFullExportRollingWork(boolean enabled) {
        fullExportRollingWorkAllowed.set(enabled);
    }

    /**
     * Получить расчетное количество кампаний на один воркер ре-экспорта
     *
     * @return сохраненное значение или {@code 0}
     */
    public int getFullExportChunkPerWorker() {
        return fullExportChunkPerWorker.getOrDefault(0);
    }

    /**
     * Задать расчетное количество кампаний на один воркер ре-экспорта
     */
    public void setFullExportChunkPerWorker(int chunkSize) {
        fullExportChunkPerWorker.set(chunkSize);
    }

    /**
     * Получить ограничение на размер очереди ре-экспорта (в каждом из шардов)
     *
     * @return сохраненное значение или {@code 0}
     */
    public int getFullExportMaximumCampaignsInQueue() {
        return fullExportMaxCampaignsInQueue.getOrDefault(0);
    }

    /**
     * Задать ограничение на размер очереди ре-экспорта (в каждом из шардов)
     */
    public void setFullExportMaximumCampaignsInQueue(int limit) {
        fullExportMaxCampaignsInQueue.set(limit);
    }

    /**
     * Получить ограничение на количество кампаний, добавляемых мастер-процессом в очередь ре-экспорта за один раз
     *
     * @return сохраненное значение или {@code 0}
     */
    public int getFullExportMaximumChunkPerIteration() {
        return fullExportMaxChunkPerIteration.getOrDefault(0);
    }

    /**
     * Задать ограничение на количество кампаний, добавляемых мастер-процессом в очередь ре-экспорта за один раз
     */
    public void setFullExportMaximumChunkPerIteration(int chunkSize) {
        fullExportMaxChunkPerIteration.set(chunkSize);
    }

    /**
     * Получить свойство для хранения номера последней обработанной кампании
     *
     * @param shard шард для которого требуется свойство
     */
    public PpcProperty<Long> getFullExportLastProcessedCampaignIdProperty(int shard) {
        return ppcPropertiesSupport.get(PpcPropertyNames.fullLbExportLastProcessedCampaignId(shard));
    }

    /**
     * Получить известное (по {@link WorkerSpec}) количество воркеров указанного типа.
     * Для типов {@link WorkerType#MASTER} и {@link WorkerType#NOSEND} - значение не имеет смысла
     *
     * @param workerType тип воркера
     * @return количество воркеров такого типа или 0
     */
    public static int getKnownWorkersNum(WorkerType workerType) {
        return WORKERS_NUM_BY_TYPE.getOrDefault(workerType, 0);
    }
}
