package ru.yandex.direct.core.entity.bs.export.model;

import java.util.function.BinaryOperator;
import java.util.stream.Stream;

import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportQueueStat;

/**
 * Для отправки "чего" предназначен воркер.
 * Получается комбинацией вариантов: ре-экспорт, только кампания, контексты+баннеры+ключевики, ставки.
 */
public enum WorkerPurpose {
    STUB(false, false, false, false),
    ONLY_CAMPAIGNS(true, false, false, false),
    CONTEXTS_AND_BANNERS(false, true, false, false),
    DATA(true, true, false, false),
    DATA_AND_PRICES(true, true, true, false),
    ONLY_PRICES(false, false, true, false),
    FULL_EXPORT(false, false, false, true);

    private final boolean designedToFullExport;
    private final boolean designedToSendPrices;
    private final boolean designedToSendCampaigns;
    private final boolean designedToSendContextsAndBanners;

    WorkerPurpose(boolean designedToSendCampaigns, boolean designedToSendContextsAndBanners,
            boolean designedToSendPrices, boolean designedToFullExport)
    {
        if ((designedToSendPrices || designedToSendCampaigns || designedToSendContextsAndBanners)
                && designedToFullExport)
        {
            throw new IllegalArgumentException("full export isn't compatible with sending data, campaigns, prices");
        }

        this.designedToFullExport = designedToFullExport;
        this.designedToSendPrices = designedToSendPrices;
        this.designedToSendCampaigns = designedToSendCampaigns;
        this.designedToSendContextsAndBanners = designedToSendContextsAndBanners;
    }

    public boolean isDesignedToFullExport() {
        return designedToFullExport;
    }

    public boolean isDesignedToSendPrices() {
        return designedToSendPrices;
    }

    public boolean isDesignedToSendCampaigns() {
        return designedToSendCampaigns;
    }

    public boolean isDesignedToSendContextsAndBanners() {
        return designedToSendContextsAndBanners;
    }

    private boolean isStatSatisfy(BsExportQueueStat queueStat, BinaryOperator<Boolean> mergeOperator) {
        Stream.Builder<Boolean> stream = Stream.builder();

        if (designedToFullExport) {
            stream.add(queueStat.getNeedFullExport());
        }

        if (designedToSendCampaigns) {
            stream.add(queueStat.getCampaignsCount() > 0);
        }

        if (designedToSendContextsAndBanners) {
            stream.add(queueStat.getContextsCount() > 0);
            stream.add(queueStat.getBannersCount() > 0);
            stream.add(queueStat.getKeywordsCount() > 0);
        }

        if (designedToSendPrices) {
            stream.add(queueStat.getPricesCount() > 0);
        }

        return stream.build().reduce(mergeOperator).orElseThrow();
    }

    /**
     * Проверяет, что указанным типом может быть что-то отправлено
     */
    public boolean isApplicable(BsExportQueueStat queueStat) {
        return isStatSatisfy(queueStat, Boolean::logicalOr);
    }
}
