package ru.yandex.direct.core.entity.bs.export.model;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;

/**
 * Назначение воркера экспорта
 */
public enum WorkerType {
    /**
     * Обычный воркер по отправке данных (но умеет заодно отправить и ставки по взятым кампаниям)&
     * Обрабатывает только кампании без явно назначенного {@link QueueType}
     */
    STD(WorkerPurpose.DATA),

    /**
     * Отправка ставок.
     * Обрабатывает только кампании без явно назначенного {@link QueueType}
     */
    STDPRICE(WorkerPurpose.ONLY_PRICES),

    /**
     * Отправка данных для внутренних объявлений (другой EngineID)
     */
    INTERNAL_ADS(WorkerPurpose.DATA),

    /**
     * Отправка данных и ставок кампаний, которым назначен тип {@link QueueType#HEAVY}
     */
    HEAVY(WorkerPurpose.DATA_AND_PRICES, QueueType.HEAVY),

    /**
     * Отправка кампаний и новых кошельков
     */
    CAMP(WorkerPurpose.ONLY_CAMPAIGNS),

    /**
     * Отправка только кампаний (без контекстов), используется при мигригровании на
     * <a href="https://st.yandex-team.ru/DIRECT-88156">новую схему зачислений</a>
     * Работает только с кампаниями, которым назначен тип {@link QueueType#CAMPS_ONLY}
     */
    CAMPS_ONLY(WorkerPurpose.ONLY_CAMPAIGNS, QueueType.CAMPS_ONLY),

    /**
     * Отправка данных и ставок, но только для кампаний с указанноым типом {@link QueueType#FAST}
     */
    FAST(WorkerPurpose.DATA_AND_PRICES, QueueType.FAST),

    /**
     * Отправляет с limtest1 данные кампаний, которым назначен тип {@link QueueType#DEV1}
     */
    DEV1(WorkerPurpose.DATA, QueueType.DEV1),

    /**
     * Отправляет с limtest2 данные кампаний, которым назначен тип {@link QueueType#DEV2}
     */
    DEV2(WorkerPurpose.DATA, QueueType.DEV2),

    /**
     * Отправляет с limtest1 ставки кампаний, которым назначен тип {@link QueueType#DEV1}
     */
    DEVPRICE1(WorkerPurpose.ONLY_PRICES, QueueType.DEV1),

    /**
     * ООтправляет с limtest2 ставки кампаний, которым назначен тип {@link QueueType#DEV2}
     */
    DEVPRICE2(WorkerPurpose.ONLY_PRICES, QueueType.DEV2),

    /**
     * Отправка данных и ставок в препрод БК (bssoap-pre) для кампаний с назначенным типом {@link QueueType#PREPROD}
     */
    PREPROD(WorkerPurpose.DATA_AND_PRICES, QueueType.PREPROD),

    /**
     * Очередь обработки поломанных и подозрительных кампаний, которым назначен тип {@link QueueType#BUGGY}.
     * Отправляет данные и ставки.
     */
    BUGGY(WorkerPurpose.DATA_AND_PRICES, QueueType.BUGGY),

    /**
     * Фоновая переотправка данных в контент-систему
     */
    FULL_LB_EXPORT(WorkerPurpose.FULL_EXPORT),

    /**
     * Мастер-процесс (не воркер)
     */
    MASTER(WorkerPurpose.STUB),

    /**
     * Соответствутет {@link QueueType#NOSEND}. Нет воркеров, обслуживающих данный тип
     */
    NOSEND(WorkerPurpose.STUB, QueueType.NOSEND);

    private final QueueType mappedQueueType;
    private final WorkerPurpose workerPurpose;

    WorkerType(WorkerPurpose workerPurpose, QueueType mappedQueueType) {
        this.workerPurpose = workerPurpose;
        this.mappedQueueType = mappedQueueType;
    }

    WorkerType(WorkerPurpose workerPurpose) {
        this(workerPurpose, null);
    }

    @Nullable
    public QueueType getMappedQueueType() {
        return mappedQueueType;
    }

    public WorkerPurpose getWorkerPurpose() {
        return workerPurpose;
    }

    /**
     * Является ли тип очереди "временным" для отправки кампании.
     * После отправки всех соответствующих данных - типизация кампании будет снята
     */
    public boolean isTemporaryQueueType() {
        if (mappedQueueType == null) {
            return false;
        }

        switch (mappedQueueType) {
            case HEAVY:
            case FAST:
            case BUGGY:
            case CAMPS_ONLY:
                return true;

            default:
                return false;
        }
    }
}
