package ru.yandex.direct.core.entity.bs.export.queue.repository;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.DatePart;
import org.jooq.Field;
import org.jooq.InsertOnDuplicateSetMoreStep;
import org.jooq.InsertValuesStep8;
import org.jooq.Record1;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.bs.export.model.CampaignIdWithSyncValue;
import ru.yandex.direct.core.entity.bs.export.model.WorkerPurpose;
import ru.yandex.direct.core.entity.bs.export.model.WorkerSpec;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportQueueInfo;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportQueueStat;
import ru.yandex.direct.dbschema.ppc.enums.BsExportSpecialsParType;
import ru.yandex.direct.dbschema.ppc.tables.records.BsExportQueueRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.iif;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.bs.export.model.WorkerSpec.MASTER;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_EXPORT_CANDIDATES;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_EXPORT_QUEUE;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_EXPORT_SPECIALS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbutil.SqlUtils.STRAIGHT_JOIN;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Репозиторий для работы с очередью экспорта данных директа в БКBsExportIterationContext
 */
@Repository
@ParametersAreNonnullByDefault
public class BsExportQueueRepository {
    private static final Logger logger = LoggerFactory.getLogger(BsExportQueueRepository.class);

    public static final Long EMPTY_STAT = 0L;
    private static final Long STATE_NEED_FULL_EXPORT = RepositoryUtils.TRUE;
    private static final Long STATE_COMMON_EXPORT = RepositoryUtils.FALSE;
    private static final Long MAX_SYNC_VAL_BEFORE_MOVING_TO_END_OF_QUEUE = 5L;

    private static final int CAMPAIGN_IDS_CHUNK_SIZE = 2500;

    /**
     * Типовые par_type которые потоки не должны брать. используется для {@literal camp} и {@literal full_lb_export}
     * Копия словаря {@code @BS::Export::COMMON_EXCLUDE_PAR_TYPES} из perl
     * <p>
     * На jooq-типах, так как снаружи пока не понадобился. Правильно использовать отвязанный тип, и здесь генерировать
     * словарь из него.
     */
    private static final Set<BsExportSpecialsParType> COMMON_EXCLUDE_PAR_TYPES = ImmutableSet.of(
            BsExportSpecialsParType.dev1,
            BsExportSpecialsParType.dev2,
            BsExportSpecialsParType.nosend,
            BsExportSpecialsParType.buggy
    );

    private static final Field<LocalDateTime> QUEUE_TIME_BY_SYNC_VAL = JooqMapperUtils.mysqlIf(
            BS_EXPORT_QUEUE.SYNC_VAL.gt(MAX_SYNC_VAL_BEFORE_MOVING_TO_END_OF_QUEUE),
            DSL.currentLocalDateTime(),
            BS_EXPORT_QUEUE.QUEUE_TIME);

    private static final Field<LocalDateTime> FULL_EXPORT_SEQ_TIME_BY_IS_FULL_EXPORT = JooqMapperUtils.mysqlIf(
            BS_EXPORT_QUEUE.IS_FULL_EXPORT.eq(STATE_COMMON_EXPORT),
            DSL.currentLocalDateTime(),
            BS_EXPORT_QUEUE.FULL_EXPORT_SEQ_TIME);

    public static final Condition EXCLUDE_SPECIAL_PAR_TYPES = BS_EXPORT_SPECIALS.PAR_TYPE.isNull()
            .or(BS_EXPORT_SPECIALS.PAR_TYPE.notIn(COMMON_EXCLUDE_PAR_TYPES));

    private static final Duration DEFAULT_DELAY_DURATION = Duration.ofMinutes(2);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<BsExportQueueInfo> queueInfoJooqMapper;
    private final Duration delaySeqTime;

    @Autowired
    public BsExportQueueRepository(DslContextProvider dslContextProvider, DirectConfig directConfig) {
        this(dslContextProvider,
                directConfig.findDuration("bs_export_worker.delay_seq_time").orElse(DEFAULT_DELAY_DURATION));
    }

    public BsExportQueueRepository(DslContextProvider dslContextProvider, Duration delaySeqTime) {
        this.dslContextProvider = dslContextProvider;
        this.queueInfoJooqMapper = createQueueInfoMapper();
        this.delaySeqTime = delaySeqTime;
    }

    /**
     * Получить множество идентификаторов кампаний, которые находятся в очереди на отправку в БК или скоро в ней
     * окажутся.
     * <p>
     * Аналог выборки {@code $in_bs_export} из перлового {@code Direct::Campaigns::get_by}
     *
     * @param shard       шард
     * @param campaignIds список идентификаторов кампаний, которые нужно проверить на наличие в очереди
     */
    public Set<Long> getCampaignsIdsInQueue(int shard, Collection<Long> campaignIds) {
        SelectConditionStep<Record1<Long>> exportQueue = dslContextProvider.ppc(shard)
                .select(BS_EXPORT_QUEUE.CID)
                .from(BS_EXPORT_QUEUE)
                .where(BS_EXPORT_QUEUE.CID.in(campaignIds))
                .and(BS_EXPORT_QUEUE.PAR_ID.isNull()
                        .or(BS_EXPORT_QUEUE.PAR_ID.ne(WorkerSpec.NOSEND_FOR_DROP_SANDBOX_CLIENT.getWorkerId())));

        SelectConditionStep<Record1<Long>> exportCandidates = dslContextProvider.ppc(shard)
                .select(BS_EXPORT_CANDIDATES.CID)
                .from(BS_EXPORT_CANDIDATES)
                .where(BS_EXPORT_CANDIDATES.CID.in(campaignIds));

        return exportQueue
                .union(exportCandidates)
                .fetchSet(Record1::value1);
    }

    /**
     * Получить множество идентификаторов кампаний, которые находятся в очереди на отправку в БК,
     * кроме тех,  которые добавлены мастер экспортом (par_id == 99)
     */
    public Set<Long> getCampaignIdsInQueueExceptMasterExport(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(BS_EXPORT_QUEUE.CID)
                .from(BS_EXPORT_QUEUE)
                .where(BS_EXPORT_QUEUE.CID.in(campaignIds)
                        .and(BS_EXPORT_QUEUE.PAR_ID.ne(MASTER.getWorkerId())))
                .fetchSet(BS_EXPORT_QUEUE.CID, Long.class);
    }

    /**
     * Снять флаг необходимости ре-экспорта у кампаний в очереди.
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний для снятия флага
     * @return количество обновленных записей
     * @implNote запросы к базе выполняются чанками по {@link #CAMPAIGN_IDS_CHUNK_SIZE}
     */
    public int removeCampaignsFullExportFlag(int shard, Collection<Long> campaignIds) {
        int count = 0;
        for (List<Long> chunk : Iterables.partition(campaignIds, CAMPAIGN_IDS_CHUNK_SIZE)) {
            count += dslContextProvider.ppc(shard)
                    .update(BS_EXPORT_QUEUE)
                    .set(BS_EXPORT_QUEUE.IS_FULL_EXPORT, STATE_COMMON_EXPORT)
                    .where(BS_EXPORT_QUEUE.CID.in(chunk)
                            .and(BS_EXPORT_QUEUE.IS_FULL_EXPORT.eq(STATE_NEED_FULL_EXPORT)))
                    .execute();
        }
        return count;
    }

    /**
     * Выставить флаг необходимости ре-экспорта кампаниям в очереди.
     * Отсутствующие в очереди кампании будут добавлены.
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний для взведения флага
     * @return количество измененных записей
     * @implNote запросы к базе выполняются чанками по {@link #CAMPAIGN_IDS_CHUNK_SIZE}
     * @implNote в предыдущей реализации метода (perl) sequence time ре-экспорта не обновлялся
     */
    public int addCampaignsFullExportFlag(int shard, Collection<Long> campaignIds) {
        int count = 0;
        LocalDateTime queueTime = LocalDateTime.now();

        for (List<Long> chunk : Iterables.partition(campaignIds, CAMPAIGN_IDS_CHUNK_SIZE)) {
            InsertValuesStep8<BsExportQueueRecord, Long, Long, LocalDateTime, Long, Long, Long, Long, Long> insertStep =
                    dslContextProvider.ppc(shard).insertInto(BS_EXPORT_QUEUE,
                            BS_EXPORT_QUEUE.CID,
                            BS_EXPORT_QUEUE.IS_FULL_EXPORT,
                            BS_EXPORT_QUEUE.QUEUE_TIME,
                            BS_EXPORT_QUEUE.CAMPS_NUM,
                            BS_EXPORT_QUEUE.BANNERS_NUM,
                            BS_EXPORT_QUEUE.CONTEXTS_NUM,
                            BS_EXPORT_QUEUE.BIDS_NUM,
                            BS_EXPORT_QUEUE.PRICES_NUM
                    );

            for (Long campaignId : chunk) {
                insertStep.values(campaignId, STATE_NEED_FULL_EXPORT, queueTime,
                        EMPTY_STAT, EMPTY_STAT, EMPTY_STAT, EMPTY_STAT, EMPTY_STAT);
            }

            InsertOnDuplicateSetMoreStep<BsExportQueueRecord> onDuplicateInsert = insertStep.onDuplicateKeyUpdate()
                    .set(BS_EXPORT_QUEUE.SYNC_VAL, BS_EXPORT_QUEUE.SYNC_VAL.add(1))
                    .set(BS_EXPORT_QUEUE.FULL_EXPORT_SEQ_TIME, FULL_EXPORT_SEQ_TIME_BY_IS_FULL_EXPORT)
                    .set(BS_EXPORT_QUEUE.QUEUE_TIME, QUEUE_TIME_BY_SYNC_VAL)
                    .set(BS_EXPORT_QUEUE.IS_FULL_EXPORT, STATE_NEED_FULL_EXPORT);

            count += onDuplicateInsert.execute();
        }

        return count;
    }

    /**
     * Получить текущее количество кампаний в очереди ре-экспорта.
     * Не учитывает кампании, со специализацией очереди {@link #COMMON_EXCLUDE_PAR_TYPES}
     *
     * @param shard шард
     * @return количество кампаний
     */
    public int getCampaignsCountInFullExportQueue(int shard) {
        return getCampaignsCountInFullExportQueue(dslContextProvider.ppc(shard));
    }

    /**
     * @see #getCampaignsCountInFullExportQueue(int)
     */
    public int getCampaignsCountInFullExportQueue(DSLContext dslContext) {
        return dslContext.selectCount()
                .from(BS_EXPORT_QUEUE)
                .leftJoin(BS_EXPORT_SPECIALS).on(BS_EXPORT_QUEUE.CID.eq(BS_EXPORT_SPECIALS.CID))
                .where(BS_EXPORT_QUEUE.IS_FULL_EXPORT.eq(STATE_NEED_FULL_EXPORT)
                        .and(EXCLUDE_SPECIAL_PAR_TYPES))
                .fetchOne()
                .value1();
    }

    /**
     * Получить сведения о кампании в очереди экспорта в БК.
     * <p>
     * Не массовый, так как используется только в тестах.
     * Когда понадобится массовое получение данных - нужно будет переделать метод
     *
     * @param shard      шард
     * @param campaignId идентификатор кампании
     * @return модель записи очереди или {@code null}
     */
    public BsExportQueueInfo getBsExportQueueInfo(int shard, Long campaignId) {
        return getBsExportQueueInfo(shard, Collections.singleton(campaignId))
                .get(campaignId);
    }

    /**
     * Получить сведения о кампаниях в очереди экспорта в БК.
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний
     * @return словарь, где ключами явлюяются номера кампаний, а значеними - информация о них в очереди
     */
    public Map<Long, BsExportQueueInfo> getBsExportQueueInfo(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(queueInfoJooqMapper.getFieldsToRead())
                .from(BS_EXPORT_QUEUE)
                .where(BS_EXPORT_QUEUE.CID.in(campaignIds))
                .fetchMap(BS_EXPORT_QUEUE.CID, queueInfoJooqMapper::fromDb);
    }

    /**
     * Добавить запись в очередь целиком.
     * Не обрабатывает дубликаты.
     * <p>
     * Пока используется только в тестах
     *
     * @param dslContext контекст БД
     * @param model      данные записи в очереди
     */
    public void insertRecord(DSLContext dslContext, BsExportQueueInfo model) {
        new InsertHelper<>(dslContext, BS_EXPORT_QUEUE).add(queueInfoJooqMapper, model).execute();
    }

    /**
     * Добавить запись в очередь целиком.
     * Не обрабатывает дубликаты.
     * <p>
     * Пока используется только в тестах
     *
     * @param shard шард
     * @param model данные записи в очереди
     */
    public void insertRecord(int shard, BsExportQueueInfo model) {
        insertRecord(dslContextProvider.ppc(shard), model);
    }

    /**
     * Сбросить на текущее время {@code bs_export_queue.queue_time} кампаниям в очереди.
     *
     * @param campaignIds коллекция id кампаний, которым нужно сбросить {@code queue_time}
     */
    public void resetQueueTime(DSLContext dslContext, Collection<Long> campaignIds) {
        for (List<Long> chunk : Iterables.partition(campaignIds, CAMPAIGN_IDS_CHUNK_SIZE)) {
            dslContext
                    .update(BS_EXPORT_QUEUE)
                    .set(BS_EXPORT_QUEUE.QUEUE_TIME, DSL.currentLocalDateTime())
                    .where(BS_EXPORT_QUEUE.CID.in(chunk))
                    .execute();
        }
    }

    public void delayCampaigns(int shard, Collection<Long> campaignIds, boolean isFullExport) {
        delayCampaigns(dslContextProvider.ppc(shard), campaignIds, isFullExport);
    }

    /**
     * Меняет время постановки кампании в очередь либо на {@link #delaySeqTime}, либо на текущее время
     * Если выполняется полный экспорт, то изменяется поле FULL_EXPORT_SEQ_TIME, инчае SEQ_TIME
     *
     * @param campaignIds коллекция id кампаний, которым нужно сбросить {@code queue_time}
     */
    public void delayCampaigns(DSLContext dslContext, Collection<Long> campaignIds, boolean isFullExport) {
        var seqTimeField = isFullExport ? BS_EXPORT_QUEUE.FULL_EXPORT_SEQ_TIME : BS_EXPORT_QUEUE.SEQ_TIME;
        for (List<Long> chunk : Iterables.partition(campaignIds, CAMPAIGN_IDS_CHUNK_SIZE)) {
            dslContext
                    .update(BS_EXPORT_QUEUE)
                    .set(seqTimeField, DSL.least(DSL.currentLocalDateTime(),
                            DSL.localDateTimeAdd(seqTimeField, delaySeqTime.getSeconds(), DatePart.SECOND)))
                    .where(BS_EXPORT_QUEUE.CID.in(chunk))
                    .execute();
        }
    }

    public void resetQueueTime(int shard, Collection<Long> campaignIds) {
        resetQueueTime(dslContextProvider.ppc(shard), campaignIds);
    }

    /**
     * Сбросить на текущее время {@code bs_export_queue.seq_time}, {@code bs_export_queue.queue_time} и
     * {@code bs_export_queue.full_export_seq_time} кампаниям в очереди.
     *
     * @param shard       номер шарда
     * @param campaignIds коллекция id кампаний, которым нужно сбросить поля.
     */
    public void resetAllTimeFields(int shard, Collection<Long> campaignIds) {
        Field<LocalDateTime> currentTime = DSL.currentLocalDateTime();
        dslContextProvider.ppc(shard)
                .update(BS_EXPORT_QUEUE)
                .set(BS_EXPORT_QUEUE.SEQ_TIME, currentTime)
                .set(BS_EXPORT_QUEUE.QUEUE_TIME, currentTime)
                .set(BS_EXPORT_QUEUE.FULL_EXPORT_SEQ_TIME, currentTime)
                .where(BS_EXPORT_QUEUE.CID.in(campaignIds))
                .execute();
    }

    /**
     * Метод возвращает id кампаний, которые были добавлены в очередь раньше {@param expireDateTime}
     * и либо находятся в очереди NOSEND, либо не существуют в Директе.
     * Id кампаний попадают в ключи ответа, в значениях указано, залочена ли кампания транспортом.
     * @param shard — шард
     * @param expireDateTime — верхняя строгая граница времени добавления кампании в очередь
     * @return словарь из id кампаний в признак того, что кампания залочена в очереди
     */
    public Map<Long, Boolean> getExpiredAndBrokenCampaignIds(int shard, LocalDateTime expireDateTime) {
        return dslContextProvider.ppc(shard)
                .select(BS_EXPORT_QUEUE.CID, BS_EXPORT_QUEUE.PAR_ID)
                .hint(STRAIGHT_JOIN)
                .from(BS_EXPORT_QUEUE)
                .leftJoin(BS_EXPORT_SPECIALS)
                        .on(BS_EXPORT_QUEUE.CID.eq(BS_EXPORT_SPECIALS.CID))
                .leftJoin(CAMPAIGNS)
                        .on(BS_EXPORT_QUEUE.CID.eq(CAMPAIGNS.CID))
                .where(BS_EXPORT_QUEUE.QUEUE_TIME.lessThan(expireDateTime))
                        .and(BS_EXPORT_SPECIALS.PAR_TYPE.eq(BsExportSpecialsParType.nosend)
                                .or(CAMPAIGNS.CID.isNull()))
                .fetchMap(BS_EXPORT_QUEUE.CID, rec -> rec.get(BS_EXPORT_QUEUE.PAR_ID) != null);
    }

    public int delete(int shard, Collection<Long> ids) {
        return delete(dslContextProvider.ppc(shard), ids);
    }


    public int deleteIfSyncValueNotChanged(int shard, Collection<CampaignIdWithSyncValue> campaignIdsWithSyncValues) {
        var rows = campaignIdsWithSyncValues.stream()
                .map(campaignIdWithSyncValue -> row(campaignIdWithSyncValue.getCampaignId(),
                        campaignIdWithSyncValue.getSynchronizeValue()))
                .collect(toList());

        return dslContextProvider.ppc(shard)
                .deleteFrom(BS_EXPORT_QUEUE)
                .where(row(BS_EXPORT_QUEUE.CID, BS_EXPORT_QUEUE.SYNC_VAL)
                        .in(rows))
                .execute();
    }

    public int delete(DSLContext dslContext, Collection<Long> ids) {
        return dslContext
                .deleteFrom(BS_EXPORT_QUEUE)
                .where(BS_EXPORT_QUEUE.CID.in(ids))
                .execute();
    }

    /**
     * Залочить кампании в очереди определенным воркером.
     * Обрабатывает только незаблокированные и кампании этого же воркера.
     *
     * @param campaignIds id кампаний для блокировки
     * @param workerId    id worker'а, которым блокируем кампании
     */
    public void lockCampaigns(int shard, Collection<Long> campaignIds, long workerId) {
        logger.debug("lock campaigns {} to workerId {}", campaignIds, workerId);

        if (campaignIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard)
                .update(BS_EXPORT_QUEUE)
                .set(BS_EXPORT_QUEUE.PAR_ID, workerId)
                .where(BS_EXPORT_QUEUE.CID.in(campaignIds)
                        .and(BS_EXPORT_QUEUE.PAR_ID.isNull())
                        .or(BS_EXPORT_QUEUE.PAR_ID.eq(workerId)))
                .execute();
    }

    /**
     * Разблокировать кампании в очереди
     *
     * @param campaignIds id кампаний для разблокировки
     * @param workerId    id worker'а, для которого разрешена разблокировка
     */
    public void unlockCampaigns(int shard, Collection<Long> campaignIds, long workerId) {
        logger.debug("unlock campaigns {} from workerId {}", campaignIds, workerId);

        if (campaignIds.isEmpty()) {
            return;
        }

        // TODO DIRECT-110304: реализовать задержку кампаний в очереди при ошибках
        dslContextProvider.ppc(shard)
                .update(BS_EXPORT_QUEUE)
                .set(BS_EXPORT_QUEUE.PAR_ID, (Long) null)
                .where(BS_EXPORT_QUEUE.CID.in(campaignIds)
                        .and(BS_EXPORT_QUEUE.PAR_ID.eq(workerId)))
                .execute();
    }


    //TODO DIRECT-120202: переделать частичную отметку кампании как отправленной
    public void resetHandledFields(int shard, Collection<CampaignIdWithSyncValue> campaignIdWithSyncValues,
                                   long workerId,
                                   WorkerPurpose workerPurpose) {
        if (campaignIdWithSyncValues.isEmpty()) {
            return;
        }
        var fieldsToReset = getFieldsToReset(workerPurpose);
        if (fieldsToReset.isEmpty()) {
            return;
        }
        var cidToSyncValueMap = campaignIdWithSyncValues.stream()
                .collect(toMap(CampaignIdWithSyncValue::getCampaignId,
                        CampaignIdWithSyncValue::getSynchronizeValue));

        Field<Long> syncValCaseStatement = makeCaseStatement(BS_EXPORT_QUEUE.CID, BS_EXPORT_QUEUE.SYNC_VAL,
                cidToSyncValueMap);

        var updateStep = dslContextProvider.ppc(shard)
                .update(BS_EXPORT_QUEUE)
                .set(fieldsToReset.get(0), calculateValueDependsOnSyncValue(fieldsToReset.get(0),
                        syncValCaseStatement));
        for (var i = 1; i < fieldsToReset.size(); i++) {
            updateStep = updateStep
                    .set(fieldsToReset.get(i), calculateValueDependsOnSyncValue(fieldsToReset.get(i),
                            syncValCaseStatement));
        }

        var cids = campaignIdWithSyncValues.stream().map(CampaignIdWithSyncValue::getCampaignId).collect(toList());
        updateStep
                .where(BS_EXPORT_QUEUE.CID.in(cids)
                        .and(BS_EXPORT_QUEUE.PAR_ID.eq(workerId)))
                .execute();
    }

    private Field<Long> calculateValueDependsOnSyncValue(Field<Long> field, Field<Long> syncValCaseStatement) {
        return iif(BS_EXPORT_QUEUE.SYNC_VAL.eq(syncValCaseStatement), EMPTY_STAT,
                field);
    }

    private List<Field<Long>> getFieldsToReset(WorkerPurpose workerPurpose) {
        var resetFields = new ArrayList<Field<Long>>();
        if (workerPurpose.isDesignedToSendCampaigns()) {
            resetFields.add(BS_EXPORT_QUEUE.CAMPS_NUM);
        }
        if (workerPurpose.isDesignedToSendContextsAndBanners()) {
            resetFields.add(BS_EXPORT_QUEUE.BANNERS_NUM);
            resetFields.add(BS_EXPORT_QUEUE.BIDS_NUM);
            resetFields.add(BS_EXPORT_QUEUE.CONTEXTS_NUM);
        }
        if (workerPurpose.isDesignedToSendPrices()) {
            resetFields.add(BS_EXPORT_QUEUE.PRICES_NUM);
        }
        if (workerPurpose.isDesignedToFullExport()) {
            resetFields.add(BS_EXPORT_QUEUE.IS_FULL_EXPORT);
        }
        return resetFields;
    }

    public static <T extends
            BsExportQueueStat> JooqMapperWithSupplierBuilder<T> createBaseMapperBuilder(
            Supplier<T> modelSupplier) {
        return JooqMapperWithSupplierBuilder.builder(modelSupplier)
                .map(property(BsExportQueueStat.CAMPAIGN_ID, BS_EXPORT_QUEUE.CID))
                .map(property(BsExportQueueStat.CAMPAIGNS_COUNT, BS_EXPORT_QUEUE.CAMPS_NUM))
                .map(property(BsExportQueueStat.BANNERS_COUNT, BS_EXPORT_QUEUE.BANNERS_NUM))
                .map(property(BsExportQueueStat.CONTEXTS_COUNT, BS_EXPORT_QUEUE.CONTEXTS_NUM))
                .map(property(BsExportQueueStat.KEYWORDS_COUNT, BS_EXPORT_QUEUE.BIDS_NUM))
                .map(property(BsExportQueueStat.PRICES_COUNT, BS_EXPORT_QUEUE.PRICES_NUM))
                .map(booleanProperty(BsExportQueueStat.NEED_FULL_EXPORT, BS_EXPORT_QUEUE.IS_FULL_EXPORT));
    }

    public static JooqMapperWithSupplier<BsExportQueueInfo> createQueueInfoMapper() {
        return createBaseMapperBuilder(BsExportQueueInfo::new)
                .map(property(BsExportQueueInfo.SEQUENCE_TIME, BS_EXPORT_QUEUE.SEQ_TIME))
                .map(property(BsExportQueueInfo.QUEUE_TIME, BS_EXPORT_QUEUE.QUEUE_TIME))
                .map(property(BsExportQueueInfo.FULL_EXPORT_SEQUENCE_TIME,
                        BS_EXPORT_QUEUE.FULL_EXPORT_SEQ_TIME))
                .map(integerProperty(BsExportQueueInfo.LOCKED_BY, BS_EXPORT_QUEUE.PAR_ID))
                .map(property(BsExportQueueInfo.SYNCHRONIZE_VALUE, BS_EXPORT_QUEUE.SYNC_VAL))
                .build();
    }
}
