package ru.yandex.direct.core.entity.bs.export.queue.repository;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.AggregateFunction;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.Record1;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportSpecials;
import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbschema.ppc.enums.BsExportSpecialsParType;
import ru.yandex.direct.dbschema.ppc.tables.records.BsExportSpecialsRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.enums.BsExportSpecialsParType.buggy;
import static ru.yandex.direct.dbschema.ppc.tables.BsExportSpecials.BS_EXPORT_SPECIALS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class BsExportSpecialsRepository {
    private static final Logger logger = LoggerFactory.getLogger(BsExportSpecialsRepository.class);

    private static final AggregateFunction<Integer> COUNT_BY_CID_FIELD = DSL.count(BS_EXPORT_SPECIALS.CID);

    private static final List<BsExportSpecialsParType> PERMANENT_PART_TYPES = getPermanentParTypes();
    private final JooqMapperWithSupplier<BsExportSpecials> bsExportSpecialsMapper;
    private final Collection<Field<?>> bsExportSpecialsFieldsToRead;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public BsExportSpecialsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.bsExportSpecialsMapper = JooqMapperWithSupplierBuilder.builder(BsExportSpecials::new)
                .map(property(BsExportSpecials.CAMPAIGN_ID, BS_EXPORT_SPECIALS.CID))
                .map(convertibleProperty(BsExportSpecials.TYPE, BS_EXPORT_SPECIALS.PAR_TYPE,
                        QueueType::fromSource, QueueType::toSource))
                .build();

        this.bsExportSpecialsFieldsToRead =
                bsExportSpecialsMapper.getFieldsToRead(BsExportSpecials.allModelProperties());
    }

    public void add(int shard, List<BsExportSpecials> bsExportSpecialsToAdd) {
        add(dslContextProvider.ppc(shard), bsExportSpecialsToAdd);
    }

    public void add(DSLContext context, List<BsExportSpecials> bsExportSpecialsToAdd) {
        InsertHelper<BsExportSpecialsRecord> helper = new InsertHelper<>(context, BS_EXPORT_SPECIALS)
                .addAll(bsExportSpecialsMapper, bsExportSpecialsToAdd);

        if (helper.hasAddedRecords()) {
            helper.onDuplicateKeyUpdate()
                    .set(BS_EXPORT_SPECIALS.PAR_TYPE, MySQLDSL.values(BS_EXPORT_SPECIALS.PAR_TYPE));
        }

        helper.executeIfRecordsAdded();
    }

    public List<BsExportSpecials> getByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(bsExportSpecialsFieldsToRead)
                .from(BS_EXPORT_SPECIALS)
                .where(BS_EXPORT_SPECIALS.CID.in(campaignIds))
                .fetch(bsExportSpecialsMapper::fromDb);
    }

    /**
     * Если тип очереди кампании в списке nosend, dev1, dev2, preprod, то тип очереди не изменится
     * В остальных случаях переместится в очередь с типом buggy
     */
    public void moveToBuggy(int shard, Collection<Long> campaignIds) {
        InsertValuesStep2<BsExportSpecialsRecord, Long, BsExportSpecialsParType> insertQuery =
                dslContextProvider.ppc(shard)
                        .insertInto(BS_EXPORT_SPECIALS)
                        .columns(BS_EXPORT_SPECIALS.CID, BS_EXPORT_SPECIALS.PAR_TYPE);
        for (var campaignId : campaignIds) {
            insertQuery.values(campaignId, buggy);
        }
        Field<BsExportSpecialsParType> newParType =
                DSL.iif(
                        BS_EXPORT_SPECIALS.PAR_TYPE.in(PERMANENT_PART_TYPES),
                        BS_EXPORT_SPECIALS.PAR_TYPE,
                        MySQLDSL.values(BS_EXPORT_SPECIALS.PAR_TYPE));
        insertQuery
                .onDuplicateKeyUpdate()
                .set(BS_EXPORT_SPECIALS.PAR_TYPE, newParType)
                .execute();

    }

    /**
     * @param type         тип специальной очереди
     * @param campaignType тип кампании
     * @return количество кампаний определенного типа в специальной очереди
     */
    public int campaignsByTypeSizeInQueue(int shard, QueueType type, CampaignType campaignType) {
        return getQueueSize(shard, type, campaignType);
    }

    /**
     * @param type тип специальной очереди
     * @return количество кампаний в специальной очереди
     */
    public int getQueueSize(int shard, QueueType type) {
        return getQueueSize(shard, type, null);
    }

    /**
     * @param type         тип специальной очереди
     * @param campaignType тип кампании для ограничения выборки, может быть null
     * @return количество кампаний в специальной очереди
     */
    @QueryWithoutIndex("Используется в jobs")
    private int getQueueSize(int shard, QueueType type, @Nullable CampaignType campaignType) {
        SelectConditionStep<Record1<Integer>> selectStep = dslContextProvider.ppc(shard)
                .select(COUNT_BY_CID_FIELD)
                .from(BS_EXPORT_SPECIALS)
                .leftJoin(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BS_EXPORT_SPECIALS.CID))
                .where(BS_EXPORT_SPECIALS.PAR_TYPE.eq(QueueType.toSource(type)));
        if (campaignType != null) {
            selectStep = selectStep
                    .and(CAMPAIGNS.TYPE.eq(CampaignType.toSource(campaignType)));
        }

        return selectStep.fetchOne(COUNT_BY_CID_FIELD);
    }

    /**
     * Удалить назначенный кампаниям тип специальной очереди
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний для удаления
     */
    public void remove(int shard, Collection<Long> campaignIds) {
        remove(shard, campaignIds, null);
    }

    /**
     * Удалить назначенный кампаниям тип специальной очереди
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний для удаления
     * @param queueType   тип очереди, из которой надо удалить кампании, если у кампании другой тип очереди - она в
     *                    ней останется
     */
    public void remove(int shard, Collection<Long> campaignIds, @Nullable QueueType queueType) {
        logger.debug("delete campaigns from bs_export_specials: {}", campaignIds);
        BsExportSpecialsParType bsExportSpecialsParType = QueueType.toSource(queueType);
        var deleteCondition = BS_EXPORT_SPECIALS.CID.in(campaignIds);

        if (Objects.nonNull(queueType)) {
            deleteCondition = deleteCondition.and(BS_EXPORT_SPECIALS.PAR_TYPE.eq(bsExportSpecialsParType));
        }
        var deleteStep = dslContextProvider.ppc(shard)
                .deleteFrom(BS_EXPORT_SPECIALS)
                .where(deleteCondition);
        deleteStep.execute();
    }

    private static List<BsExportSpecialsParType> getPermanentParTypes() {
        return Arrays.stream(WorkerType.values())
                .filter(workerType -> !workerType.isTemporaryQueueType())
                .map(WorkerType::getMappedQueueType)
                .filter(Objects::nonNull)
                .map(QueueType::toSource)
                .collect(Collectors.toList());
    }
}
