package ru.yandex.direct.core.entity.bs.export.queue.service;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.core.entity.bs.export.queue.model.BsExportSpecials;
import ru.yandex.direct.core.entity.bs.export.queue.model.QueueType;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportSpecialsRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для работы с очередью экспорта данных директа в БК
 */
@Service
@ParametersAreNonnullByDefault
public class BsExportQueueService {
    /**
     * Типы воркеров, блокировка заказов которыми не приводит к наличию позаказных локов в БК
     */
    public static final Set<WorkerType> WORKER_TYPES_NOT_RESTRICTED_BY_LOCKED_WALLETS = Set.of(
            WorkerType.STDPRICE, WorkerType.DEVPRICE1, WorkerType.DEVPRICE2,
            WorkerType.FULL_LB_EXPORT,
            WorkerType.MASTER);

    private static final int CHUNK_SIZE = 2000;
    private static final int UPDATE_CHUNK_SIZE = 1000;

    private final BsExportQueueRepository bsExportQueueRepository;
    private final BsExportSpecialsRepository bsExportSpecialsRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public BsExportQueueService(BsExportQueueRepository bsExportQueueRepository,
            BsExportSpecialsRepository bsExportSpecialsRepository,
            ShardHelper shardHelper)
    {
        this.bsExportQueueRepository = bsExportQueueRepository;
        this.bsExportSpecialsRepository = bsExportSpecialsRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Получить множество идентификаторов кампаний, которые находятся в очереди на отправку в БК или скоро в ней окажутся.
     *
     * @param campaignIds список идентификаторов кампаний, которые нужно проверить на наличие в очереди
     * @return идентификаторов кампаний находящиеся в очереди
     * @implNote Не предназначен для использования в песочнице, так как часть логики не реализована
     * @see BsExportQueueRepository#getCampaignsIdsInQueue(int, Collection)
     */
    public Set<Long> getCampaignIdsGoingToBeSent(Collection<Long> campaignIds) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID).chunkedBy(CHUNK_SIZE).stream()
                .mapKeyValue(bsExportQueueRepository::getCampaignsIdsInQueue)
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());
    }

    /**
     * Назначить кампаниям тип очереди, воркерами которого они будет отправлена в БК.
     *
     * @param queueType   тип очереди, значение {@code null} соответсвует {@literal std}.
     * @param campaignIds идентификаторы кампаний
     */
    public void setQueueType(@Nullable QueueType queueType, Collection<Long> campaignIds) {
        if (queueType == null) {
            remove(campaignIds);
            return;
        }

        List<BsExportSpecials> specialsToAdd =
                mapList(campaignIds, id -> new BsExportSpecials().withCampaignId(id).withType(queueType));

        shardHelper.groupByShard(specialsToAdd, ShardKey.CID, BsExportSpecials::getCampaignId)
                .chunkedBy(UPDATE_CHUNK_SIZE)
                .stream()
                .forKeyValue(bsExportSpecialsRepository::add);
    }

    private void remove(Collection<Long> campaignIds) {
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedBy(UPDATE_CHUNK_SIZE)
                .stream()
                .forKeyValue(bsExportSpecialsRepository::remove);
    }

    /**
     * Сбросить поля {@code bs_export_queue.seq_time}, {@code bs_export_queue.queue_time} и {@code bs_export_queue.full_export_seq_time} в очереди.
     *
     * @param campaignIds коллекция id кампаний, которым нужно сбросить поля.
     */
    public void resetTimeFields(Collection<Long> campaignIds) {
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedBy(UPDATE_CHUNK_SIZE)
                .stream()
                .forKeyValue(bsExportQueueRepository::resetAllTimeFields);
    }
}
