package ru.yandex.direct.core.entity.bs.export.queue.service;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.bs.export.BsExportParametersService;
import ru.yandex.direct.core.entity.bs.export.model.WorkerType;
import ru.yandex.direct.core.entity.bs.export.queue.repository.BsExportQueueRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;

/**
 * Сервис для работы с очередью ре-экспорта данных директа в БК через LogBroker.
 */
@Service
@ParametersAreNonnullByDefault
public class FullExportQueueService {

    private final BsExportQueueRepository bsExportQueueRepository;
    private final BsExportParametersService parametersService;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;


    @Autowired
    public FullExportQueueService(BsExportQueueRepository bsExportQueueRepository,
                                  BsExportParametersService bsExportParametersService,
                                  CampaignRepository campaignRepository,
                                  ShardHelper shardHelper) {
        this.bsExportQueueRepository = bsExportQueueRepository;
        this.parametersService = bsExportParametersService;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * "Удаляет" кампании из очереди ре-экспорта в БК через LB.
     * <p>
     * Вместо реального удаления - только снимает флаг необходимости ре-экспорта.
     * Если других данных по кампании к отправке нет - из очереди её удалит мастер-процесс.
     *
     * @param campaignIds идентификаторы кампаний для удалени из "очереди" ре-экспорта
     * @return количество обновленных записей
     */
    public int removeCampaignsFromFullExportQueue(Collection<Long> campaignIds) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID).stream()
                .mapKeyValue(bsExportQueueRepository::removeCampaignsFullExportFlag)
                .mapToInt(u -> u)
                .sum();
    }

    /**
     * Добавить кампании в очередь ре-экспорта в БК через LB.
     *
     * @param campaignIds идентификаторы кампаний для добавления в "очередь" ре-экспорта
     * @return количество обновленных записей
     */
    public int addCampaignsToFullExportQueue(Collection<Long> campaignIds) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID).stream()
                .mapKeyValue(bsExportQueueRepository::addCampaignsFullExportFlag)
                .mapToInt(u -> u)
                .sum();
    }

    /**
     * Получить экземпляр мастер-процесса для конкретного шарда
     *
     * @param shard шард который будет обрабатывать мастер-процесс
     */
    public Master getMaster(int shard) {
        return new Master(shard);
    }

    /**
     * Мастер-процесс ре-экспорта.
     * <p>
     * Термины:<ul>
     * <li><b>max_campaigns_in_queue</b> - наибольшее возможное количество кампаний в очереди экспорта в БК
     * с взведенным флагом is_full_export (в таблице bs_export_queue)</li>
     * <li><b>current_campaigns_in_queue</b> - текущее количество кампаний в таблице bs_export_queue с флагом is_full_export</li>
     * <li>{@code maximumChunkPerIteration} - ограничение на количество кампаний, добавляемых в очередь за один раз</li>
     * <li><b>chunk_per_worker</b> - какое количество кампаний поддерживать в очереди в расчете на один воркер</li>
     * <li><b>shard_workers</b> - число воркеров соответствующего типа из {@link BsExportParametersService#getWorkersNum(WorkerType, int)}</li>
     * <li>{@code lastProcessedCampaignId} - последний обработанный campaignId</li>
     * <li><b>allow_rolling_work</b> - разрешена ли циклическая (начинать с начала по завершению) работа</li>
     * </ul>
     */
    public class Master {
        private final Logger logger = LoggerFactory.getLogger(Master.class);
        private final int shard;

        private final PpcProperty<Long> lastProcessedCampaignIdProperty;

        private Master(int shard) {
            checkArgument(shard > 0, "shard number should be positive");
            this.shard = shard;
            lastProcessedCampaignIdProperty = parametersService.getFullExportLastProcessedCampaignIdProperty(shard);
        }

        /**
         * Выполнить одну итерацию мастер-процесса.
         * Значения {@code maximumCampaignsInQueue}, {@code maximumChunkPerIteration} и {@code lastProcessedCampaignId}
         * получаются из соответствующих ppc_property
         *
         * @see #iteration(int, int, long)
         */
        public void iteration() {
            int maximumCampaignsInQueue = getMaximumCampaignsInQueue();
            int maximumChunkPerIteration = parametersService.getFullExportMaximumChunkPerIteration();
            long lastProcessedCampaignId = lastProcessedCampaignIdProperty.getOrDefault(-1L);
            iteration(maximumCampaignsInQueue, maximumChunkPerIteration, lastProcessedCampaignId);
        }

        /**
         * Процесс переотправки обрабатывает кампании по следующему алгоритму (в пределах шарда):
         * <pre>
         * 0. Определеяет, что считать за max_campaigns_in_queue по формуле {@code
         *    if(chunk_per_worker > 0,
         *       min(maximumCampaignsInQueue, chunk_per_worker * shard_workers),
         *       maximumCampaignsInQueue)
         *    }
         * 1. Проверяет значение max_campaigns_in_queue. Если не больше ноля - ничего дальше не делаем
         * 2. Проверяет current_campaigns_in_queue:
         *  - Если оно больше max_campaigns_in_queue - ничего дальше не делаем.
         *  - Если меньше, то в очередь будет добавлено сколько-то (N) кампаний,
         *    но не больше чем {@code maximumChunkPerIteration}
         *    и не больше разницы max_campaigns_in_queue и current_campaigns_in_queue
         * 3. В зависимости от значения {@code lastProcessedCampaignId}:
         *  - определено и положительное - выбираем из campaigns cid'ы со statusEmpty = "No",
         *    начиная со значения {@code lastProcessedCampaignId - 1} и меньшие, количеством N (выборка по убыванию cid)
         *  - не больше ноля и allow_rolling_work ложно - дальше ничего не делаем
         *  - не больше ноля и allow_rolling_work истинно - выбираем в шарде max(cid) по условию statusEmpty = "No",
         *    сохраняем значение max(cid) + 1 в качестве {@code lastProcessedCampaignId},
         *    и выполняем обычные действия для п. 3.
         * </pre>
         *
         * @param maximumCampaignsInQueue  ограничение на количество кампаний в очереди на переотправку
         * @param maximumChunkPerIteration ограничение на количество добавляемых кампаний за одну итерацию
         * @param lastProcessedCampaignId  id последней кампании добавленной в прошлый раз
         */
        void iteration(int maximumCampaignsInQueue, int maximumChunkPerIteration, long lastProcessedCampaignId) {
            logger.info("Iteration parameters: lastProcessedCampaignId - {}"
                            + ", maximumChunkPerIteration - {}"
                            + ", maximumCampaignsInQueue - {}"
                            + " (shard {})",
                    lastProcessedCampaignId, maximumChunkPerIteration, maximumCampaignsInQueue, shard);

            if (maximumCampaignsInQueue <= 0) {
                logger.error("Skip iteration due to maximumCampaignsInQueue is not set");
                return;
            }
            if (maximumChunkPerIteration <= 0) {
                logger.error("Skip iteration due to maximumChunkPerIteration is not set");
                return;
            }

            logger.debug("Fetching current campaigns in queue count from DB");
            int campaignsCount = bsExportQueueRepository.getCampaignsCountInFullExportQueue(shard);
            if (campaignsCount >= maximumCampaignsInQueue) {
                logger.warn("Skip iteration - reached maximum campaigns in queue: count {}, limit {} (shard {})",
                        campaignsCount, maximumCampaignsInQueue, shard);
                return;
            }

            if (lastProcessedCampaignId <= 0) {
                logger.debug("lastProcessedCampaignId is not set, let's check can we proceed from start");
                if (parametersService.canFullExportRollingWork()) {
                    logger.debug("Fetching new lastProcessedCampaignId value from DB (shard {})", shard);
                    Long newLastProcessedCampaignId = campaignRepository.getNewLastProcessedCampaignId(shard);
                    if (newLastProcessedCampaignId == null || newLastProcessedCampaignId <= ID_NOT_SET) {
                        logger.warn("Skip iteration - no suitable lastProcessedCampaignId value"
                                + " (possible there are no campaigns in this shard)");
                        return;
                    } else {
                        logger.info("Update lastProcessedCampaignId value {} -> {} (shard {})",
                                lastProcessedCampaignId, newLastProcessedCampaignId, shard);
                        lastProcessedCampaignIdProperty.set(newLastProcessedCampaignId);
                        lastProcessedCampaignId = newLastProcessedCampaignId;
                    }
                } else {
                    logger.warn(
                            "Skip iteration due to lastProcessedCampaignId is not set and rolling work is not allowed");
                    return;
                }
            }

            int restLimit = Math.min(maximumChunkPerIteration, maximumCampaignsInQueue - campaignsCount);

            logger.debug("Fetching new {} campaigns for queuing", restLimit);
            List<Long> newIds =
                    campaignRepository.getCampaignIdsForFullExport(shard, lastProcessedCampaignId, restLimit);

            logger.debug("Adding campaigns to full lb export queue: {}", newIds);
            int rowsUpdated = bsExportQueueRepository.addCampaignsFullExportFlag(shard, newIds);
            logger.info("Successfully added {} campaigns, {} rows updated (shard {})",
                    newIds.size(), rowsUpdated, shard);

            long newLastProcessedCampaignId;
            if (newIds.size() == restLimit) {
                newLastProcessedCampaignId = newIds.get(newIds.size() - 1);
            } else {
                // выбралось меньше чем просили - значит больше ничего нет, останавливаемся
                newLastProcessedCampaignId = 0;
            }
            logger.info("Update lastProcessedCampaignId value {} -> {} (shard {})",
                    lastProcessedCampaignId, newLastProcessedCampaignId, shard);
            lastProcessedCampaignIdProperty.set(newLastProcessedCampaignId);
        }

        int getMaximumCampaignsInQueue() {
            int limit = parametersService.getFullExportMaximumCampaignsInQueue();
            int chunkPerWorker = parametersService.getFullExportChunkPerWorker();
            if (chunkPerWorker > 0) {
                int currentWorkersInShard = parametersService.getWorkersNum(WorkerType.FULL_LB_EXPORT, shard);
                return Math.min(limit, chunkPerWorker * currentWorkersInShard);
            } else {
                return limit;
            }
        }
    }
}
