package ru.yandex.direct.core.entity.bs.resync.queue.model;

import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;
import static ru.yandex.direct.utils.NumbersCheckUtils.checkInsideBounds;

/**
 * Описывает одну запись для ленивой переотправки в БК
 */
public class BsResyncItem {
    public static final long MIN_PRIORITY = -128;
    public static final long MAX_PRIORITY = 127;

    private final Long campaignId;
    private final Long bannerId;
    private final Long adgroupId;
    private final Long priority;

    /**
     * Создать запись для ленивой переотправки в БК кампании, баннера и/или группы.
     * <p>
     * если не заданы или равны нулю bannerId и adgroupId - будет переотправлена только кампания
     * если задан ненулевой bannerId - будет переотправлен баннер
     * если задан ненулевой adgroupId - будет переотправлено условие
     *
     * @param priority   приоритет объекта. обязательный параметр
     * @param campaignId номер кампании, объект которой нужно переотправить. обязательный параметр
     * @param bannerId   номер баннера для переотправки, опционально
     * @param adgroupId  номер группы для переотправки, опционально
     */
    public BsResyncItem(@Nonnull Long priority, @Nonnull Long campaignId, @Nullable Long bannerId,
                        @Nullable Long adgroupId) {
        checkNotNull(campaignId, "campaignId is required");
        this.campaignId = campaignId;

        checkNotNull(priority, "priority is required");
        checkInsideBounds(priority, MIN_PRIORITY, MAX_PRIORITY);
        this.priority = priority;

        if (bannerId != null) {
            checkArgument(bannerId >= 0, "invalid bannerId");
            this.bannerId = bannerId;
        } else {
            this.bannerId = ID_NOT_SET;
        }

        if (adgroupId != null) {
            checkArgument(adgroupId >= 0, "invalid adgroupId");
            this.adgroupId = adgroupId;
        } else {
            this.adgroupId = ID_NOT_SET;
        }
    }

    /**
     * @see #BsResyncItem(BsResyncPriority, Long, Long, Long)
     */
    public BsResyncItem(@Nonnull BsResyncPriority priority, @Nonnull Long campaignId, @Nullable Long bannerId,
                        @Nullable Long adgroupId) {
        this(priority.value(), campaignId, bannerId, adgroupId);
    }

    /**
     * Создать запись для ленивой переотправки в БК кампании.
     *
     * @param priority   приоритет с которым кампанию нужно добавить в очередь
     * @param campaignId номер кампании для переотправки
     */
    public BsResyncItem(@Nonnull BsResyncPriority priority, @Nonnull Long campaignId) {
        this(priority.value(), campaignId, null, null);
    }

    public BsResyncItem(BsResyncQueueInfo bsResyncQueueInfo) {
        this(bsResyncQueueInfo.getPriority(), bsResyncQueueInfo.getCampaignId(), bsResyncQueueInfo.getBannerId(),
                bsResyncQueueInfo.getAdgroupId());
    }

    public Long getCampaignId() {
        return campaignId;
    }

    public Long getBannerId() {
        return bannerId;
    }

    public Long getAdgroupId() {
        return adgroupId;
    }

    public Long getPriority() {
        return priority;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BsResyncItem that = (BsResyncItem) o;
        return campaignId.equals(that.campaignId) &&
                Objects.equals(bannerId, that.bannerId) &&
                Objects.equals(adgroupId, that.adgroupId) &&
                priority.equals(that.priority);
    }

    @Override
    public int hashCode() {
        return Objects.hash(campaignId, bannerId, adgroupId, priority);
    }

    @Override
    public String toString() {
        return "BsResyncItem{" +
                "campaignId=" + campaignId +
                ", bannerId=" + bannerId +
                ", adgroupId=" + adgroupId +
                ", priority=" + priority +
                '}';
    }
}
