package ru.yandex.direct.core.entity.bs.resync.queue.repository;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertOnDuplicateSetMoreStep;
import org.jooq.InsertValuesStep4;
import org.jooq.Record5;
import org.jooq.impl.DSL;
import org.jooq.types.DayToSecond;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncQueueInfo;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncQueueStat;
import ru.yandex.direct.dbschema.ppc.tables.records.BsResyncQueueRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.BsResyncQueue.BS_RESYNC_QUEUE;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;
import static ru.yandex.direct.dbutil.SqlUtils.NULL_FILED;
import static ru.yandex.direct.dbutil.SqlUtils.convertToDuration;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Работа с ленивой очередью переотправки в БК
 */
@Repository
@ParametersAreNonnullByDefault
public class BsResyncQueueRepository {
    private static final Field<BigDecimal> CAMPS_NUM_FIELD =
            DSL.sum(DSL.iif(BS_RESYNC_QUEUE.BID.eq(ID_NOT_SET).and(BS_RESYNC_QUEUE.PID.eq(ID_NOT_SET)), 1L, 0L))
                    .as("camps_num");
    private static final Field<BigDecimal> BANNERS_NUM_FIELD =
            DSL.sum(DSL.iif(BS_RESYNC_QUEUE.BID.gt(ID_NOT_SET), 1L, 0L)).as("banners_num");
    private static final Field<BigDecimal> CONTEXTS_NUM_FIELD =
            DSL.sum(DSL.iif(BS_RESYNC_QUEUE.PID.gt(ID_NOT_SET), 1L, 0L)).as("contexts_num");
    private static final Field<DayToSecond> MAX_AGE_FIELD =
            DSL.localDateTimeDiff(DSL.currentLocalDateTime(), DSL.min(BS_RESYNC_QUEUE.SEQUENCE_TIME)).as("max_age");

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<BsResyncQueueInfo> jooqMapper;

    @Autowired
    public BsResyncQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        jooqMapper = JooqMapperWithSupplierBuilder.builder(BsResyncQueueInfo::new)
                .map(property(BsResyncQueueInfo.CAMPAIGN_ID, BS_RESYNC_QUEUE.CID))
                .map(property(BsResyncQueueInfo.ADGROUP_ID, BS_RESYNC_QUEUE.PID))
                .map(property(BsResyncQueueInfo.BANNER_ID, BS_RESYNC_QUEUE.BID))
                .map(property(BsResyncQueueInfo.PRIORITY, BS_RESYNC_QUEUE.PRIORITY))
                .map(property(BsResyncQueueInfo.SEQUENCE_TIME, BS_RESYNC_QUEUE.SEQUENCE_TIME))
                .build();
    }

    /**
     * Постановка данных в очередь на переотправку в БК.
     * <p>
     * Для уже находящихся в очереди записей {@link BsResyncItem} вместо добавления обновляется приоритет:
     * он принимает наибольшее значение из текущего (в базе) и переданного (параметром).
     *
     * @param items данные для ленивой переотправки
     * @param shard номер шарда в который добавляем данные
     * @return кол-во добавленных записей
     * @see BsResyncItem
     */
    public int addToResync(int shard, Collection<BsResyncItem> items) {
        return addToResync(dslContextProvider.ppc(shard), items);
    }

    public int addToResync(DSLContext dslContext, Collection<BsResyncItem> items) {
        if (items.isEmpty()) {
            return 0;
        }

        InsertValuesStep4<BsResyncQueueRecord, Long, Long, Long, Long> insert = dslContext
                .insertInto(BS_RESYNC_QUEUE, BS_RESYNC_QUEUE.CID, BS_RESYNC_QUEUE.BID, BS_RESYNC_QUEUE.PID,
                        BS_RESYNC_QUEUE.PRIORITY);
        for (BsResyncItem item : items) {
            insert = insert.values(item.getCampaignId(), item.getBannerId(), item.getAdgroupId(), item.getPriority());
        }
        InsertOnDuplicateSetMoreStep<BsResyncQueueRecord> onDuplicateInsert = insert.onDuplicateKeyUpdate()
                .set(BS_RESYNC_QUEUE.PRIORITY,
                        DSL.greatest(BS_RESYNC_QUEUE.PRIORITY, MySQLDSL.values(BS_RESYNC_QUEUE.PRIORITY)));
        return onDuplicateInsert.execute();
    }


    /**
     * Получить список объектов кампании для постановки в очередь на переотправку в БК.
     *
     * @param shard       номер шарда
     * @param campaignIds список идентификаторов кампаний
     * @param priority    приоритет для проставления в бин, чтобы потом можно было сразу записать в очередь
     * @return список объектов для добавления в ленивую очередь
     */
    public List<BsResyncItem> fetchCampaignItemsForBsResync(int shard, Collection<Long> campaignIds,
                                                            BsResyncPriority priority) {
        return dslContextProvider.ppc(shard)
                .select(PHRASES.CID, PHRASES.PID, BANNERS.BID)
                .from(PHRASES)
                .join(BANNERS).on(BANNERS.PID.eq(PHRASES.PID))
                .where(PHRASES.CID.in(campaignIds))
                .fetch(record -> new BsResyncItem(priority, record.get(PHRASES.CID), record.get(BANNERS.BID),
                        record.get(PHRASES.PID)));
    }

    /**
     * Получить список объектов кампании из очереди.
     *
     * @param shard       номер шарда
     * @param campaignIds список идентификаторов кампаний
     * @return список объектов для добавления в ленивую очередь
     */
    public List<BsResyncQueueInfo> getCampaignItemsFromQueue(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(BS_RESYNC_QUEUE)
                .where(BS_RESYNC_QUEUE.CID.in(campaignIds))
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Получить статистику по очереди переотправки, сгруппированную по приоритетам
     *
     * @param shard шард для получения информации
     * @return список статистик для каждого из приоритетов, используемых в очереди
     */
    public List<BsResyncQueueStat> getQueueStat(int shard) {
        return getQueueStat(dslContextProvider.ppc(shard));
    }

    public List<BsResyncQueueStat> getQueueStat(DSLContext dslContext) {
        return dslContext
                .select(BS_RESYNC_QUEUE.PRIORITY, CAMPS_NUM_FIELD, CONTEXTS_NUM_FIELD, BANNERS_NUM_FIELD, MAX_AGE_FIELD)
                .from(BS_RESYNC_QUEUE)
                .groupBy(BS_RESYNC_QUEUE.PRIORITY)
                .orderBy(NULL_FILED)
                .fetch(BsResyncQueueRepository::statMapper);
    }

    private static BsResyncQueueStat statMapper(Record5<Long, BigDecimal, BigDecimal, BigDecimal, DayToSecond> record) {
        DayToSecond maxAge = record.get(MAX_AGE_FIELD);

        return new BsResyncQueueStat()
                .withPriority(record.get(BS_RESYNC_QUEUE.PRIORITY))
                .withCampaignsNum(record.get(CAMPS_NUM_FIELD).longValueExact())
                .withContextsNum(record.get(CONTEXTS_NUM_FIELD).longValueExact())
                .withBannersNum(record.get(BANNERS_NUM_FIELD).longValueExact())
                .withMaximumAge(convertToDuration(maxAge));
    }
}
