package ru.yandex.direct.core.entity.bs.resync.queue.service;

import java.time.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncQueueInfo;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncQueueStat;
import ru.yandex.direct.core.entity.bs.resync.queue.repository.BsResyncQueueRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;

import static java.util.stream.Collectors.counting;
import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.direct.common.configuration.CommonConfiguration.DIRECT_EXECUTOR_SERVICE;

@Service
public class BsResyncService {

    private static final int CHUNK_SIZE = 100_000;
    private static final long NO_OBJECTS_ADDED = 0L;

    private final ShardHelper shardHelper;
    private final BsResyncQueueRepository bsResyncQueueRepository;
    private final ExecutorService executorService;

    @Autowired
    public BsResyncService(ShardHelper shardHelper, BsResyncQueueRepository bsResyncQueueRepository,
                           @Qualifier(DIRECT_EXECUTOR_SERVICE) ExecutorService executorService) {
        this.shardHelper = shardHelper;
        this.bsResyncQueueRepository = bsResyncQueueRepository;
        this.executorService = executorService;
    }

    /**
     * Постановка данных уровня кампании вместе со всеми баннерами и условиями в очередь на переотправку в БК
     * Данные разбиваются на чанки {@link #CHUNK_SIZE}
     *
     * @param campaignIds список идентификаторов кампаний
     * @param priority    приоритет с которым кампанию нужно добавить в очередь
     * @return хеш с кол-вом добавленных объектов в очередь для каждой кампании: campaignId -> objectsCount
     * в objectsCount попадают случаи, если объекты уже были в очереди с необ. приоритетом
     */
    public Map<Long, Long> addWholeCampaignsToResync(Collection<Long> campaignIds, BsResyncPriority priority) {
        Map<Long, Long> addedObjectsCount = new HashMap<>(campaignIds.size());

        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .chunkedBy(CHUNK_SIZE)
                .forEach((shard, campaignIdsGroupByShard) -> {
                    List<BsResyncItem> bsResyncItems = bsResyncQueueRepository
                            .fetchCampaignItemsForBsResync(shard, campaignIdsGroupByShard, priority);

                    //add campaign to bsResyncItems
                    campaignIdsGroupByShard
                            .forEach(campaignId -> bsResyncItems.add(new BsResyncItem(priority, campaignId)));

                    Lists.partition(bsResyncItems, CHUNK_SIZE)
                            .forEach(partition -> bsResyncQueueRepository.addToResync(shard, partition));

                    addedObjectsCount.putAll(bsResyncItems.stream()
                            .collect(groupingBy(BsResyncItem::getCampaignId, counting()))
                    );
                });

        campaignIds.forEach(campaignId -> addedObjectsCount.putIfAbsent(campaignId, NO_OBJECTS_ADDED));

        return addedObjectsCount;
    }

    /**
     * Постановка данных в очередь на переотправку в БК
     * Данные разбиваются на чанки {@link #CHUNK_SIZE}
     *
     * @param bsResyncItems данные объектов, которые надо поставить в очередь
     * @return кол-во добавленных объектов в очередь
     */
    public long addObjectsToResync(Collection<BsResyncItem> bsResyncItems) {
        Map<Integer, Long> addedObjectsCountPerShard = new HashMap<>(shardHelper.dbShards().size());

        shardHelper.groupByShard(bsResyncItems, ShardKey.CID, BsResyncItem::getCampaignId)
                .chunkedBy(CHUNK_SIZE)
                .forEach((shard, bsResyncItemsGroupedByShard) -> {
                    long addedRecordsCount = bsResyncQueueRepository.addToResync(shard, bsResyncItemsGroupedByShard);

                    addedObjectsCountPerShard.merge(shard, addedRecordsCount, (x, y) -> x + y);
                });

        return addedObjectsCountPerShard.values().stream()
                .mapToLong(Long::intValue)
                .sum();
    }

    /**
     * Получение записей из очереди на переотправку в БК для указанных кампании по campaignIds
     *
     * @param campaignIds номера кампаний, для которых надо получить данные из очереди
     * @return данные из очереди
     */
    public List<BsResyncQueueInfo> getBsResyncItemsByCampaignIds(List<Long> campaignIds) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID, c -> c)
                .stream()
                .map(integerListEntry -> bsResyncQueueRepository
                        .getCampaignItemsFromQueue(integerListEntry.getKey(), integerListEntry.getValue()))
                .flatMap(List::stream)
                .collect(Collectors.toList());
    }

    /**
     * Получить статистику по очереди переотправки во всех шардах, сгруппированную по приоритетам
     *
     * @return список статистик для каждого из приоритетов, используемых в очереди
     * @implNote использует {@link #executorService} для параллельного опроса шардов
     */
    public Collection<BsResyncQueueStat> getQueueStatForAllShards() {
        return StreamEx
                .ofValues(shardHelper.forEachShardParallel(bsResyncQueueRepository::getQueueStat, executorService))
                .flatMap(Collection::stream)
                .toMap(BsResyncQueueStat::getPriority, Function.identity(), BsResyncService::mergeStat)
                .values();
    }

    /**
     * Обобщить статистику об очереди переотправки.
     * Количество объектов - суммируется, возраст и приоритет - берутся наибольшие.
     *
     * @return новый инстанс {@link BsResyncQueueStat} содержащий обобщенную статистику
     */
    public static BsResyncQueueStat mergeStat(BsResyncQueueStat stat1, BsResyncQueueStat stat2) {
        Duration age1 = stat1.getMaximumAge();
        Duration age2 = stat2.getMaximumAge();

        return new BsResyncQueueStat()
                .withCampaignsNum(stat1.getCampaignsNum() + stat2.getCampaignsNum())
                .withContextsNum(stat1.getContextsNum() + stat2.getContextsNum())
                .withBannersNum(stat1.getBannersNum() + stat2.getBannersNum())
                .withPriority(Math.max(stat1.getPriority(), stat2.getPriority()))
                .withMaximumAge(age1.compareTo(age2) > 0 ? age1 : age2);
    }
}
