package ru.yandex.direct.core.entity.calltrackingsettings;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.calltrackingsettings.repository.CalltrackingSettingsRepository;
import ru.yandex.direct.core.entity.calltrackingsettings.validation.CalltrackingSettingsValidationService;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Операция для добавления телефонов типа
 * {@link ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings}
 */
public class CalltrackingSettingsAddOperation extends SimpleAbstractAddOperation<CalltrackingSettings, Long> {

    private final CalltrackingSettingsRepository calltrackingSettingsRepository;
    private final TelephonyPhoneService telephonyPhoneService;

    private final ClientId clientId;
    private final Set<Long> clientAvailableCounterIds;
    private final Set<Long> operatorEditableCounterIds;
    private final Map<Long, String> domainByDomainId;

    public CalltrackingSettingsAddOperation(
            ClientId clientId,
            Set<Long> clientAvailableCounterIds,
            Set<Long> operatorEditableCounterIds,
            List<CalltrackingSettings> calltrackingSettings,
            Map<Long, String> domainByDomainId,
            TelephonyPhoneService telephonyPhoneService,
            CalltrackingSettingsRepository calltrackingSettingsRepository
    ) {
        super(Applicability.PARTIAL, calltrackingSettings);
        this.clientAvailableCounterIds = clientAvailableCounterIds;
        this.operatorEditableCounterIds = operatorEditableCounterIds;
        this.clientId = clientId;
        this.telephonyPhoneService = telephonyPhoneService;
        this.calltrackingSettingsRepository = calltrackingSettingsRepository;
        this.domainByDomainId = domainByDomainId;
    }

    @Override
    protected void validate(ValidationResult<List<CalltrackingSettings>, Defect> preValidationResult) {
        var validationResult = CalltrackingSettingsValidationService
                .validateCalltrackingSettingsList(
                        preValidationResult, clientAvailableCounterIds, operatorEditableCounterIds, domainByDomainId);
        preValidationResult.merge(validationResult);
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<CalltrackingSettings> modelsValidatedStep) {
        Collection<CalltrackingSettings> calltrackingSettings = modelsValidatedStep.getValidModelsMap().values();
        calltrackingSettings.forEach(s -> s.setClientId(clientId));
        LocalDateTime now = LocalDateTime.now();
        calltrackingSettings.stream()
                .map(CalltrackingSettings::getPhonesToTrack)
                .flatMap(Collection::stream)
                .forEach(p -> p.setCreateTime(now));
    }

    @Override
    protected List<Long> execute(List<CalltrackingSettings> validModelsToApply) {
        return calltrackingSettingsRepository.add(clientId, validModelsToApply);
    }

    @Override
    protected void onExecuted(Map<Integer, CalltrackingSettings> validModelsMapToApply) {
        Set<Long> counterIdsToEnableCalltracking = StreamEx.of(validModelsMapToApply.values())
                .map(CalltrackingSettings::getCounterId)
                .toSet();
        if (!counterIdsToEnableCalltracking.isEmpty()) {
            telephonyPhoneService.tryEnableCalltrackingForCounters(counterIdsToEnableCalltracking);
        }
    }
}
