package ru.yandex.direct.core.entity.calltrackingsettings;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.calltracking.model.SettingsPhone;
import ru.yandex.direct.core.entity.calltrackingsettings.repository.CalltrackingSettingsRepository;
import ru.yandex.direct.core.entity.calltrackingsettings.validation.CalltrackingSettingsValidationService;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция для добавления настроек колтрекинга на сейте
 * {@link ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings}
 */
public class CalltrackingSettingsUpdateOperation extends SimpleAbstractUpdateOperation<CalltrackingSettings, Long> {

    private final CalltrackingSettingsRepository calltrackingSettingsRepository;
    private final TelephonyPhoneService telephonyPhoneService;
    private final ModelChangesValidationTool updateValidationTool;

    private final int shard;
    private final ClientId clientId;
    private final Set<Long> clientAvailableCounterIds;
    private final Set<Long> operatorEditableCounterIds;
    private final Map<Long, String> domainIdByDomain;

    public CalltrackingSettingsUpdateOperation(
            int shard,
            ClientId clientId,
            List<ModelChanges<CalltrackingSettings>> modelChanges,
            Set<Long> clientAvailableCounterIds,
            Set<Long> operatorEditableCounterIds,
            Map<Long, String> domainByDomainId,
            TelephonyPhoneService telephonyPhoneService,
            CalltrackingSettingsRepository calltrackingSettingsRepository
    ) {
        super(Applicability.PARTIAL, modelChanges, id -> new CalltrackingSettings().withCalltrackingSettingsId(id));
        this.shard = shard;
        this.clientId = clientId;
        this.clientAvailableCounterIds = clientAvailableCounterIds;
        this.operatorEditableCounterIds = operatorEditableCounterIds;
        this.domainIdByDomain = domainByDomainId;
        this.telephonyPhoneService = telephonyPhoneService;
        this.calltrackingSettingsRepository = calltrackingSettingsRepository;
        updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    @Override
    protected ValidationResult<List<ModelChanges<CalltrackingSettings>>, Defect> validateModelChanges(
            List<ModelChanges<CalltrackingSettings>> modelChanges
    ) {
        List<Long> ids = mapList(modelChanges, ModelChanges::getId);
        List<CalltrackingSettings> calltrackingSettings = calltrackingSettingsRepository.getByIds(clientId, ids);
        Set<Long> existingIds = listToSet(calltrackingSettings, CalltrackingSettings::getId);
        return updateValidationTool.validateModelChangesList(modelChanges, existingIds);
    }

    @Override
    protected ValidationResult<List<CalltrackingSettings>, Defect> validateAppliedChanges(
            ValidationResult<List<CalltrackingSettings>, Defect> validationResult) {
        return CalltrackingSettingsValidationService.validateCalltrackingSettingsList(
                validationResult, clientAvailableCounterIds, operatorEditableCounterIds, domainIdByDomain);
    }

    @Override
    protected Collection<CalltrackingSettings> getModels(Collection<Long> calltrackingSettingsIds) {
        return calltrackingSettingsRepository.getByIds(clientId, calltrackingSettingsIds);
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<CalltrackingSettings> appliedChangesValidatedStep) {
        LocalDateTime now = LocalDateTime.now();
        appliedChangesValidatedStep.getValidAppliedChanges().forEach(c -> {
            List<SettingsPhone> oldPhones =
                    c.getOldValue(CalltrackingSettings.PHONES_TO_TRACK);
            Map<String, SettingsPhone> oldPhonesMap = listToMap(oldPhones, SettingsPhone::getPhone);
            List<SettingsPhone> newPhones =
                    c.getNewValue(CalltrackingSettings.PHONES_TO_TRACK);
            Map<String, SettingsPhone> newPhonesMap = listToMap(newPhones, SettingsPhone::getPhone);
            if (newPhonesMap != null) {
                newPhonesMap.keySet().forEach(p -> {
                    if (oldPhonesMap == null || !oldPhonesMap.containsKey(p)) {
                        newPhonesMap.get(p).setCreateTime(now);
                    } else {
                        newPhonesMap.get(p).setCreateTime(oldPhonesMap.get(p).getCreateTime());
                    }
                });
                c.modify(CalltrackingSettings.PHONES_TO_TRACK,
                        new ArrayList<>(newPhonesMap.values()));
            }
        });
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<CalltrackingSettings>> appliedChanges) {
        calltrackingSettingsRepository.update(shard, appliedChanges);
        return mapList(appliedChanges, c -> c.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<CalltrackingSettings> executionStep) {
        Collection<AppliedChanges<CalltrackingSettings>> appliedChanges =
                executionStep.getAppliedChangesForExecution();
        Set<Long> countersToEnableCalltracking = appliedChanges.stream()
                .filter(ac -> ac.changed(CalltrackingSettings.COUNTER_ID))
                .map(ac -> ac.getNewValue(CalltrackingSettings.COUNTER_ID))
                .collect(Collectors.toSet());
        if (!countersToEnableCalltracking.isEmpty()) {
            telephonyPhoneService.tryEnableCalltrackingForCounters(countersToEnableCalltracking);
        }
    }
}
