package ru.yandex.direct.core.entity.calltrackingsettings.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettingsWithCampaignId;
import ru.yandex.direct.core.entity.calltrackingsettings.repository.mapper.CalltrackingSettingsMapper;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.dbschema.ppc.tables.CalltrackingSettings.CALLTRACKING_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.tables.CampCalltrackingSettings.CAMP_CALLTRACKING_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.tables.Domains.DOMAINS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapperhelper.InsertHelper.saveModelObjectsToDbTable;

@Repository
@ParametersAreNonnullByDefault
public class CalltrackingSettingsRepository {
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    private final JooqMapperWithSupplier<CalltrackingSettings> mapper = createMapper();
    private final JooqMapperWithSupplier<CalltrackingSettingsWithCampaignId> mapperWithId = createMapperWithCid();

    private JooqMapperWithSupplier<CalltrackingSettings> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(CalltrackingSettings::new)
                .map(property(CalltrackingSettings.CALLTRACKING_SETTINGS_ID,
                        CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID))
                .map(clientIdProperty(CalltrackingSettings.CLIENT_ID, CALLTRACKING_SETTINGS.CLIENT_ID))
                .map(property(CalltrackingSettings.DOMAIN_ID, CALLTRACKING_SETTINGS.DOMAIN_ID))
                .map(property(CalltrackingSettings.COUNTER_ID, CALLTRACKING_SETTINGS.COUNTER_ID))
                .map(convertibleProperty(CalltrackingSettings.PHONES_TO_TRACK, CALLTRACKING_SETTINGS.PHONES_TO_TRACK,
                        CalltrackingSettingsMapper::phonesToTrackFromJson,
                        CalltrackingSettingsMapper::phonesToTrackToJson))
                .map(convertibleProperty(CalltrackingSettings.IS_AVAILABLE_COUNTER, CALLTRACKING_SETTINGS.IS_AVAILABLE_COUNTER,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLongDefaultTrue))
                .build();
    }

    private JooqMapperWithSupplier<CalltrackingSettingsWithCampaignId> createMapperWithCid() {
        return JooqMapperWithSupplierBuilder.builder(CalltrackingSettingsWithCampaignId::new)
                .map(property(CalltrackingSettingsWithCampaignId.CALLTRACKING_SETTINGS_ID,
                        CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID))
                .map(clientIdProperty(CalltrackingSettingsWithCampaignId.CLIENT_ID, CALLTRACKING_SETTINGS.CLIENT_ID))
                .map(property(CalltrackingSettingsWithCampaignId.DOMAIN_ID, CALLTRACKING_SETTINGS.DOMAIN_ID))
                .map(property(CalltrackingSettingsWithCampaignId.COUNTER_ID, CALLTRACKING_SETTINGS.COUNTER_ID))
                .map(convertibleProperty(CalltrackingSettingsWithCampaignId.PHONES_TO_TRACK, CALLTRACKING_SETTINGS.PHONES_TO_TRACK,
                        CalltrackingSettingsMapper::phonesToTrackFromJson,
                        CalltrackingSettingsMapper::phonesToTrackToJson))
                .map(convertibleProperty(CalltrackingSettingsWithCampaignId.IS_AVAILABLE_COUNTER, CALLTRACKING_SETTINGS.IS_AVAILABLE_COUNTER,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLongDefaultTrue))
                .map(convertibleProperty(CalltrackingSettingsWithCampaignId.CAMPAIGN_ID, CAMP_CALLTRACKING_SETTINGS.CID,
                        RepositoryUtils::zeroToNull, RepositoryUtils::nullToZero))
                .build();
    }

    public CalltrackingSettingsRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    public List<Long> add(ClientId clientId, Collection<CalltrackingSettings> calltrackingSettings) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<Long> ids = shardHelper.generateCalltrackingSettingsId(calltrackingSettings.size());

        StreamEx.of(calltrackingSettings).zipWith(ids.stream())
                .forKeyValue(CalltrackingSettings::setCalltrackingSettingsId);

        saveModelObjectsToDbTable(dslContextProvider.ppc(shard), CALLTRACKING_SETTINGS, mapper, calltrackingSettings);
        return ids;
    }

    public void update(int shard, Collection<AppliedChanges<CalltrackingSettings>> appliedChanges) {
        DSLContext context = dslContextProvider.ppc(shard);

        new UpdateHelper<>(context, CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID)
                .processUpdateAll(mapper, appliedChanges)
                .execute();
    }

    public void updateCounterAvailability(int shard, ClientId clientId, Collection<Long> counterIds, boolean isAvailable) {
        dslContextProvider.ppc(shard)
        .update(CALLTRACKING_SETTINGS)
        .set(CALLTRACKING_SETTINGS.IS_AVAILABLE_COUNTER, RepositoryUtils.booleanToLong(isAvailable))
        .where(DSL.and(
                CALLTRACKING_SETTINGS.CLIENT_ID.eq(clientId.asLong()),
                CALLTRACKING_SETTINGS.COUNTER_ID.in(counterIds)
        )).execute();
    }

    public void updateCounterAvailability(int shard, Collection<Long> counterIds, boolean isAvailable) {
        dslContextProvider.ppc(shard)
                .update(CALLTRACKING_SETTINGS)
                .set(CALLTRACKING_SETTINGS.IS_AVAILABLE_COUNTER, RepositoryUtils.booleanToLong(isAvailable))
                .where(CALLTRACKING_SETTINGS.COUNTER_ID.in(counterIds))
                .execute();
    }

    public List<CalltrackingSettings> getByIds(int shard, ClientId clientId, Collection<Long> calltrackingSettingsIds) {
        Condition idCondition = CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.in(calltrackingSettingsIds);
        Condition clientIdCondition = CALLTRACKING_SETTINGS.CLIENT_ID.eq(clientId.asLong());
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CALLTRACKING_SETTINGS)
                .where(idCondition.and(clientIdCondition))
                .fetch(mapper::fromDb);
    }

    public List<CalltrackingSettings> getByIds(ClientId clientId, Collection<Long> calltrackingSettingsIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return getByIds(shard, clientId, calltrackingSettingsIds);
    }

    public Optional<CalltrackingSettings> getByDomainId(int shard, ClientId clientId, Long domainId) {
        return Optional.ofNullable(getByDomainIds(shard, clientId, List.of(domainId)).get(domainId));
    }

    public Map<Long, CalltrackingSettings> getByDomainIds(int shard, ClientId clientId, Collection<Long> domainIds) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CALLTRACKING_SETTINGS)
                .where(DSL.and(
                        CALLTRACKING_SETTINGS.CLIENT_ID.eq(clientId.asLong()),
                        CALLTRACKING_SETTINGS.DOMAIN_ID.in(domainIds)))
                .fetchMap(r -> r.getValue(CALLTRACKING_SETTINGS.DOMAIN_ID), mapper::fromDb);
    }

    public List<CalltrackingSettings> getByDomain(int shard, ClientId clientId, Collection<String> domains) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CALLTRACKING_SETTINGS)
                .join(DOMAINS)
                .on(DOMAINS.DOMAIN_ID.eq(CALLTRACKING_SETTINGS.DOMAIN_ID))
                .where(CALLTRACKING_SETTINGS.CLIENT_ID.eq(clientId.asLong()))
                .and(DOMAINS.DOMAIN.in(domains))
                .fetch(mapper::fromDb);
    }

    public Map<ClientId, List<CalltrackingSettingsWithCampaignId>> getWithCampaignId(int shard) {
        return dslContextProvider.ppc(shard)
                .select(mapperWithId.getFieldsToRead())
                .from(CALLTRACKING_SETTINGS)
                .join(CAMP_CALLTRACKING_SETTINGS)
                .on(CAMP_CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID
                        .eq(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID))
                .fetchGroups(
                        r -> ClientId.fromLong(r.getValue(CALLTRACKING_SETTINGS.CLIENT_ID)),
                        mapperWithId::fromDb);
    }

    public Map<Long, Long> getDomainIdsByCalltrackingSettingsIds(int shard, ClientId clientId, Collection<Long> calltrackingSettingsIds) {
        if (calltrackingSettingsIds.isEmpty()) {
            return Collections.emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(List.of(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID, CALLTRACKING_SETTINGS.DOMAIN_ID))
                .from(CALLTRACKING_SETTINGS)
                .where(
                        DSL.and(
                                CALLTRACKING_SETTINGS.CLIENT_ID.eq(clientId.asLong()),
                                CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.in(calltrackingSettingsIds)
                        )
                )
                .fetchMap(
                        row -> row.getValue(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID),
                        row -> row.getValue(CALLTRACKING_SETTINGS.DOMAIN_ID)
                );
    }

    public Map<Long, Long> getCalltrackingSettingsIdsByDomainIds(int shard, ClientId clientId, Collection<Long> domainIds) {
        if (domainIds.isEmpty()) {
            return Collections.emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(List.of(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID, CALLTRACKING_SETTINGS.DOMAIN_ID))
                .from(CALLTRACKING_SETTINGS)
                .where(
                        DSL.and(
                                CALLTRACKING_SETTINGS.CLIENT_ID.eq(clientId.asLong()),
                                CALLTRACKING_SETTINGS.DOMAIN_ID.in(domainIds)
                        )
                )
                .fetchMap(
                        row -> row.getValue(CALLTRACKING_SETTINGS.DOMAIN_ID),
                        row -> row.getValue(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID)
                );
    }

    public CalltrackingSettings getById(int shard, long calltrackingSettingsId) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(CALLTRACKING_SETTINGS)
                .where(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.eq(calltrackingSettingsId))
                .fetchOne(mapper::fromDb);
    }

    public void updatePhonesToTrack(int shard, String jsonPhonesToTrack, long calltrackingSettingsId) {
        dslContextProvider.ppc(shard)
                .update(CALLTRACKING_SETTINGS)
                .set(CALLTRACKING_SETTINGS.PHONES_TO_TRACK, jsonPhonesToTrack)
                .where(CALLTRACKING_SETTINGS.CALLTRACKING_SETTINGS_ID.eq(calltrackingSettingsId))
                .execute();
    }
}
