package ru.yandex.direct.core.entity.calltrackingsettings.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.MapUtils;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettingsInput;
import ru.yandex.direct.core.entity.calltracking.model.SettingsPhone;
import ru.yandex.direct.core.entity.calltrackingsettings.CalltrackingSettingsAddOperation;
import ru.yandex.direct.core.entity.calltrackingsettings.CalltrackingSettingsUpdateOperation;
import ru.yandex.direct.core.entity.calltrackingsettings.repository.CalltrackingSettingsRepository;
import ru.yandex.direct.core.entity.calltrackingsettings.repository.mapper.CalltrackingSettingsMapper;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.campcalltrackingsettings.repository.CampCalltrackingSettingsRepository;
import ru.yandex.direct.core.entity.clientphone.TelephonyPhoneService;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
public class CalltrackingSettingsService {

    private final CalltrackingSettingsRepository calltrackingSettingsRepository;
    private final TelephonyPhoneService telephonyPhoneService;
    private final CampCalltrackingSettingsRepository campCalltrackingSettingsRepository;
    private final BannersUrlHelper bannersUrlHelper;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final DomainService domainService;
    private final MetrikaGoalsService metrikaGoalsService;
    private final ShardHelper shardHelper;

    public CalltrackingSettingsService(
            CalltrackingSettingsRepository calltrackingSettingsRepository,
            TelephonyPhoneService telephonyPhoneService,
            CampCalltrackingSettingsRepository campCalltrackingSettingsRepository,
            BannersUrlHelper bannersUrlHelper,
            CampMetrikaCountersService campMetrikaCountersService,
            DomainService domainService,
            MetrikaGoalsService metrikaGoalsService,
            ShardHelper shardHelper
    ) {
        this.calltrackingSettingsRepository = calltrackingSettingsRepository;
        this.telephonyPhoneService = telephonyPhoneService;
        this.campCalltrackingSettingsRepository = campCalltrackingSettingsRepository;
        this.bannersUrlHelper = bannersUrlHelper;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.domainService = domainService;
        this.metrikaGoalsService = metrikaGoalsService;
        this.shardHelper = shardHelper;
    }

    public MassResult<Long> save(
            int shard,
            ClientId clientId,
            Long operatorUid,
            List<CalltrackingSettingsInput> items,
            Map<String, String> domainByUrl,
            Map<String, Long> domainIdByDomain
    ) {
        var operation = buildOperation(shard, clientId, operatorUid, domainIdByDomain);
        var coreModels = toCoreModel(shard, clientId, items, domainByUrl, domainIdByDomain);
        var massResult = operation.execute(coreModels);
        return SplitAndMergeOperationAggregator.getMassResultWithAggregatedValidationResult(massResult);
    }

    public Result<Long> save(
            ClientId clientId,
            Long operatorUid,
            String url,
            Long counterId,
            List<String> phones
    ) {
        int shard = shardHelper.getShardByClientId(clientId);
        var items = List.of(
                new CalltrackingSettingsInput()
                        .withUrl(url)
                        .withCounterId(counterId)
                        .withPhones(phones)
        );
        Map<String, String> domainByUrl = getDomainByUrl(items);
        Map<String, Long> domainIdByDomain = domainService.getOrCreateDomainIdByDomain(shard, domainByUrl.values());
        return save(shard, clientId, operatorUid, items, domainByUrl, domainIdByDomain).get(0);
    }

    public Map<Long, Goal> getCallGoals(ClientId clientId, Long operatorUid, @Nullable List<Integer> counterIds) {
        if (isEmpty(counterIds)) {
            return emptyMap();
        }
        var goals = metrikaGoalsService.getMetrikaGoalsByCounter(
                operatorUid,
                clientId,
                mapList(counterIds, Integer::longValue),
                null,
                CampaignType.TEXT
        );
        return StreamEx.of(goals)
                .filter(g -> g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.CALL)
                .toMap(Goal::getId, Function.identity());
    }

    private CalltrackingSettingsAddOperation getAddOperation(
            ClientId clientId,
            List<CalltrackingSettings> calltrackingSettings,
            Set<Long> clientAvailableCounterIds,
            Set<Long> operatorEditableCounterIds,
            Map<Long, String> domainIdByDomain
    ) {
        return new CalltrackingSettingsAddOperation(
                clientId,
                clientAvailableCounterIds,
                operatorEditableCounterIds,
                calltrackingSettings,
                domainIdByDomain,
                telephonyPhoneService,
                calltrackingSettingsRepository
        );
    }

    private CalltrackingSettingsUpdateOperation getUpdateOperation(
            int shard,
            ClientId clientId,
            List<ModelChanges<CalltrackingSettings>> modelChanges,
            Set<Long> clientAvailableCounterIds,
            Set<Long> operatorEditableCounterIds,
            Map<Long, String> domainByDomainId
    ) {
        return new CalltrackingSettingsUpdateOperation(
                shard,
                clientId,
                modelChanges,
                clientAvailableCounterIds,
                operatorEditableCounterIds,
                domainByDomainId,
                telephonyPhoneService,
                calltrackingSettingsRepository
        );
    }

    public void updateCounterAvailability(
            int shard,
            ClientId clientId,
            Set<Long> availableCounterIds,
            Set<Long> notAvailableCounterIds
    ) {
        calltrackingSettingsRepository.updateCounterAvailability(shard, clientId, availableCounterIds, true);
        calltrackingSettingsRepository.updateCounterAvailability(shard, clientId, notAvailableCounterIds, false);
    }

    public List<CalltrackingSettings> getByIds(int shard, ClientId clientId, Collection<Long> calltrackingSettingsIds) {
        return calltrackingSettingsRepository.getByIds(shard, clientId, calltrackingSettingsIds);
    }

    public List<CalltrackingSettings> getByDomain(int shard, ClientId clientId, Collection<String> domains) {
        return calltrackingSettingsRepository.getByDomain(shard, clientId, domains);
    }

    public Map<Long, CalltrackingSettings> getByDomainIds(int shard, ClientId clientId, Collection<Long> domainIds) {
        return calltrackingSettingsRepository.getByDomainIds(shard, clientId, domainIds);
    }

    public Optional<CalltrackingSettings> getByDomainId(int shard, ClientId clientId, Long domainId) {
        return calltrackingSettingsRepository.getByDomainId(shard, clientId, domainId);
    }

    public void resetSettings(int shard, long calltrackingSettingsId, Set<String> resetPhones, LocalDateTime now) {
        CalltrackingSettings settings = calltrackingSettingsRepository.getById(shard, calltrackingSettingsId);
        List<SettingsPhone> settingsPhones = settings.getPhonesToTrack();
        settingsPhones.forEach(s -> {
            if (resetPhones.contains(s.getPhone())) {
                s.setCreateTime(now);
            }
        });
        String jsonPhonesToTrack = CalltrackingSettingsMapper.phonesToTrackToJson(settingsPhones);
        calltrackingSettingsRepository.updatePhonesToTrack(shard, jsonPhonesToTrack, calltrackingSettingsId);
    }

    public Map<String, String> getDomainByUrl(List<CalltrackingSettingsInput> items) {
        Map<String, String> domainByUrl = new HashMap<>();
        for (var item : items) {
            String url = item.getUrl();
            String domain = bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(url);
            if (domain != null) {
                domainByUrl.put(url, domain);
            }
        }
        return domainByUrl;
    }

    @Nonnull
    public Set<Long> getCampaignIds(ClientId clientId, String url) {
        int shard = shardHelper.getShardByClientId(clientId);
        var calltrackingSettings = getByUrl(shard, clientId, url);
        if (calltrackingSettings == null) {
            return emptySet();
        }
        return getCampaignIds(shard, calltrackingSettings.getId());
    }

    @Nonnull
    public Set<Long> getCampaignIds(int shard, Long calltrackingSettingsId) {
        return campCalltrackingSettingsRepository.getCampaignIds(shard, calltrackingSettingsId);
    }

    @Nullable
    public CalltrackingSettings getByUrl(int shard, ClientId clientId, String url) {
        String domain = bannersUrlHelper.extractHostFromHrefWithoutWwwOrNull(url);
        if (domain == null) {
            return null;
        }
        var calltrackingSettingsList = getByDomain(shard, clientId, List.of(domain));
        if (calltrackingSettingsList.isEmpty()) {
            return null;
        }
        return calltrackingSettingsList.get(0);
    }

    private SplitAndMergeOperationAggregator<CalltrackingSettings, Long> buildOperation(
            int shard,
            ClientId clientId,
            Long operatorUid,
            Map<String, Long> domainIdByDomain
    ) {
        var countersContainer =
                campMetrikaCountersService.getAvailableCountersByClientAndOperatorUid(clientId, operatorUid);
        var clientCounterIds =
                mapSet(countersContainer.getClientAvailableCounters(), MetrikaCounterWithAdditionalInformation::getId);
        var operatorEditableCounterIds = countersContainer.getOperatorEditableCounterIds();
        var domainByDomainId = MapUtils.invertMap(domainIdByDomain);
        OperationCreator<CalltrackingSettings, Operation<Long>> addOperationCreator = items ->
                getAddOperation(clientId, items, clientCounterIds, operatorEditableCounterIds, domainByDomainId);

        OperationCreator<CalltrackingSettings, Operation<Long>> updateOperationCreator =
                items -> {
                    var modelChanges = mapList(
                            items,
                            item -> new ModelChanges<>(item.getId(), CalltrackingSettings.class)
                                    .process(item.getPhonesToTrack(), CalltrackingSettings.PHONES_TO_TRACK)
                                    .process(item.getCounterId(), CalltrackingSettings.COUNTER_ID)
                                    .process(true, CalltrackingSettings.IS_AVAILABLE_COUNTER));
                    return getUpdateOperation(
                            shard,
                            clientId,
                            modelChanges,
                            clientCounterIds,
                            operatorEditableCounterIds,
                            domainByDomainId
                    );
                };

        return SplitAndMergeOperationAggregator.builderForPartialOperations()
                .addSubOperation(
                        settings -> settings.getId() == null,
                        addOperationCreator
                ).addSubOperation(
                        settings -> settings.getId() != null,
                        updateOperationCreator
                ).build();
    }

    private List<CalltrackingSettings> toCoreModel(
            int shard,
            ClientId clientId,
            List<CalltrackingSettingsInput> items,
            Map<String, String> domainByUrl,
            Map<String, Long> domainIdByDomain
    ) {
        Map<String, Long> domainIdByUrl = getDomainIdByUrl(items, domainByUrl, domainIdByDomain);
        var settingsByDomainId = getByDomainIds(shard, clientId, domainIdByUrl.values());
        return mapList(
                items,
                item -> {
                    Long domainId = domainIdByUrl.get(item.getUrl());
                    CalltrackingSettings existingSettings = settingsByDomainId.get(domainId);
                    Long id = existingSettings == null ? null : existingSettings.getId();
                    List<SettingsPhone> phonesToTrack = StreamEx.of(item.getPhones())
                            .distinct()
                            .map(p -> new SettingsPhone().withPhone(p))
                            .toList();
                    return new CalltrackingSettings()
                            .withCalltrackingSettingsId(id)
                            .withClientId(clientId)
                            .withDomainId(domainId)
                            .withCounterId(item.getCounterId())
                            .withPhonesToTrack(phonesToTrack);
                }
        );
    }

    private Map<String, Long> getDomainIdByUrl(
            List<CalltrackingSettingsInput> items,
            Map<String, String> domainByUrl,
            Map<String, Long> domainIdByDomain
    ) {
        Map<String, Long> domainIdByUrl = new HashMap<>();
        for (var item : items) {
            var url = item.getUrl();
            var domain = domainByUrl.get(url);
            if (domain != null) {
                var domainId = domainIdByDomain.get(domain);
                domainIdByUrl.put(url, domainId);
            }
        }
        return domainIdByUrl;
    }
}
