package ru.yandex.direct.core.entity.calltrackingsettings.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.direct.core.entity.calltracking.model.CalltrackingSettings;
import ru.yandex.direct.core.entity.calltracking.model.SettingsPhone;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper.YANDEX_DOMAINS_PATTERN;
import static ru.yandex.direct.core.entity.calltrackingsettings.validation.CalltrackingSettingDefects.counterNotAvailableForClient;
import static ru.yandex.direct.core.entity.calltrackingsettings.validation.CalltrackingSettingDefects.forbiddenDomain;
import static ru.yandex.direct.core.entity.calltrackingsettings.validation.CalltrackingSettingDefects.noWritePermissionsOnCounter;
import static ru.yandex.direct.core.entity.clientphone.validation.ClientPhoneValidationService.rusNumberConstraint;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidFormat;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

public class CalltrackingSettingsValidationService {
    private static final int MAX_PHONES_COUNT_PER_SETTINGS = 5;

    private CalltrackingSettingsValidationService() {
    }

    public static ValidationResult<List<CalltrackingSettings>, Defect> validateCalltrackingSettingsList(
            ValidationResult<List<CalltrackingSettings>, Defect> validationResult,
            Set<Long> clientAvailableCounterIds,
            Set<Long> operatorEditableCounterIds,
            Map<Long, String> domainByDomainId
    ) {
        ListValidationBuilder<CalltrackingSettings, Defect> lvb = ListValidationBuilder.of(validationResult.getValue());
        lvb.checkEach(notNull())
                .checkEachBy(cs -> validateCalltrackingSettings(cs, clientAvailableCounterIds,
                        operatorEditableCounterIds, domainByDomainId), When.isValid());
        return lvb.getResult();
    }

    public static ValidationResult<CalltrackingSettings, Defect> validateCalltrackingSettings(
            CalltrackingSettings calltrackingSettings,
            Set<Long> clientAvailableCounterIds,
            Set<Long> operatorEditableCounterIds,
            Map<Long, String> domainByDomainId
    ) {
        ModelItemValidationBuilder<CalltrackingSettings> vb = ModelItemValidationBuilder.of(calltrackingSettings);
        String domain = domainByDomainId.get(calltrackingSettings.getDomainId());
        vb.item(CalltrackingSettings.DOMAIN_ID)
                .check(notNull(), invalidValue())
                .check(unconditional(forbiddenDomain()), When.isTrue(isDomainForbidden(domain)));

        vb.item(CalltrackingSettings.COUNTER_ID)
                .check(notNull())
                .checkByFunction(counterId -> validateCounterPermissions(
                        counterId, clientAvailableCounterIds, operatorEditableCounterIds));

        var item = vb.item(CalltrackingSettings.PHONES_TO_TRACK);
        item.check(maxListSize(MAX_PHONES_COUNT_PER_SETTINGS));

        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        var phones = calltrackingSettings.getPhonesToTrack()
                .stream()
                .map(SettingsPhone::getPhone)
                .collect(Collectors.toList());
        var lvb = item.list(phones, "redirectPhone");
        lvb.checkEach(rusNumberConstraint(invalidFormat()));

        return vb.getResult();
    }

    private static Defect<Void> validateCounterPermissions(Long counterId,
                                                           Set<Long> clientAvailableCounterIds,
                                                           Set<Long> operatorEditableCounterIds) {
        if (!clientAvailableCounterIds.contains(counterId)) {
            return counterNotAvailableForClient();
        }

        if (!operatorEditableCounterIds.contains(counterId)) {
            return noWritePermissionsOnCounter();
        }

        return null;
    }

    private static boolean isDomainForbidden(String domain) {
        return domain != null && YANDEX_DOMAINS_PATTERN.matcher(domain).matches();
    }
}
