package ru.yandex.direct.core.entity.campaign;

import java.math.BigDecimal;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.core.entity.client.model.ClientAutoOverdraftInfo;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.currency.CurrencyCode.AUTO_OVERDRAFT_AVAILABLE_CURRENCIES;
import static ru.yandex.direct.utils.math.MathUtils.max;
import static ru.yandex.direct.utils.math.MathUtils.min;

/**
 * Логика расчёта денег для автоовердрафта (порог отключения).
 */
@ParametersAreNonnullByDefault
public class AutoOverdraftUtils {
    // Список клиентов с двумя активными кошельками. см DIRECT-87190
    public static final Set<Long> CLIENT_IDS_WITH_TWO_ACTIVE_WALLETS = ImmutableSet.of(
            656873L, 958950L, 1026231L, 1627773L, 3131528L,
            3205125L, 4081112L, 1703366L, 6870093L, 15814128L, 6652704L, 2995358L, 5508797L,
            6833998L, 7841354L, 7817363L, 8247544L, 8877584L, 9293020L, 10970263L, 14722902L);

    public static final Set<CurrencyCode> AVAILABLE_CURRENCY_CODES = AUTO_OVERDRAFT_AVAILABLE_CURRENCIES;
    public static final BigDecimal AUTOOVERDRAFT_LIMIT_DEFAULT_MIN_VALUE = new BigDecimal("1");

    /**
     * Возвращает true, если к кошельку клиента применим автоовердрафт
     * Это зависит от значения валюты кошелька, значения порога отключения (auto_overdraft_lim) и ClientID
     * Валюта должна быть RUB, BYN или KZT, порог отключения должен быть больше нуля,
     * а клиент не должен быть одним из кривых клиентов
     * (у которых есть более одного нефишечного активного кошелька -- см. DIRECT-87190)
     * <p>
     * В противном случае возвращает false.
     */
    private static boolean isAutoOverdraftApplicable(ClientId clientId,
                                                     CurrencyCode walletCurrency, BigDecimal autoOverdraftLimit) {
        return !CLIENT_IDS_WITH_TWO_ACTIVE_WALLETS.contains(clientId.asLong())
                && AVAILABLE_CURRENCY_CODES.contains(walletCurrency)
                && autoOverdraftLimit.compareTo(BigDecimal.ZERO) > 0;
    }

    /**
     * Возвращает true, если клиент был забанен Балансом
     * или если Баланс сбросил ему overdraft_lim в ноль при неотрицательном auto_overdraft_lim.
     */
    private static boolean isClientBannedInBalance(ClientAutoOverdraftInfo clientInfo) {
        return clientInfo.getStatusBalanceBanned()
                || (clientInfo.getAutoOverdraftLimit().compareTo(BigDecimal.ZERO) > 0
                && clientInfo.getOverdraftLimit().compareTo(BigDecimal.ZERO) == 0);
    }

    /**
     * Вычисляет и возвращает денежную "добавку" по автоовердрафту для кошелька клиента -- те деньги,
     * которые он может потенциально использовать (или уже использовал) по автоовердрафтам.
     * <p>
     * Возвращает 0, если автоовердрафт неприменим к указанному кошельку клиента.
     *
     * @param walletCurrency Валюта кошелька
     * @param walletSum      Сумма зачислений на кошельке
     * @param walletSumSpent Баланс денег под кошельком, меньше или равен нулю
     * @param clientInfo     Инфо о клиенте, относящееся к автоовердрафтам
     */
    public static BigDecimal calculateAutoOverdraftAddition(CurrencyCode walletCurrency,
                                                            BigDecimal walletSum, BigDecimal walletSumSpent, ClientAutoOverdraftInfo clientInfo) {
        // Если к клиенту неприменимы автоовердрафты -- сразу возвращаем 0
        if (!isAutoOverdraftApplicable(
                ClientId.fromLong(clientInfo.getClientId()),
                walletCurrency, clientInfo.getAutoOverdraftLimit())) {
            return BigDecimal.ZERO;
        }

        // Учитываем текущий долг клиента - мы не можем давать ему "двойной" кредит
        BigDecimal availableAutoOverdraftLim =
                max(BigDecimal.ZERO, clientInfo.getAutoOverdraftLimit().subtract(clientInfo.getDebt()));

        // Отдельный кейс для забаненных в Балансе клиентов -- см DIRECT-88599
        if (isClientBannedInBalance(clientInfo)) {
            // Получим сумму открутов для общего счета по всем входящим в него кампаниям, walletSumSpent <= 0
            checkState(walletSumSpent.compareTo(BigDecimal.ZERO) <= 0);
            // Нам интересна сумма зачислений в валюте, поэтому по идее нужно вычесть из sum стоимость потраченных фишек (chips_cost)
            // Но мы этого делать не будем, т.к. они также учтены и в walletSumSpent (стоимость фишек включается в sum_spent)
            // Поэтому если вычитать, то нужно вычитать и из walletSumSpent, а из неравенств ниже следует, что этого можно не делать
            // (слагаемые в любом случае взаимно уничтожатся)
            BigDecimal limitedSum =
                    max(walletSum, min(walletSumSpent.negate(), walletSum.add(availableAutoOverdraftLim)));
            // Так как выполняем max(walletSum, ...), то результат будет не меньше, чем walletSum
            // Поэтому добавка будет неотрицательной
            return limitedSum.subtract(walletSum);
        } else {
            return availableAutoOverdraftLim;
        }
    }
}
