package ru.yandex.direct.core.entity.campaign;

import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.campaign.model.CampaignEmailEvent;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;

import static java.util.function.Predicate.not;
import static ru.yandex.direct.utils.CommonUtils.isValidId;

@ParametersAreNonnullByDefault
public class CampaignNotificationUtils {

    // Типы кампаний, которым недоступно SMS уведомление 'о результатах мониторинга сайтов'
    private static final Set<CampaignType> CAMPAIGN_TYPES_WITHOUT_SITE_MONITORING_SMS_EVENT =
            Set.of(CampaignType.CONTENT_PROMOTION, CampaignType.PERFORMANCE);

    // Типы кампаний, которым недоступно email уведомление 'остановка по достижению ДБ'
    private static final Set<CampaignType> CAMPAIGN_TYPES_WITHOUT_STOP_BY_REACH_DAILY_BUDGET_EMAIL_EVENT =
            Set.of(CampaignType.INTERNAL_FREE, CampaignType.INTERNAL_DISTRIB);

    /**
     * Мапа, в котором каждому из расчитываемых sms-флагов для кампании ставится в соответствие предикат,
     * который должен выполняться для того, чтобы мы добавили sms-флаг в набор доступных
     */
    private static final Map<SmsFlag, Predicate<SmsCheckNode>> SMS_FLAG_PREDICATES =
            ImmutableMap.<SmsFlag, Predicate<SmsCheckNode>>builder()
                    .put(SmsFlag.CAMP_FINISHED_SMS, checkNode -> true)
                    .put(SmsFlag.MODERATE_RESULT_SMS, checkNode -> true)
                    .put(SmsFlag.NOTIFY_METRICA_CONTROL_SMS, SmsCheckNode::isAllowedSiteMonitoringSmsEvent)
                    .put(SmsFlag.ACTIVE_ORDERS_MONEY_OUT_SMS, not(SmsCheckNode::isCampaignUnderWallet))
                    .put(SmsFlag.NOTIFY_ORDER_MONEY_IN_SMS, not(SmsCheckNode::isCampaignUnderWallet))
                    .put(SmsFlag.PAUSED_BY_DAY_BUDGET_SMS, not(SmsCheckNode::isCampaignUnderWallet))
                    .build();

    /**
     * Мапа, в котором каждому из расчитываемых событий для отправки email на кампанию ставится в соответствие предикат,
     * который должен выполняться для того, чтобы мы добавили событие в набор доступных
     */
    private static final Map<CampaignEmailEvent, Predicate<EmailCheckNode>> EMAIL_EVENTS_PREDICATES =
            ImmutableMap.<CampaignEmailEvent, Predicate<EmailCheckNode>>builder()
                    .put(CampaignEmailEvent.WARNING_BALANCE,
                            EmailCheckNode::isAllowedWarningBalanceIntervalEmailEvent)
                    .put(CampaignEmailEvent.CHECK_POSITION,
                            EmailCheckNode::isAllowedCheckPositionIntervalEmailEvent)
                    .put(CampaignEmailEvent.STOP_BY_REACH_DAILY_BUDGET, EmailCheckNode::isAllowedStopByReachDailyBudget)
                    .put(CampaignEmailEvent.SEND_ACCOUNT_NEWS, EmailCheckNode::isServicingCampaign)
                    .put(CampaignEmailEvent.XLS_READY, EmailCheckNode::isAllowedXlsReadyEmailEvent)
                    .build();


    public static Set<SmsFlag> getAvailableSmsFlags(boolean isCampaignUnderWallet, CampaignType campaignType) {
        boolean isCampaignWithSiteMonitoringSmsEvent =
                !CAMPAIGN_TYPES_WITHOUT_SITE_MONITORING_SMS_EVENT.contains(campaignType);
        return getAvailableSmsFlags(isCampaignUnderWallet, isCampaignWithSiteMonitoringSmsEvent);
    }

    public static Set<SmsFlag> getAvailableSmsFlags(boolean isCampaignUnderWallet,
                                                    boolean isCampaignWithSiteMonitoringSmsEvent) {
        var smsCheckNode = new SmsCheckNode(isCampaignUnderWallet, isCampaignWithSiteMonitoringSmsEvent);
        return EntryStream.of(SMS_FLAG_PREDICATES)
                .filterValues(predicate -> predicate.test(smsCheckNode))
                .keys()
                .toImmutableSet();
    }

    public static Set<CampaignEmailEvent> getAvailableEmailEvents(Long walletId, CampaignType campaignType,
                                                                  @Nullable Long managerUserId) {
        return getAvailableEmailEvents(isValidId(walletId), campaignType, isValidId(managerUserId));
    }

    public static Set<CampaignEmailEvent> getAvailableEmailEvents(
            boolean isCampaignUnderWallet,
            CampaignType campaignType,
            boolean isCampaignUnderManager) {
        var emailCheckNode = new EmailCheckNode(isCampaignUnderWallet, campaignType, isCampaignUnderManager);
        return EntryStream.of(EMAIL_EVENTS_PREDICATES)
                .filterValues(predicate -> predicate.test(emailCheckNode))
                .keys()
                .toImmutableSet();
    }

    private static class SmsCheckNode {
        private final boolean isCampaignUnderWallet;
        private final boolean isCampaignWithSiteMonitoringSmsEvent;

        private SmsCheckNode(boolean isCampaignUnderWallet, boolean isCampaignWithSiteMonitoringSmsEvent) {
            this.isCampaignUnderWallet = isCampaignUnderWallet;
            this.isCampaignWithSiteMonitoringSmsEvent = isCampaignWithSiteMonitoringSmsEvent;
        }

        private boolean isCampaignUnderWallet() {
            return isCampaignUnderWallet;
        }

        private boolean isAllowedSiteMonitoringSmsEvent() {
            return isCampaignWithSiteMonitoringSmsEvent;
        }
    }

    private static class EmailCheckNode {

        private static final Set<CampaignType> BLACK_LIST_OF_CAMPAIGN_TYPES_FOR_CHECK_POSITION_INTERVAL_EVENT =
                Set.of(CampaignType.MCBANNER, CampaignType.CPM_BANNER, CampaignType.CPM_DEALS,
                        CampaignType.CPM_YNDX_FRONTPAGE, CampaignType.CONTENT_PROMOTION);

        private final boolean isCampaignUnderWallet;
        private final CampaignType campaignType;
        private final boolean isServicingCampaign;

        private EmailCheckNode(boolean isCampaignUnderWallet, CampaignType campaignType, boolean isServicingCampaign) {
            this.isCampaignUnderWallet = isCampaignUnderWallet;
            this.campaignType = campaignType;
            this.isServicingCampaign = isServicingCampaign;
        }

        private boolean isServicingCampaign() {
            return isServicingCampaign;
        }

        private boolean isAllowedWarningBalanceIntervalEmailEvent() {
            // INTERNAL_AUTOBUDGET могут создаваться в клиентах-продуктах, где при создании был создан кошелёк;
            // но этот тип кампании под кошелёк не подпадает никогда, так что WARNING_BALANCE
            // у него должен быть всегда.
            // Для остальных типов внутренней рекламы WARNING_BALANCE не должно быть, т.к. для них нет денег
            return campaignType == CampaignType.INTERNAL_AUTOBUDGET
                    || (!isCampaignUnderWallet && !CampaignTypeKinds.INTERNAL.contains(campaignType));
        }

        private boolean isAllowedXlsReadyEmailEvent() {
            return campaignType != CampaignType.CONTENT_PROMOTION;
        }

        private boolean isAllowedCheckPositionIntervalEmailEvent() {
            return !BLACK_LIST_OF_CAMPAIGN_TYPES_FOR_CHECK_POSITION_INTERVAL_EVENT.contains(campaignType);
        }

        private static boolean isAllowedStopByReachDailyBudget(EmailCheckNode emailCheckNode) {
            return !CAMPAIGN_TYPES_WITHOUT_STOP_BY_REACH_DAILY_BUDGET_EMAIL_EVENT.contains(emailCheckNode.campaignType);
        }
    }

}
