package ru.yandex.direct.core.entity.campaign.container;

import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.dbutil.model.ClientId;

import static java.util.Collections.singleton;

/**
 * Условие для выборки данных по кампаниям, с возможностью префильтрации.
 * Все заданные условия объединятся через "И".
 * <p>
 * Выборка только по неключевым колонкам не осуществляется (смотри код в репозитории).
 * Если ни одна ключевая колонка не была указана - будет брошено исключение.
 * Если хотя бы одна из колонок определена как пустой список - запрос в репозиторий сделан не будет.
 */
@ParametersAreNonnullByDefault
public class CampaignsSelectionCriteria {

    private Set<Long> campaignIds;
    private Set<Long> orderIds;
    private Set<ClientId> clientIds;
    private Set<CampaignType> campaignTypes;
    private Boolean statusesEmpty;
    private Boolean statusesArchived;
    private Boolean statusesActive;
    private Boolean subCampaigns;

    /**
     * Выборка по идентификаторам кампаний
     */
    public CampaignsSelectionCriteria withCampaignIds(@Nonnull Iterable<Long> campaignIds) {
        this.campaignIds = StreamEx.of(campaignIds.spliterator()).toSet();
        return this;
    }

    /**
     * Выборка по БКшным идентификаторам кампаний
     */
    public CampaignsSelectionCriteria withOrderIds(@Nonnull Iterable<Long> orderIds) {
        this.orderIds = StreamEx.of(orderIds.spliterator()).toSet();
        return this;
    }

    /**
     * Выборка по идентификаторам клиентов
     */
    public CampaignsSelectionCriteria withClientIds(@Nonnull Iterable<ClientId> clientIds) {
        this.clientIds = StreamEx.of(clientIds.spliterator()).toSet();
        return this;
    }

    /**
     * Выборка по идентификатору клиента.
     */
    public CampaignsSelectionCriteria withClientId(@Nonnull ClientId clientId) {
        this.clientIds = singleton(clientId);
        return this;
    }

    /**
     * Префильтрация по типам кампаний
     */
    public CampaignsSelectionCriteria withCampaignTypes(@Nonnull Iterable<CampaignType> campaignTypes) {
        this.campaignTypes = StreamEx.of(campaignTypes.spliterator()).toSet();
        return this;
    }

    /**
     * Префильтрация по признаку "создание кампании не завершено"
     */
    public CampaignsSelectionCriteria withStatusEmpty(@Nonnull Boolean statusEmpty) {
        this.statusesEmpty = statusEmpty;
        return this;
    }

    /**
     * Префильтрация по признаку "кампания в архиве"
     */
    public CampaignsSelectionCriteria withStatusArchived(@Nonnull Boolean statusArchived) {
        this.statusesArchived = statusArchived;
        return this;
    }

    /**
     * Префильтрация по признаку "кампания активна"
     */
    public CampaignsSelectionCriteria withStatusActived(@Nonnull Boolean statusActived) {
        this.statusesActive = statusActived;
        return this;
    }

    /**
     * Префильтрация по признаку "кампания - подлежащая"
     */
    public CampaignsSelectionCriteria withSubCampaigns(@Nonnull Boolean subCampaigns) {
        this.subCampaigns = subCampaigns;
        return this;
    }

    @Nullable
    public Set<Long> getCampaignIds() {
        return campaignIds;
    }

    @Nullable
    public Set<Long> getOrderIds() {
        return orderIds;
    }

    @Nullable
    public Set<ClientId> getClientIds() {
        return clientIds;
    }

    @Nullable
    public Set<CampaignType> getCampaignTypes() {
        return campaignTypes;
    }

    @Nullable
    public Boolean getStatusEmpty() {
        return statusesEmpty;
    }

    @Nullable
    public Boolean getStatusArchived() {
        return statusesArchived;
    }

    @Nullable
    public Boolean getStatusActive() {
        return statusesActive;
    }

    @Nullable
    public Boolean getSubCampaigns() {
        return subCampaigns;
    }
}
