package ru.yandex.direct.core.entity.campaign.container;

import java.util.Collection;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.Multimap;

import ru.yandex.common.util.collections.Pair;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Maps.uniqueIndex;
import static com.google.common.collect.Multimaps.index;

/**
 * Структура данных, представляющая отношения между кампаниями-кошельками и кампаниями,
 * которые к ним привязаны.
 * <p>
 * Замена {@code Multimap<WalletCampaign, Campaign>}, мотивированная некорректностью хранения
 * мутабельных моделей в качестве ключей хеш-таблицы.
 */
@ParametersAreNonnullByDefault
public final class WalletsWithCampaigns {

    private final Map<Long, WalletCampaign> wallets;
    private final Multimap<Long, WalletCampaign> campaignsByWalletId;

    public WalletsWithCampaigns(Collection<WalletCampaign> wallets, Collection<WalletCampaign> campaigns) {
        this.campaignsByWalletId = ImmutableMultimap.copyOf(index(campaigns, WalletCampaign::getWalletId));
        this.wallets = ImmutableMap.copyOf(uniqueIndex(wallets, WalletCampaign::getId));
    }

    /**
     * @param wallet кампания-кошелёк
     * @return Кампании под этим общим счётом.
     */
    public Collection<WalletCampaign> getCampaignsBoundTo(WalletCampaign wallet) {
        return campaignsByWalletId.get(wallet.getId());
    }

    /**
     * @param walletId id кампании-кошелька
     * @return {@code true}, если в структуре содержится кампания кошелёк с таким id, иначе – {@code false}.
     */
    public boolean containsWalletWithId(Long walletId) {
        return wallets.containsKey(walletId);
    }

    /**
     * @param walletId id кампании-кошелька
     * @return Кампания-кошелёк, {@link WalletCampaign}.
     */
    public WalletCampaign getWallet(Long walletId) {
        checkArgument(wallets.containsKey(walletId),
                "Requested wallet (%s) is absent in the index.", walletId);
        return wallets.get(walletId);
    }

    public @Nullable Pair<WalletCampaign, Collection<WalletCampaign>> getPair(@Nullable Long walletId) {
        if (walletId == null || !wallets.containsKey(walletId)) return null;
        var wallet = wallets.get(walletId);
        return Pair.of(wallet, getCampaignsBoundTo(wallet));
    }

    /**
     * @return Все кошельки, содержащиеся в структуре.
     */
    public Collection<WalletCampaign> getAllWallets() {
        return wallets.values();
    }
}
