package ru.yandex.direct.core.entity.campaign.converter;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ser.DefaultSerializerProvider;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.BroadMatch;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignEmailNotification;
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCustomStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoalsWithRequiredFields;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoalStringValue;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.campaign.model.NowOptimizingBy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterPermission;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.vcard.model.ContactInfo;
import ru.yandex.direct.core.entity.vcard.model.InstantMessenger;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.PointPrecision;
import ru.yandex.direct.core.entity.vcard.model.PointType;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsMetatype;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsPerformanceNowOptimizingBy;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.WalletCampaignsIsSumAggregated;
import ru.yandex.direct.jooqmapper.read.FieldValues;
import ru.yandex.direct.jooqmapper.write.PropertyValues;
import ru.yandex.direct.metrika.client.model.response.CounterInfoDirect;
import ru.yandex.direct.metrika.client.model.response.UserCountersExtended;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonMap;
import static java.util.Map.entry;
import static java.util.Objects.isNull;
import static java.util.function.Function.identity;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.common.util.RepositoryUtils.setFromDb;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isUnavailableAutoGoalsAllowedForCampaignWithSource;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isUnavailableAutoGoalsAllowedForCampaignWithStrategy;
import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils.isUnavailableGoalsAllowed;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampOptions.CAMP_OPTIONS;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJsonCollectionEmptyToNull;

@ParametersAreNonnullByDefault
public class CampaignConverter {
    public static final ObjectMapper MAPPER = new ObjectMapper(
            new YAMLFactory()
                    .enable(YAMLGenerator.Feature.MINIMIZE_QUOTES)
                    .disable(YAMLGenerator.Feature.SPLIT_LINES));
    public static final String SPRAV = "sprav";
    public static final String TURBODIRECT = "turbodirect";
    public static final String SYSTEM = "system";
    public static final String PARTNER = "partner";
    public static final String MARKET = "market";
    public static final String EDA = "eda";

    static {
        DefaultSerializerProvider.Impl serializerProvider = new DefaultSerializerProvider.Impl();
        serializerProvider.setNullValueSerializer(new ContactInfoNullSerializer());
        MAPPER.setSerializerProvider(serializerProvider);
    }

    private static final Logger logger = LoggerFactory.getLogger(CampaignConverter.class);

    @SuppressWarnings("java:S3252")
    private static final Set<ModelProperty<? super CommonCampaign, ?>> SUPPORTED_CAMPAIGN_OPTS = Set.of(
            CommonCampaign.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED,
            CommonCampaign.HAS_TITLE_SUBSTITUTION,
            CommonCampaign.HAS_EXTENDED_GEO_TARGETING,
            CommonCampaign.USE_CURRENT_REGION,
            CommonCampaign.USE_REGULAR_REGION,
            CommonCampaign.ENABLE_CPC_HOLD,
            CommonCampaign.ENABLE_COMPANY_INFO,
            CommonCampaign.IS_ALONE_TRAFARET_ALLOWED,
            CommonCampaign.HAS_TURBO_SMARTS,
            CommonCampaign.IS_TOUCH,
            CommonCampaign.HAS_TURBO_APP,
            CommonCampaign.IS_VIRTUAL,
            CommonCampaign.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS,
            CommonCampaign.IS_UNIVERSAL,
            CommonCampaign.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED,
            CommonCampaign.IS_MEANINGFUL_GOALS_VALUES_FROM_METRIKA_ENABLED,
            CommonCampaign.IS_NEW_IOS_VERSION_ENABLED,
            CommonCampaign.IS_SKAD_NETWORK_ENABLED,
            CommonCampaign.IS_ALLOWED_ON_ADULT_CONTENT,
            CommonCampaign.IS_BRAND_LIFT_HIDDEN,
            CommonCampaign.IS_S2S_TRACKING_ENABLED,
            CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED,
            CommonCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED,
            CommonCampaign.IS_WW_MANAGED_ORDER
    );

    @SuppressWarnings("java:S3252")
    private static final Map<ModelProperty<? super CommonCampaign, Boolean>, Boolean> FLAG_INVERSION_BY_PROPERTY =
            Map.ofEntries(
                    entry(CommonCampaign.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED, false),
                    entry(CommonCampaign.HAS_TITLE_SUBSTITUTION, true),
                    entry(CommonCampaign.HAS_EXTENDED_GEO_TARGETING, true),
                    entry(CommonCampaign.USE_CURRENT_REGION, false),
                    entry(CommonCampaign.USE_REGULAR_REGION, false),
                    entry(CommonCampaign.ENABLE_COMPANY_INFO, true),
                    entry(CommonCampaign.ENABLE_CPC_HOLD, false),
                    entry(CommonCampaign.IS_ALONE_TRAFARET_ALLOWED, false),
                    entry(CommonCampaign.HAS_TURBO_SMARTS, false),
                    entry(CommonCampaign.IS_TOUCH, false),
                    entry(CommonCampaign.HAS_TURBO_APP, false),
                    entry(CommonCampaign.IS_VIRTUAL, false),
                    entry(CommonCampaign.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS, false),
                    entry(CommonCampaign.IS_UNIVERSAL, false),
                    entry(CommonCampaign.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED, false),
                    entry(CommonCampaign.IS_MEANINGFUL_GOALS_VALUES_FROM_METRIKA_ENABLED, false),
                    entry(CommonCampaign.IS_NEW_IOS_VERSION_ENABLED, false),
                    entry(CommonCampaign.IS_SKAD_NETWORK_ENABLED, false),
                    entry(CommonCampaign.IS_ALLOWED_ON_ADULT_CONTENT, false),
                    entry(CommonCampaign.IS_BRAND_LIFT_HIDDEN, false),
                    entry(CommonCampaign.IS_S2S_TRACKING_ENABLED, false),
                    entry(CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED, false),
                    entry(CommonCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED, false),
                    entry(CommonCampaign.IS_WW_MANAGED_ORDER, false)
            );

    @SuppressWarnings("java:S3252")
    private static final Map<ModelProperty<? super CommonCampaign, Boolean>, CampaignOpts> OPTS_BY_PROPERTY =
            Map.ofEntries(
                    entry(CommonCampaign.HAS_TITLE_SUBSTITUTION, CampaignOpts.NO_TITLE_SUBSTITUTE),
                    entry(CommonCampaign.HAS_EXTENDED_GEO_TARGETING, CampaignOpts.NO_EXTENDED_GEOTARGETING),
                    entry(CommonCampaign.USE_CURRENT_REGION, CampaignOpts.USE_CURRENT_REGION),
                    entry(CommonCampaign.USE_REGULAR_REGION, CampaignOpts.USE_REGULAR_REGION),
                    entry(CommonCampaign.ENABLE_COMPANY_INFO, CampaignOpts.HIDE_PERMALINK_INFO),
                    entry(CommonCampaign.ENABLE_CPC_HOLD, CampaignOpts.ENABLE_CPC_HOLD),
                    entry(CommonCampaign.IS_ALONE_TRAFARET_ALLOWED, CampaignOpts.IS_ALONE_TRAFARET_ALLOWED),
                    entry(CommonCampaign.HAS_TURBO_SMARTS, CampaignOpts.HAS_TURBO_SMARTS),
                    entry(CommonCampaign.IS_TOUCH, CampaignOpts.IS_TOUCH),
                    entry(CommonCampaign.HAS_TURBO_APP, CampaignOpts.HAS_TURBO_APP),
                    entry(CommonCampaign.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS,
                            CampaignOpts.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS),
                    entry(CommonCampaign.IS_VIRTUAL, CampaignOpts.IS_VIRTUAL),
                    entry(CommonCampaign.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED,
                            CampaignOpts.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED),
                    entry(CommonCampaign.IS_UNIVERSAL, CampaignOpts.IS_UNIVERSAL),
                    entry(CommonCampaign.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED,
                            CampaignOpts.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED),
                    entry(CommonCampaign.IS_MEANINGFUL_GOALS_VALUES_FROM_METRIKA_ENABLED,
                            CampaignOpts.MEANINGFUL_GOALS_VALUES_FROM_METRIKA),
                    entry(CommonCampaign.IS_NEW_IOS_VERSION_ENABLED, CampaignOpts.IS_NEW_IOS_VERSION_ENABLED),
                    entry(CommonCampaign.IS_SKAD_NETWORK_ENABLED, CampaignOpts.IS_SKADNETWORK_ENABLED),
                    entry(CommonCampaign.IS_ALLOWED_ON_ADULT_CONTENT, CampaignOpts.IS_ALLOWED_ON_ADULT_CONTENT),
                    entry(CommonCampaign.IS_BRAND_LIFT_HIDDEN, CampaignOpts.IS_BRAND_LIFT_HIDDEN),
                    entry(CommonCampaign.IS_S2S_TRACKING_ENABLED, CampaignOpts.S2S_TRACKING_ENABLED),
                    entry(CommonCampaign.IS_RECOMMENDATIONS_MANAGEMENT_ENABLED,
                            CampaignOpts.RECOMMENDATIONS_MANAGEMENT_ENABLED),
                    entry(CommonCampaign.IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED,
                            CampaignOpts.PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED),
                    entry(CommonCampaign.IS_WW_MANAGED_ORDER, CampaignOpts.IS_WW_MANAGED_ORDER)
            );

    private CampaignConverter() {
    }

    public static List<MetrikaCounter> toMetrikaCounters(
            List<Long> counters,
            Map<Long, MetrikaCounterWithAdditionalInformation> availableCounters
    ) {
        return mapList(counters, counterId -> toMetrikaCounterContainer(counterId, availableCounters.get(counterId)));
    }

    private static MetrikaCounter toMetrikaCounterContainer(
            Long counterId, @Nullable MetrikaCounterWithAdditionalInformation counterInformation
    ) {
        var counter = Optional.ofNullable(counterInformation)
                .orElseGet(() -> new MetrikaCounterWithAdditionalInformation().withId(counterId));
        return new MetrikaCounter()
                .withId(counter.getId())
                .withHasEcommerce(counter.getHasEcommerce())
                .withSource(counter.getSource());
    }

    public static String campaignOptsToDb(PropertyValues<CommonCampaign> values) {
        return EntryStream.of(FLAG_INVERSION_BY_PROPERTY)
                .filterKeyValue((property, flagIsInvertedForDb) -> extractBooleanFlag(values, property,
                        flagIsInvertedForDb))
                .keys()
                .map(property -> OPTS_BY_PROPERTY.get(property).getTypedValue())
                .joining(",");
    }

    private static Boolean extractBooleanFlag(
            PropertyValues<CommonCampaign> values,
            ModelProperty<? super CommonCampaign, Boolean> property,
            boolean flagIsInvertedForDb) {

        if (flagIsInvertedForDb) {
            return !nvl(values.get(property), true);
        }
        return nvl(values.get(property), false);
    }

    public static String campaignOptsToDb(CommonCampaign campaign) {
        return campaignOptsToDb(new PropertyValues<>(SUPPORTED_CAMPAIGN_OPTS, campaign));
    }

    public static Boolean hasTitleSubstituteFromDb(String opts) {
        return !opts.contains(CampaignOpts.NO_TITLE_SUBSTITUTE.getTypedValue());
    }

    public static Boolean enableCompanyInfoFromDb(String opts) {
        return !opts.contains(CampaignOpts.HIDE_PERMALINK_INFO.getTypedValue());
    }

    public static Boolean hasExtendedGeoTargetingFromDb(String opts) {
        return !opts.contains(CampaignOpts.NO_EXTENDED_GEOTARGETING.getTypedValue());
    }

    public static Boolean useCurrentRegionFromDb(String opts) {
        return opts.contains(CampaignOpts.USE_CURRENT_REGION.getTypedValue());
    }
    public static Boolean useRegularRegionFromDb(String opts) {
        return opts.contains(CampaignOpts.USE_REGULAR_REGION.getTypedValue());
    }

    public static Boolean hasEnableCpcHoldFromDb(String opts) {
        return opts.contains(CampaignOpts.ENABLE_CPC_HOLD.getTypedValue());
    }

    public static Boolean hasTurboAppFromDb(String opts) {
        return opts.contains(CampaignOpts.HAS_TURBO_APP.getTypedValue());
    }

    public static boolean isUniversalFromDb(String opts) {
        return setFromDb(opts, identity()).contains(CampaignOpts.IS_UNIVERSAL.getTypedValue());
    }

    public static Boolean isVirtualFromDb(String opts) {
        return opts.contains(CampaignOpts.IS_VIRTUAL.getTypedValue());
    }

    public static Boolean isAloneTrafaretAllowedFromDb(String opts) {
        return opts.contains(CampaignOpts.IS_ALONE_TRAFARET_ALLOWED.getTypedValue());
    }

    public static Boolean hasTurboSmarts(String opts) {
        return opts.contains(CampaignOpts.HAS_TURBO_SMARTS.getTypedValue());
    }

    public static Boolean isS2sTrackingEnabled(String opts) {
        return opts.contains(CampaignOpts.S2S_TRACKING_ENABLED.getTypedValue());
    }

    public static Boolean isRecommendationsManagementEnabled(String opts) {
        return opts.contains(CampaignOpts.RECOMMENDATIONS_MANAGEMENT_ENABLED.getTypedValue());
    }

    public static Boolean isPriceRecommendationsManagementEnabled(String opts) {
        return opts.contains(CampaignOpts.PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED.getTypedValue());
    }

    public static Boolean isSimplifiedStrategyViewEnabled(String opts) {
        return opts.contains(CampaignOpts.IS_SIMPLIFIED_STRATEGY_VIEW_ENABLED.getTypedValue());
    }

    public static Boolean isOrderPhraseLengthPrecedenceEnabled(String opts) {
        return opts.contains(CampaignOpts.IS_ORDER_PHRASE_LENGTH_PRECEDENCE_ENABLED.getTypedValue());
    }

    public static Boolean isMeaningfulGoalsValuesFromMetrika(String opts) {
        return opts.contains(CampaignOpts.MEANINGFUL_GOALS_VALUES_FROM_METRIKA.getTypedValue());
    }

    public static Boolean isNewIosVersionEnabled(String opts) {
        return opts.contains(CampaignOpts.IS_NEW_IOS_VERSION_ENABLED.getTypedValue());
    }

    public static Boolean isSkadNetworkEnabled(String opts) {
        return opts.contains(CampaignOpts.IS_SKADNETWORK_ENABLED.getTypedValue());
    }

    public static Boolean isAllowedOnAdultContentFromDb(String opts) {
        return opts.contains(CampaignOpts.IS_ALLOWED_ON_ADULT_CONTENT.getTypedValue());
    }

    public static Boolean isBrandLiftHiddenFromDb(String opts) {
        return opts.contains(CampaignOpts.IS_BRAND_LIFT_HIDDEN.getTypedValue());
    }

    public static Boolean isRequireFiltrationByDontShowDomains(String opts) {
        return opts.contains(CampaignOpts.REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS.getTypedValue());
    }

    public static Boolean isWwManagedOrder(String opts) {
        return opts.contains(CampaignOpts.IS_WW_MANAGED_ORDER.getTypedValue());
    }

    public static Boolean isTouchFromDb(String opts) {
        return opts.contains(CampaignOpts.IS_TOUCH.getTypedValue());
    }

    public static String emailNotificationsToDb(PropertyValues<CommonCampaign> values) {
        List<String> notifications = new ArrayList<>();
        if (values.get(CommonCampaign.ENABLE_PAUSED_BY_DAY_BUDGET_EVENT)) {
            notifications.add(CampaignEmailNotification.PAUSED_BY_DAY_BUDGET.getTypedValue());
        }
        return String.join(",", notifications);
    }

    public static boolean hasPausedByDayBudgetFromDb(@Nullable String emailNotifications) {
        return emailNotifications != null
                && emailNotifications.contains(CampaignEmailNotification.PAUSED_BY_DAY_BUDGET.getTypedValue());
    }

    public static String meaningfulGoalsToDb(List<MeaningfulGoal> meaningfulGoals,
                                             boolean isMeaningfulGoalValueSerializeAsString) {
        if (isMeaningfulGoalValueSerializeAsString) {
            List<MeaningfulGoalStringValue> convertedGoals = mapList(meaningfulGoals,
                    goal -> new MeaningfulGoalStringValue()
                            .withGoalId(goal.getGoalId())
                            .withConversionValue(goal.getConversionValue())
                            .withIsMetrikaSourceOfValue(goal.getIsMetrikaSourceOfValue())
            );
            return toJsonCollectionEmptyToNull(convertedGoals);
        }
        return toJsonCollectionEmptyToNull(meaningfulGoals);
    }

    public static List<MeaningfulGoal> meaningfulGoalsFromDb(@Nullable String meaningfulGoal) {
        return ifNotNull(meaningfulGoal, goal -> List.of(fromJson(goal, MeaningfulGoal[].class)));
    }

    public static <M extends Model, V> AppliedChanges<M> modifyFieldIfNull(
            AppliedChanges<M> changes, ModelProperty<? super M, V> property, V value) {
        if (property.get(changes.getModel()) == null) {
            changes.modify(property, value);
        }
        return changes;
    }

    public static <M extends Model, V> AppliedChanges<M> setOldValueIfNewNull(
            AppliedChanges<M> changes, ModelProperty<? super M, V> property) {
        if (property.get(changes.getModel()) == null) {
            V oldValue = Objects.requireNonNull(changes.getOldValue(property));
            changes.modify(property, oldValue);
        }
        return changes;
    }

    public static <M extends Model, V> M setFieldIfNull(
            M campaign, ModelProperty<M, V> property, V value) {
        if (property.get(campaign) == null) {
            property.set(campaign, value);
        }
        return campaign;
    }

    public static DbStrategy strategyFromDb(FieldValues fields) {
        DbStrategy dbStrategy = new DbStrategy();
        dbStrategy.withAutobudget(CampaignsAutobudget.fromSource(fields.getValue(CAMPAIGNS.AUTOBUDGET)))
                .withStrategyName(StrategyName.fromSource(fields.getValue(CAMPAIGNS.STRATEGY_NAME)))
                .withPlatform(CampaignsPlatform.fromSource(fields.getValue(CAMPAIGNS.PLATFORM)))
                .withStrategyData(CampaignMappings.strategyDataFromDb(fields.getValue(CAMPAIGNS.STRATEGY_DATA)))
                .withStrategy(CampOptionsStrategy.fromSource(fields.getValue(CAMP_OPTIONS.STRATEGY)));
        return dbStrategy;
    }

    public static BroadMatch broadMatchFromDb(FieldValues fields) {
        return new BroadMatch()
                .withBroadMatchFlag(CampaignMappings.broadMatchFlagFromDb(fields.getValue(Tables.CAMP_OPTIONS.BROAD_MATCH_FLAG)))
                .withBroadMatchLimit(RepositoryUtils.intFromLong(fields.getValue(Tables.CAMP_OPTIONS.BROAD_MATCH_LIMIT)))
                .withBroadMatchGoalId(fields.getValue(Tables.CAMP_OPTIONS.BROAD_MATCH_GOAL_ID));
    }

    public static boolean campaignStatusShowFromDb(CampaignsStatusshow status) {
        return CampaignsStatusshow.Yes == status;
    }

    public static CampaignsStatusshow campaignStatusShowToDb(boolean status) {
        return status ? CampaignsStatusshow.Yes : CampaignsStatusshow.No;
    }

    public static boolean campaignStatusActiveFromDb(CampaignsStatusactive status) {
        return CampaignsStatusactive.Yes == status;
    }

    public static CampaignsStatusactive campaignStatusActiveToDb(boolean status) {
        return status ? CampaignsStatusactive.Yes : CampaignsStatusactive.No;
    }

    public static boolean campaignStatusEmptyFromDb(CampaignsStatusempty status) {
        return CampaignsStatusempty.Yes == status;
    }

    public static CampaignsStatusempty campaignStatusEmptyToDb(boolean status) {
        return status ? CampaignsStatusempty.Yes : CampaignsStatusempty.No;
    }

    public static CampaignsSource campaignSourceOrDefaultToDb(CampaignSource value) {
        var res = CampaignSource.toSource(value);
        return res == null ? CampaignsSource.direct : res;
    }

    public static CampaignsMetatype campaignMetatypeOrDefaultToDb(CampaignMetatype value) {
        var res = CampaignMetatype.toSource(value);
        return res == null ? CampaignsMetatype.default_ : res;
    }

    @Nullable
    private static Vcard toVcard(@Nullable ContactInfo contactInfo) {
        if (contactInfo == null) {
            return null;
        }
        PointOnMap manualPoint = toPointOnMap(contactInfo.getManualBounds(), contactInfo.getManualPoint());
        PointOnMap autoPoint = toPointOnMap(contactInfo.getAutoBounds(), contactInfo.getAutoPoint());

        PointPrecision precision = ifNotNull(contactInfo.getPrecision(), p -> PointPrecision.valueOf(p.toUpperCase()));

        String ogrn = contactInfo.getOgrn() == null ? contactInfo.getOgrnNum() : contactInfo.getOgrn();

        return new Vcard()
                .withCompanyName(contactInfo.getCompanyName())
                .withWorkTime(contactInfo.getWorkTime())
                .withContactPerson(contactInfo.getContactPerson())
                .withPhone(toPhone(contactInfo))
                .withEmail(contactInfo.getEmail())
                .withInstantMessenger(toInstantMessenger(contactInfo))
                .withExtraMessage(contactInfo.getExtraMessage())
                .withOrgDetailsId(contactInfo.getOrgDetailsId())
                .withOgrn(ogrn)
                .withGeoId(contactInfo.getGeoId())
                .withCountry(contactInfo.getCountry())
                .withCity(contactInfo.getCity())
                .withStreet(contactInfo.getStreet())
                .withHouse(contactInfo.getHouse())
                .withBuild(contactInfo.getBuild())
                .withApart(contactInfo.getApart())
                .withMetroId(contactInfo.getMetroId())
                .withManualPoint(manualPoint)
                .withAutoPoint(autoPoint)
                .withPointType(ifNotNull(contactInfo.getPointType(), PointType::valueOf))
                .withPrecision(precision);
    }

    @Nullable
    private static PointOnMap toPointOnMap(String bounds, String point) {
        if (isBlank(bounds) || isBlank(point)) {
            return null;
        }
        List<BigDecimal> boundCoords = toCoords(bounds);
        checkState(boundCoords.size() == 4, "Bounds should have 4 coords");
        List<BigDecimal> pointCoords = toCoords(point);
        checkState(pointCoords.size() == 2, "Points should have 2 coords");

        return new PointOnMap()
                .withX(pointCoords.get(0))
                .withY(pointCoords.get(1))
                .withX1(boundCoords.get(0))
                .withY1(boundCoords.get(1))
                .withX2(boundCoords.get(2))
                .withY2(boundCoords.get(3));
    }

    private static List<BigDecimal> toCoords(String points) {
        return StreamEx.split(points, ",")
                .map(BigDecimal::new)
                .toList();
    }

    @Nullable
    private static InstantMessenger toInstantMessenger(ContactInfo contactInfo) {
        if (contactInfo.getInstantMessengerType() == null && contactInfo.getInstantMessengerLogin() == null) {
            return null;
        }
        return new InstantMessenger()
                .withType(contactInfo.getInstantMessengerType())
                .withLogin(contactInfo.getInstantMessengerLogin());
    }

    private static Phone toPhone(ContactInfo contactInfo) {
        return new Phone()
                .withPhoneNumber(contactInfo.getPhone())
                .withCityCode(contactInfo.getCityCode())
                .withCountryCode(contactInfo.getCountryCode())
                .withExtension(contactInfo.getExt());
    }

    @Nullable
    public static Vcard vcardFromDb(String contactInfo) {
        try {
            return toVcard(contactInfoFromDb(contactInfo));
        } catch (IllegalArgumentException e) {
            logger.warn("invalid contact info: " + contactInfo, e);
            return null;
        }
    }

    @Nullable
    private static ContactInfo contactInfoFromDb(String contactInfo) {
        return !isBlank(contactInfo) ? deSerialiseContactInfoFromDb(contactInfo) :
                null;
    }

    private static ContactInfo deSerialiseContactInfoFromDb(String contactInfo) {
        try {
            return MAPPER.readValue(contactInfo, ContactInfo.class);
        } catch (IOException e) {
            throw new IllegalArgumentException("can not deserialize object from YAML", e);
        }
    }

    public static String vcardToDb(Vcard vcard) {
        ContactInfo contactInfo = toContactInfo(vcard);
        return contactInfo != null ? serialiseContactInfo(contactInfo) : "";
    }

    private static String serialiseContactInfo(ContactInfo contactInfo) {
        try {
            return MAPPER.writeValueAsString(contactInfo);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to yaml", e);
        }
    }

    @Nullable
    private static ContactInfo toContactInfo(@Nullable Vcard vcard) {
        if (vcard == null) {
            return null;
        }
        InstantMessenger instantMessenger = vcard.getInstantMessenger();
        Phone phone = vcard.getPhone();
        return new ContactInfo()
                .withCompanyName(vcard.getCompanyName())
                .withWorkTime(vcard.getWorkTime())
                .withContactPerson(vcard.getContactPerson())
                .withCityCode(phone.getCityCode())
                .withCountryCode(phone.getCountryCode())
                .withPhone(phone.getPhoneNumber())
                .withExt(phone.getExtension())
                .withEmail(vcard.getEmail())
                .withInstantMessengerType(ifNotNull(instantMessenger, InstantMessenger::getType))
                .withInstantMessengerLogin(ifNotNull(instantMessenger, InstantMessenger::getLogin))
                .withExtraMessage(vcard.getExtraMessage())
                .withOgrn(vcard.getOgrn())
                .withOrgDetailsId(vcard.getOrgDetailsId())
                .withGeoId(vcard.getGeoId())
                .withCountry(vcard.getCountry())
                .withCity(vcard.getCity())
                .withStreet(vcard.getStreet())
                .withHouse(vcard.getHouse())
                .withBuild(vcard.getBuild())
                .withApart(vcard.getApart())
                .withMetroId(vcard.getMetroId())
                .withAutoBounds(toBounds(vcard.getAutoPoint()))
                .withAutoPoint(toPoint(vcard.getAutoPoint()))
                .withManualBounds(toBounds(vcard.getManualPoint()))
                .withManualPoint(toPoint(vcard.getManualPoint()))
                .withPointType(ifNotNull(vcard.getPointType(), Enum::name))
                .withPrecision(toContactInfoPrecision(vcard.getPrecision()));
    }

    private static String toContactInfoPrecision(@Nullable PointPrecision precision) {
        if (precision == null) {
            return null;
        }
        return precision.toString().toLowerCase();
    }

    private static String toBounds(@Nullable PointOnMap pointOnMap) {
        if (pointOnMap == null) {
            return null;
        }
        return pointOnMap.getX1() + "," +
                pointOnMap.getY1() + "," +
                pointOnMap.getX2() + "," +
                pointOnMap.getY2();
    }

    private static String toPoint(@Nullable PointOnMap pointOnMap) {
        if (pointOnMap == null) {
            return null;
        }
        return pointOnMap.getX() + "," +
                pointOnMap.getY();
    }

    public static boolean isSumAggregatedFromDb(WalletCampaignsIsSumAggregated mode) {
        checkArgument(WalletCampaignsIsSumAggregated.Migrating != mode, "invalid status");
        return WalletCampaignsIsSumAggregated.Yes == mode;
    }

    public static WalletCampaignsIsSumAggregated isSumAggregatedToDb(boolean isSumAggregated) {
        return isSumAggregated ? WalletCampaignsIsSumAggregated.Yes : WalletCampaignsIsSumAggregated.No;
    }

    public static Set<MetrikaCounterWithAdditionalInformation> toMetrikaCountersWithAdditionalInformation(
            List<UserCountersExtended> countersExtended) {
        return toMetrikaCountersWithAdditionalInformation(countersExtended,
                CampMetrikaCountersService.CounterWithAdditionalInformationFilter.defaultFilter());
    }

    public static Set<MetrikaCounterWithAdditionalInformation> toMetrikaCountersWithAdditionalInformation(
            List<UserCountersExtended> countersExtended,
            CampMetrikaCountersService.CounterWithAdditionalInformationFilter filter) {
        return new HashSet<>(StreamEx.of(countersExtended)
                .mapToEntry(UserCountersExtended::getOwner, UserCountersExtended::getCounters)
                .flatMapToValue((uid, counters) ->
                        counters.stream().map(c -> toMetrikaCounterWithAdditionalInformation(uid, c)))
                .values()
                .filter(filter::isSuitable)
                .toMap(MetrikaCounterWithAdditionalInformation::getId,
                        Function.identity(), CampaignConverter::mergePermissions)
                .values());
    }

    private static MetrikaCounterWithAdditionalInformation toMetrikaCounterWithAdditionalInformation(
            Long uid, CounterInfoDirect counter) {
        MetrikaCounterSource source = null;
        MetrikaCounterPermission permission = null;
        try {
            source = toMetrikaCounterSource(counter.getCounterSource());
        } catch (IllegalArgumentException e) {
            logger.warn("Unsupported type of metrika counter for id: " + counter.getId(), e);
        }
        try {
            permission = toMetrikaCounterPermission(counter.getCounterPermission());
        } catch (IllegalArgumentException e) {
            logger.warn("Unsupported type of counterPermission (" + counter.getCounterPermission()
                    + ") for counterId: " + counter.getId(), e);
        }
        return new MetrikaCounterWithAdditionalInformation()
                .withId((long) counter.getId())
                .withDomain(counter.getSitePath())
                .withName(counter.getName())
                .withSource(source)
                .withHasEcommerce(counter.getEcommerce())
                .withPermissionsByUid(singletonMap(uid, permission));
    }

    private static MetrikaCounterWithAdditionalInformation mergePermissions(
            MetrikaCounterWithAdditionalInformation metrikaCounterFirst,
            MetrikaCounterWithAdditionalInformation metrikaCounterSecond) {
        checkState(metrikaCounterFirst.getId().equals(metrikaCounterSecond.getId()), "Different metrika counters");
        Map<Long, MetrikaCounterPermission> permissionsByUid = new HashMap<>(metrikaCounterFirst.getPermissionsByUid());
        permissionsByUid.putAll(metrikaCounterSecond.getPermissionsByUid());
        return new MetrikaCounterWithAdditionalInformation()
                .withId(metrikaCounterFirst.getId())
                .withDomain(metrikaCounterFirst.getDomain())
                .withName(metrikaCounterFirst.getName())
                .withSource(metrikaCounterFirst.getSource())
                .withHasEcommerce(metrikaCounterFirst.getHasEcommerce())
                .withPermissionsByUid(permissionsByUid);
    }

    public static MetrikaCounterSource toMetrikaCounterSource(String metrikaCounterSource) {
        if (isEmpty(metrikaCounterSource)) {
            return MetrikaCounterSource.UNKNOWN;
        }

        switch (metrikaCounterSource) {
            case SPRAV:
                return MetrikaCounterSource.SPRAV;
            case TURBODIRECT:
                return MetrikaCounterSource.TURBO;
            case SYSTEM:
                return MetrikaCounterSource.SYSTEM;
            case PARTNER:
                return MetrikaCounterSource.PARTNER;
            case MARKET:
                return MetrikaCounterSource.MARKET;
            case EDA:
                return MetrikaCounterSource.EDA;
            default:
                logger.warn("No such value: {}", metrikaCounterSource);
                return MetrikaCounterSource.UNKNOWN;
        }
    }

    public static MetrikaCounterPermission toMetrikaCounterPermission(@Nullable String metrikaCounterPermission) {
        if (isNull(metrikaCounterPermission)) {
            return null;
        }
        switch (metrikaCounterPermission) {
            case "own":
                return MetrikaCounterPermission.OWN;
            case "edit":
                return MetrikaCounterPermission.EDIT;
            case "view":
                return MetrikaCounterPermission.VIEW;
            default:
                throw new IllegalStateException("No such value: " + metrikaCounterPermission);
        }
    }

    public static String fromMetrikaCounterSource(@Nullable MetrikaCounterSource metrikaCounterSource) {
        if (metrikaCounterSource == null) {
            return null;
        }
        switch (metrikaCounterSource) {
            case SPRAV:
                return SPRAV;
            case TURBO:
                return TURBODIRECT;
            case SYSTEM:
                return SYSTEM;
            case PARTNER:
                return PARTNER;
            case MARKET:
                return MARKET;
            case EDA:
                return EDA;
            case UNKNOWN:
            default:
                throw new IllegalStateException("No such value: " + metrikaCounterSource);
        }
    }

    public static NowOptimizingBy optimizingByFromDbFormat(CampaignsPerformanceNowOptimizingBy value) {
        return NowOptimizingBy.fromSource(value);
    }

    public static CampaignsPerformanceNowOptimizingBy optimizingByToDbFormat(NowOptimizingBy value) {
        return nvl(NowOptimizingBy.toSource(value), CampaignsPerformanceNowOptimizingBy.CPC);
    }

    public static List<Long> extractCounterIdsFromCampaigns(Collection<? extends BaseCampaign> campaigns) {
        return StreamEx.of(campaigns)
                .flatCollection(CampaignConverter::extractCounterIds)
                .distinct()
                .toList();
    }

    public static List<Long> extractCounterIds(BaseCampaign campaign) {
        if (campaign instanceof CampaignWithMetrikaCounters) {
            var counters = ((CampaignWithMetrikaCounters) campaign).getMetrikaCounters();
            return counters == null ? List.of() : counters;
        }

        return List.of();
    }

    public static CampaignTypeWithCounterIds toCampaignTypeWithCounterIds(
            CampaignWithCustomStrategy campaign,
            Set<String> enabledFeatures) {
        return toCampaignTypeWithCounterIds(campaign, enabledFeatures, Map.of());
    }

    public static CampaignTypeWithCounterIds toCampaignTypeWithCounterIds(
            CampaignWithCustomStrategy campaign,
            Set<String> enabledFeatures,
            Map<Long, Long> goalIdToCounterId) {
        var counterIds = listToSet(CampaignConverter.extractCounterIds(campaign));
        if (counterIds.size() == 0) {
            counterIds = Optional.ofNullable(getCampaignGoalId(campaign))
                    .map(goalIdToCounterId::get)
                    .map(Set::of)
                    .orElse(emptySet());
        }
        return new CampaignTypeWithCounterIds()
                .withCampaignType(campaign.getType())
                .withCounterIds(counterIds)
                .withUnavailableAutoGoalsAllowed(
                        isUnavailableAutoGoalsAllowedForCampaignWithStrategy(campaign, enabledFeatures)
                )
                .withUnavailableGoalsAllowed(
                        isUnavailableGoalsAllowed(campaign, enabledFeatures)
                );
    }

    public static CampaignTypeWithCounterIds toCampaignTypeWithCounterIds(
            CampaignWithMeaningfulGoalsWithRequiredFields campaign,
            Set<Long> counterIds,
            Set<String> enabledFeatures) {
        return new CampaignTypeWithCounterIds()
                .withCampaignType(campaign.getType())
                .withCounterIds(counterIds)
                .withUnavailableAutoGoalsAllowed(
                        isUnavailableAutoGoalsAllowedForCampaignWithSource(campaign, enabledFeatures)
                )
                .withUnavailableGoalsAllowed(
                        isUnavailableGoalsAllowed(campaign, enabledFeatures)
                );
    }

    public static CampaignTypeWithCounterIds toCampaignTypeWithCounterIdsFromCampaignWithStrategy(
            CampaignWithStrategy campaign,
            Set<String> enabledFeatures) {
        return new CampaignTypeWithCounterIds()
                .withCampaignType(campaign.getType())
                .withCounterIds(new HashSet<>(extractCounterIds(campaign)))
                .withUnavailableAutoGoalsAllowed(
                        isUnavailableAutoGoalsAllowedForCampaignWithSource(campaign, enabledFeatures)
                )
                .withUnavailableGoalsAllowed(
                        isUnavailableGoalsAllowed(campaign, enabledFeatures)
                );
    }

    public static Long getCampaignGoalId(CampaignWithCustomStrategy campaign) {
        return getStrategyGoalId(campaign.getStrategy());
    }

    public static Long getStrategyGoalId(@Nullable DbStrategy strategy) {
        return Optional.ofNullable(strategy)
                .map(DbStrategy::getStrategyData)
                .map(StrategyData::getGoalId)
                .orElse(null);
    }

}
