package ru.yandex.direct.core.entity.campaign.converter;

import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.core.entity.campaign.model.MobileContentInfo;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTrackerTrackingSystem;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContentAvatarSize;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.jooqmapper.read.FieldValues;

import static ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService.generateUrlString;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_APPS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_APP_TRACKERS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_CONTENT;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class MobileContentInfoConverter {

    public static final String AVATARS_MDS_HOST = "avatars.mds.yandex.net";

    public static final Map<OsType, String> MIN_OS_VERSION = ImmutableMap.of(
            OsType.ANDROID, "1.0",
            OsType.IOS, "1.0");
    // Для старых РМП кампаний, когда у кампании нет связи с приложением и основные ссылки на стор идут на группу
    private static final OsType DEFAULT_OS_TYPE = OsType.IOS;

    private MobileContentInfoConverter() {
    }

    public static MobileContentInfo fromDbFormat(FieldValues fields) {
        OsType osType = nvl(OsType.fromSource(fields.getValue(MOBILE_CONTENT.OS_TYPE)), DEFAULT_OS_TYPE);
        String minimalOsVersionFromStore = fields.getValue(MOBILE_CONTENT.MIN_OS_VERSION);
        String currentMinimalOsVersion = fields.getValue(MOBILE_APPS.MIN_OS_VERSION);
        String iconHash = fields.getValue(MOBILE_CONTENT.ICON_HASH);
        MobileAppTrackerTrackingSystem trackingSystem =
                MobileAppTrackerTrackingSystem.fromSource(fields.getValue(MOBILE_APP_TRACKERS.TRACKING_SYSTEM));
        return new MobileContentInfo()
                // У старых РМП кампаний нет связей с campaigns_mobile_content, mobile_apps и mobile_content.
                // В старом интерфейсе для них открывалось окно редактирования кампании, чтобы выбрать
                // приложение и тем самым добавить данные связи. Для этого кейса отправляем на фронт пустые значения
                .withOsType(osType)
                .withStoreContentId(nvl(fields.getValue(MOBILE_CONTENT.STORE_CONTENT_ID), ""))
                .withStoreCountry(nvl(fields.getValue(MOBILE_CONTENT.STORE_COUNTRY), ""))
                .withStoreHref(nvl(fields.getValue(MOBILE_APPS.STORE_HREF), ""))
                .withMinimalOsVersionFromStore(getMinimalOsVersion(minimalOsVersionFromStore, osType))
                .withCurrentMinimalOsVersion(getMinimalOsVersion(currentMinimalOsVersion, osType))
                .withIconHash(iconHash)
                .withIconUrl(generateIconUrl(iconHash, osType))
                .withMobileContentName(nvl(fields.getValue(MOBILE_CONTENT.NAME), ""))
                .withTracker(ifNotNull(trackingSystem, unused -> new MobileAppTracker()
                        .withTrackingSystem(trackingSystem)
                        .withTrackerId(fields.getValue(MOBILE_APP_TRACKERS.TRACKER_ID))
                        .withUrl(fields.getValue(MOBILE_APP_TRACKERS.URL))
                        .withImpressionUrl(fields.getValue(MOBILE_APP_TRACKERS.IMPRESSION_URL))));
    }

    private static String getMinimalOsVersion(@Nullable String minimalOsVersion, OsType osType) {
        return nvl(minimalOsVersion, MIN_OS_VERSION.get(nvl(osType, DEFAULT_OS_TYPE)));
    }

    public static String generateIconUrl(@Nullable String iconHash, OsType osType) {
        if (iconHash == null) {
            return null;
        }
        // хост mds пока захардкожен продовый, иначе картинки не видны в тестовых средах
        return generateUrlString(AVATARS_MDS_HOST, osType, iconHash, MobileContentAvatarSize.ICON);
    }
}
