package ru.yandex.direct.core.entity.campaign.model

import org.jooq.Condition
import org.jooq.impl.DSL
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrency
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrencyconverted
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow
import ru.yandex.direct.dbutil.SqlUtils

/**
 * Состояние показов кампании (для API Campaigns)
 *
 * [Документация](https://yandex.ru/dev/direct/doc/dg/objects/campaign.html#status)
 */
enum class ApiCampaignState {
    ARCHIVED,
    CONVERTED,
    ENDED,
    OFF,
    ON,
    SUSPENDED,
}

fun ApiCampaignState.toCondition(): Condition =
    when (this) {
        ApiCampaignState.CONVERTED ->
            CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.Yes)
                .and(CAMPAIGNS.CURRENCY_CONVERTED.eq(CampaignsCurrencyconverted.Yes))
                .and(CAMPAIGNS.CURRENCY.eq(CampaignsCurrency.YND_FIXED))

        ApiCampaignState.ARCHIVED ->
            CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.Yes)
                .and(notConverted)

        ApiCampaignState.ENDED ->
            notArchivedOrConverted
                .and(CAMPAIGNS.FINISH_TIME.isNotNull)
                .and(CAMPAIGNS.FINISH_TIME.ne(SqlUtils.mysqlZeroLocalDate()))
                .and(CAMPAIGNS.FINISH_TIME.lt(DSL.currentLocalDate()))

        ApiCampaignState.SUSPENDED ->
            notArchivedOrConverted
                .and(notEnded)
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.No))

        ApiCampaignState.OFF ->
            notArchivedOrConverted
                .and(notEnded)
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.No))

        ApiCampaignState.ON ->
            CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No) // здесь не нужна проверка на то, конвертированная ли кампания
                .and(notEnded)
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.Yes))
    }

private val notConverted: Condition =
    CAMPAIGNS.CURRENCY_CONVERTED.eq(CampaignsCurrencyconverted.No)
        .or(CAMPAIGNS.CURRENCY.ne(CampaignsCurrency.YND_FIXED))

private val notArchivedOrConverted: Condition =
    CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No)
        .and(notConverted)

private val notEnded: Condition =
    CAMPAIGNS.FINISH_TIME.isNull
        .or(CAMPAIGNS.FINISH_TIME.eq(SqlUtils.mysqlZeroLocalDate()))
        .or(CAMPAIGNS.FINISH_TIME.ge(DSL.currentLocalDate()))
