package ru.yandex.direct.core.entity.campaign.repository;

import java.time.LocalDateTime;
import java.util.Map;
import java.util.Set;

import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.InsertValuesStepN;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.tables.records.CampAggregatedLastchangeRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.tables.CampAggregatedLastchange.CAMP_AGGREGATED_LASTCHANGE;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.mysqlIf;

@Repository
public class CampAggregatedLastchangeRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    public CampAggregatedLastchangeRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public void updateCampAggregatedLastchange(int shard, Map<Long, LocalDateTime> lastChangesByCids) {
        InsertValuesStepN<CampAggregatedLastchangeRecord> insertQuery =
                dslContextProvider.ppc(shard).insertInto(CAMP_AGGREGATED_LASTCHANGE,
                        asList(CAMP_AGGREGATED_LASTCHANGE.CID, CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE));
        for (Map.Entry<Long, LocalDateTime> lastChangeByCid : lastChangesByCids.entrySet()) {
            insertQuery.values(lastChangeByCid.getKey(), lastChangeByCid.getValue());
        }
        Condition duplicateUpdateCondition =
                CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE
                        .lessThan(MySQLDSL.values(CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE));
        Field<LocalDateTime> field =
                mysqlIf(duplicateUpdateCondition, MySQLDSL.values(CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE),
                        CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE);
        insertQuery.onDuplicateKeyUpdate().set(CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE, field).execute();
    }

    /**
     * Выбирать из списка кампаний такие, в которых были CHILDREN-изменения с момента указанного времени.
     *
     * @param shard        номер шарда
     * @param campaignIds  список ID кампаний, которые нужно проверить
     * @param fromDateTime время, с момента которого нужно проверить, были ли изменения
     * @return
     */
    public Set<Long> getChangedCampaignIds(int shard, Set<Long> campaignIds, LocalDateTime fromDateTime) {
        return dslContextProvider.ppc(shard)
                .select(Tables.CAMP_AGGREGATED_LASTCHANGE.CID)
                .from(Tables.CAMP_AGGREGATED_LASTCHANGE)
                .where(Tables.CAMP_AGGREGATED_LASTCHANGE.CID.in(campaignIds))
                .and(Tables.CAMP_AGGREGATED_LASTCHANGE.LAST_CHANGE.ge(fromDateTime))
                .fetchSet(Tables.CAMP_AGGREGATED_LASTCHANGE.CID);
    }
}
