package ru.yandex.direct.core.entity.campaign.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.InsertValuesStepN;
import org.jooq.Record2;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounter;
import ru.yandex.direct.core.entity.campaign.model.MetrikaCounterSource;
import ru.yandex.direct.core.entity.strategy.container.StrategyRepositoryContainer;
import ru.yandex.direct.core.entity.strategy.model.StrategyWithMetrikaCounters;
import ru.yandex.direct.core.entity.strategy.type.withmetrikacounters.StrategyWithMetrikaCountersRepositoryTypeSupport;
import ru.yandex.direct.dbschema.ppc.enums.MetrikaCountersSource;
import ru.yandex.direct.dbschema.ppc.tables.records.CampMetrikaCountersRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.MetrikaCountersRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static org.jooq.impl.DSL.falseCondition;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.Tables.METRIKA_COUNTERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class CampMetrikaCountersRepository {
    private final DslContextProvider dslContextProvider;
    private final StrategyWithMetrikaCountersRepositoryTypeSupport strategyWithMetrikaCountersRepositoryTypeSupport;
    private static final JooqMapperWithSupplier<MetrikaCounter> METRIKA_COUNTERS_MAPPER =
            createCampaignMetrikaCounterMapper();


    @Autowired
    public CampMetrikaCountersRepository(DslContextProvider dslContextProvider,
                                         StrategyWithMetrikaCountersRepositoryTypeSupport strategyWithMetrikaCountersRepositoryTypeSupport) {
        this.dslContextProvider = dslContextProvider;
        this.strategyWithMetrikaCountersRepositoryTypeSupport = strategyWithMetrikaCountersRepositoryTypeSupport;
    }

    /**
     * Получение счетчиков метрики по списку id-ков кампаний
     * в полученной Map'е для cid без счётчиков значениями будут пустые списки
     */
    public Map<Long, List<Long>> getMetrikaCountersByCids(int shard, Collection<Long> cids) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS)
                .from(CAMPAIGNS)
                .leftJoin(CAMP_METRIKA_COUNTERS).on(CAMPAIGNS.CID.eq(CAMP_METRIKA_COUNTERS.CID))
                .where(CAMPAIGNS.CID.in(cids))
                .fetchMap(CAMPAIGNS.CID, r -> Optional.ofNullable(r.get(CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS))
                        .map(CampaignMappings::metrikaCountersFromDb)
                        .orElse(new ArrayList<>()));
    }

    /**
     * Получение счетчиков метрики по списку id клиента
     * в полученной Map'е для cid без счётчиков значениями будут пустые списки
     */
    public Map<Long, List<Long>> getMetrikaCountersByClientId(int shard, long clientId) {
        SelectConditionStep<Record2<Long, String>> conditionStep = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS)
                .from(CAMPAIGNS)
                .leftJoin(CAMP_METRIKA_COUNTERS).on(CAMPAIGNS.CID.eq(CAMP_METRIKA_COUNTERS.CID))
                .where(CAMPAIGNS.CLIENT_ID.in(clientId));
        return conditionStep.fetchMap(CAMPAIGNS.CID,
                r -> Optional.ofNullable(r.get(CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS))
                        .map(CampaignMappings::metrikaCountersFromDb)
                        .orElse(new ArrayList<>()));
    }

    public void updateMetrikaCountersToCampaignsAndStrategies(
            int shard,
            Map<Long, List<MetrikaCounter>> countersByCid,
            StrategyRepositoryContainer containerWithUpdatedCountersForStrategies,
            Collection<AppliedChanges<StrategyWithMetrikaCounters>> strategiesChangesCollection) {
        updateMetrikaCountersForStrategy(containerWithUpdatedCountersForStrategies, strategiesChangesCollection);
        updateMetrikaCounters(shard, countersByCid);
    }

    public void updateMetrikaCountersForStrategy(
            StrategyRepositoryContainer containerWithUpdatedCountersForStrategies,
            Collection<AppliedChanges<StrategyWithMetrikaCounters>> strategiesChangesCollection
    ) {
        updateMetrikaCountersForStrategy(dslContextProvider.ppc(containerWithUpdatedCountersForStrategies.getShard()),
                containerWithUpdatedCountersForStrategies,
                strategiesChangesCollection);
    }

    public void updateMetrikaCountersForStrategy(
            DSLContext dslContext,
            StrategyRepositoryContainer containerWithUpdatedCountersForStrategies,
            Collection<AppliedChanges<StrategyWithMetrikaCounters>> strategiesChangesCollection
    ) {
        strategyWithMetrikaCountersRepositoryTypeSupport.updateAdditionTables(dslContext,
                containerWithUpdatedCountersForStrategies,
                strategiesChangesCollection);
    }

    public void updateMetrikaCounters(int shard, Map<Long, List<MetrikaCounter>> countersByCid) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        updateMetrikaCounters(dslContext, countersByCid);
    }

    public void updateMetrikaCounters(DSLContext dslContext, Map<Long, List<MetrikaCounter>> countersByCid) {
        updateTableCampMetrikaCounters(dslContext, countersByCid);
        updateTableMetrikaCounters(dslContext, countersByCid);
    }

    private void updateTableCampMetrikaCounters(DSLContext dslContext, Map<Long, List<MetrikaCounter>> countersByCid) {
        InsertValuesStepN<CampMetrikaCountersRecord> insertIntoCampMetrikaCounters = dslContext
                .insertInto(CAMP_METRIKA_COUNTERS,
                        asList(CAMP_METRIKA_COUNTERS.CID, CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS));
        for (Long cid : countersByCid.keySet()) {
            List<Long> counterIds = mapList(countersByCid.get(cid), MetrikaCounter::getId);
            insertIntoCampMetrikaCounters = insertIntoCampMetrikaCounters.values(cid,
                    CampaignMappings.metrikaCounterIdsToDb(counterIds));
        }
        insertIntoCampMetrikaCounters.onDuplicateKeyUpdate()
                .set(CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS, MySQLDSL.values(CAMP_METRIKA_COUNTERS.METRIKA_COUNTERS))
                .execute();
    }


    public void updateTableMetrikaCounters(DSLContext dslContext, Map<Long, List<MetrikaCounter>> countersByCid) {
        //удаление лишних счетчиков
        Condition deleteCondition = falseCondition();
        for (Long cid : countersByCid.keySet()) {
            Condition condition = METRIKA_COUNTERS.CID.eq(cid).and(
                    METRIKA_COUNTERS.METRIKA_COUNTER
                            .notIn(mapList(countersByCid.get(cid), MetrikaCounter::getId)));
            deleteCondition = deleteCondition.or(condition);
        }
        dslContext.deleteFrom(METRIKA_COUNTERS).where(METRIKA_COUNTERS.CID.in(countersByCid.keySet()))
                .and(deleteCondition).execute();

        //добавление
        InsertValuesStepN<MetrikaCountersRecord> insertIntoMetrikaCounters = dslContext
                .insertInto(METRIKA_COUNTERS,
                        asList(METRIKA_COUNTERS.CID, METRIKA_COUNTERS.METRIKA_COUNTER, METRIKA_COUNTERS.HAS_ECOMMERCE,
                                METRIKA_COUNTERS.SOURCE));
        for (Long cid : countersByCid.keySet()) {
            for (MetrikaCounter metrikaCounter : countersByCid.get(cid)) {
                MetrikaCountersSource source = ifNotNull(metrikaCounter.getSource(), MetrikaCounterSource::toSource);
                insertIntoMetrikaCounters = insertIntoMetrikaCounters.values(
                        cid, metrikaCounter.getId(), metrikaCounter.getHasEcommerce(),
                        nvl(source, MetrikaCountersSource.unknown));
            }
        }
        insertIntoMetrikaCounters.onDuplicateKeyUpdate()
                .set(METRIKA_COUNTERS.HAS_ECOMMERCE, MySQLDSL.values(METRIKA_COUNTERS.HAS_ECOMMERCE))
                .set(METRIKA_COUNTERS.SOURCE, MySQLDSL.values(METRIKA_COUNTERS.SOURCE))
                .execute();
    }


    public void deleteMetrikaCounters(int shard, Collection<Long> cids) {
        DSLContext dslContext = dslContextProvider.ppc(shard);

        deleteMetrikaCounters(dslContext, cids);
    }

    public void deleteMetrikaCounters(DSLContext dslContext, Collection<Long> cids) {
        dslContext.deleteFrom(CAMP_METRIKA_COUNTERS).where(CAMP_METRIKA_COUNTERS.CID.in(cids)).execute();
        dslContext.deleteFrom(METRIKA_COUNTERS).where(METRIKA_COUNTERS.CID.in(cids)).execute();
    }

    public Map<Long, List<MetrikaCounter>> getMetrikaCounterByCid(
            DSLContext dslContext, Collection<Long> campaignIds) {
        Set<Field<?>> fieldsToRead = StreamEx.of(METRIKA_COUNTERS_MAPPER.getFieldsToRead())
                .append(METRIKA_COUNTERS.CID)
                .toSet();

        return dslContext
                .select(fieldsToRead)
                .from(METRIKA_COUNTERS)
                .where(METRIKA_COUNTERS.CID.in(campaignIds))
                .fetchGroups(METRIKA_COUNTERS.CID, METRIKA_COUNTERS_MAPPER::fromDb);
    }

    /**
     * Получить список идентификаторов счетчиков заданного типа {@code source}.
     * <p>
     * - если задано {@code counterIds}, то поиск будет производиться только среди этих счетчиков
     * <p>
     * Поля {@code clientId} и {@code campaignIds} -- взаимоисключающие, хотя бы одно обязательно:
     * <p>
     * - если задано {@code clientId}, то будут отбираться счетчики по всем кампаниям, принадлежащим этому клиенту
     * <p>
     * - если задано {@code campaignIds}, то будут отбираться счетчики, принадлежащие этим кампаниям
     */
    public Set<Long> getActiveCounterIds(
            int shard,
            @Nullable ClientId clientId,
            @Nullable Collection<Long> campaignIds,
            @Nullable Collection<Long> counterIds,
            MetrikaCounterSource source
    ) {
        checkArgument(clientId != null || campaignIds != null, "At least clientId or campaignIds should be specified");
        DSLContext dslContext = dslContextProvider.ppc(shard);
        Condition isFromCampaign;
        if (campaignIds == null) {
            // Если кампании не заданы, используем все кампании клиента
            var allCampSelect =
                    dslContext.select(CAMPAIGNS.CID)
                            .from(CAMPAIGNS)
                            .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()));
            isFromCampaign = METRIKA_COUNTERS.CID.in(allCampSelect);
        } else {
            isFromCampaign = METRIKA_COUNTERS.CID.in(campaignIds);
        }
        var isSourceCounter = METRIKA_COUNTERS.SOURCE.eq(MetrikaCounterSource.toSource(source));
        var isActive = METRIKA_COUNTERS.IS_DELETED.eq(RepositoryUtils.FALSE);
        var isCounter = counterIds == null ? DSL.trueCondition() : METRIKA_COUNTERS.METRIKA_COUNTER.in(counterIds);
        return dslContext
                .select(METRIKA_COUNTERS.METRIKA_COUNTER)
                .from(METRIKA_COUNTERS)
                .where(isFromCampaign.and(isCounter).and(isSourceCounter).and(isActive))
                .fetchSet(METRIKA_COUNTERS.METRIKA_COUNTER);
    }

    public Set<Long> getActiveCounterIds(
            int shard,
            @Nullable ClientId clientId,
            @Nullable Collection<Long> campaignIds,
            MetrikaCounterSource source
    ) {
        return getActiveCounterIds(shard, clientId, campaignIds, null, source);
    }

    public Map<Long, List<MetrikaCounter>> getMetrikaCounterByCid(
            int shard, Collection<Long> campaignIds) {
        return getMetrikaCounterByCid(dslContextProvider.ppc(shard), campaignIds);
    }

    private static JooqMapperWithSupplier<MetrikaCounter> createCampaignMetrikaCounterMapper() {
        return JooqMapperWithSupplierBuilder.builder(MetrikaCounter::new)
                .map(property(MetrikaCounter.ID,
                        METRIKA_COUNTERS.METRIKA_COUNTER))
                .map(convertibleProperty(MetrikaCounter.SOURCE,
                        METRIKA_COUNTERS.SOURCE, MetrikaCounterSource::fromSource, MetrikaCounterSource::toSource))
                .map(booleanProperty(MetrikaCounter.HAS_ECOMMERCE,
                        METRIKA_COUNTERS.HAS_ECOMMERCE))
                .map(booleanProperty(MetrikaCounter.IS_DELETED,
                        METRIKA_COUNTERS.IS_DELETED))
                .build();
    }
}
