package ru.yandex.direct.core.entity.campaign.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.CampaignForAccessCheck;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignForAccessCheckRepositoryAdapter;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.utils.FunctionalUtils;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.DYNAMIC_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.VCARDS;

@Repository
@ParametersAreNonnullByDefault
public class CampaignAccessCheckRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    public CampaignAccessCheckRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получает CampaignForAccessCheck для указанных кампаний (с проверкой доступа клиента к кампаниям).
     *
     * @param shard                                   шард
     * @param campaignForAccessCheckRepositoryAdapter фильтр для выбоки кампаний из таблицы
     * @param campaignIds                             id кампаний
     * @return отображение campaignId -> CampaignForAccessCheck
     */
    public <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsForAccessCheckByCampaignIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        var step = dslContextProvider.ppc(shard)
                .select(campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields())
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds));

        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());

        List<T> campaigns = step
                .fetchStream()
                .map(campaignForAccessCheckRepositoryAdapter.recordMapper()::map)
                .collect(Collectors.toList());

        return FunctionalUtils.listToMap(campaigns, CampaignForAccessCheck::getId);
    }

    /**
     * @return отображение adGroupId -> CampaignForAccessCheck
     */
    public  <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsForAccessCheckByAdGroupIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter campaignForAccessCheckRepositoryAdapter,
            Collection<Long> adGroupIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(PHRASES.PID);
        var step = dslContextProvider.ppc(shard)
                .select(fields)
                .from(CAMPAIGNS)
                .join(PHRASES).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(PHRASES.PID.in(adGroupIds));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(PHRASES.PID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    /**
     * @return отображение bannerId -> CampaignForAccessCheck
     */
    public  <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsForAccessCheckByBannerIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> bannerIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(BANNERS.BID);
        var step = dslContextProvider.ppc(shard)
                .select(fields)
                .from(CAMPAIGNS)
                .join(BANNERS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.BID.in(bannerIds));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(BANNERS.BID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    /**
     * @return отображение dynCondIds -> CampaignForAccessCheck, берутся dynCondIds из групп с типом "по домену" и
     * "по фиду"
     */
    public  <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsByDynCondIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> dynCondIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(DYNAMIC_CONDITIONS.DYN_COND_ID);
        var step = dslContextProvider
                .ppc(shard)
                .select(fields)
                .from(DYNAMIC_CONDITIONS)
                .join(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
                .join(PHRASES).on(DYNAMIC_CONDITIONS.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynCondIds));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(DYNAMIC_CONDITIONS.DYN_COND_ID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    /**
     * @return отображение bidId -> CampaignForAccessCheck
     */
    public  <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsWithTypeByBidIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> bidIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(BIDS.ID);
        var step = dslContextProvider.ppc(shard)
                .select(fields)
                .from(CAMPAIGNS)
                .join(BIDS).on(CAMPAIGNS.CID.eq(BIDS.CID))
                .where(BIDS.ID.in(bidIds));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(BIDS.ID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    public  <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsWithTypeByVcardIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> vcardIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(VCARDS.VCARD_ID);
        var step = dslContextProvider.ppc(shard)
                .select(fields)
                .from(CAMPAIGNS)
                .join(VCARDS).on(CAMPAIGNS.CID.eq(VCARDS.CID))
                .where(VCARDS.VCARD_ID.in(vcardIds));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(VCARDS.VCARD_ID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    public  <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsWithTypeByRelevanceMatchIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> relevanceMatchIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(BIDS_BASE.BID_ID);
        var step = dslContextProvider.ppc(shard)
                .select(fields)
                .from(CAMPAIGNS)
                .join(BIDS_BASE).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
                .where(BIDS_BASE.BID_ID.in(relevanceMatchIds)
                        .and(BIDS_BASE.BID_TYPE.in(BidsBaseBidType.relevance_match,
                                BidsBaseBidType.relevance_match_search)));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(BIDS_BASE.BID_ID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    public <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsWithTypeByOfferRetargetingIds(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> offerRetargetingIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(BIDS_BASE.BID_ID);
        var step = dslContextProvider.ppc(shard)
                .select(fields)
                .from(CAMPAIGNS)
                .join(BIDS_BASE).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
                .where(BIDS_BASE.BID_ID.in(offerRetargetingIds)
                        .and(BIDS_BASE.BID_TYPE.in(BidsBaseBidType.offer_retargeting)));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(BIDS_BASE.BID_ID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    /**
     * @return отображение dynCondIds -> CampaignForAccessCheck, берутся только dynCondIds из групп с типом "по домену"
     */
    public <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsByDynCondIdsFromDomainTypeAdgroup(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> dynCondIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(DYNAMIC_CONDITIONS.DYN_COND_ID);
        var step = dslContextProvider
                .ppc(shard)
                .select(fields)
                .from(DYNAMIC_CONDITIONS)
                // join с BIDS_DYNAMIC нужен т.к. при удалении условия оно удаляется только из BIDS_DYNAMIC, но не из
                // DYNAMIC_CONDITIONS, а получать удаленные условия из этого метода мы не хотим
                .join(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
                .join(PHRASES).on(DYNAMIC_CONDITIONS.PID.eq(PHRASES.PID))
                .join(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynCondIds)
                        .and(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID.isNotNull()));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(DYNAMIC_CONDITIONS.DYN_COND_ID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }

    /**
     * @return отображение dynCondIds -> CampaignForAccessCheck, берутся dynCondIds из групп с типом "по фиду"
     */
    public <T extends CampaignForAccessCheck> Map<Long, T> getCampaignsByDynCondIdsFromFeedTypeAdgroup(
            int shard,
            CampaignForAccessCheckRepositoryAdapter<T> campaignForAccessCheckRepositoryAdapter,
            Collection<Long> dynCondIds) {
        Collection<Field<?>> fields = new ArrayList<>(
                campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFields());
        fields.add(DYNAMIC_CONDITIONS.DYN_COND_ID);
        var step = dslContextProvider
                .ppc(shard)
                .select(fields)
                .from(DYNAMIC_CONDITIONS)
                .join(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
                .join(PHRASES).on(DYNAMIC_CONDITIONS.PID.eq(PHRASES.PID))
                .join(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynCondIds)
                        .and(ADGROUPS_DYNAMIC.FEED_ID.isNotNull()));
        campaignForAccessCheckRepositoryAdapter.campaignForAccessCheckFilter().apply(step.getQuery());
        return step.fetchMap(DYNAMIC_CONDITIONS.DYN_COND_ID, campaignForAccessCheckRepositoryAdapter.recordMapper());
    }
}
