package ru.yandex.direct.core.entity.campaign.repository;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.converter.BrandSurveyStopReasonsConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignBudgetReach;
import ru.yandex.direct.core.entity.campaign.model.CampaignBudgetReachDaily;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Arrays.asList;
import static ru.yandex.direct.core.entity.campaign.converter.BrandSurveyStopReasonsConverter.fromString;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_BUDGET_REACH_DAILY;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class CampaignBudgetReachDailyRepository {
    public static final String SQL_DELETE_OLD_RECORDS_QUERY = "delete c \n" +
            "from campaigns_budget_reach_daily as c \n" +
            "left join (select cid, max(date) as date from campaigns_budget_reach_daily group by cid) as tmp \n" +
            "on (tmp.cid = c.cid and tmp.date = c.date) \n" +
            "where tmp.cid is null and c.date < {0};";

    private final JooqMapperWithSupplier<CampaignBudgetReachDaily> jooqMapper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public CampaignBudgetReachDailyRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        jooqMapper = JooqMapperWithSupplierBuilder.builder(CampaignBudgetReachDaily::new)
                .map(property(CampaignBudgetReachDaily.CAMPAIGN_ID, CAMPAIGNS_BUDGET_REACH_DAILY.CID))
                .map(property(CampaignBudgetReachDaily.DATE, CAMPAIGNS_BUDGET_REACH_DAILY.DATE))
                .map(property(CampaignBudgetReachDaily.BUDGET_SPENT, CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_SPENT))
                .map(property(CampaignBudgetReachDaily.BUDGET_ESTIMATED, CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_ESTIMATED))
                .map(property(CampaignBudgetReachDaily.BUDGET_THRESHOLD, CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_THRESHOLD))
                .map(property(CampaignBudgetReachDaily.TARGET_FORECAST, CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_FORECAST))
                .map(property(CampaignBudgetReachDaily.TARGET_THRESHOLD, CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_THRESHOLD))
                .map(property(CampaignBudgetReachDaily.TRAFFIC_LIGHT_COLOUR, CAMPAIGNS_BUDGET_REACH_DAILY.TRAFFIC_LIGHT_COLOUR))
                .map(property(CampaignBudgetReachDaily.BRAND_SURVEY_ID, CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_SURVEY_ID))
                .map(convertibleProperty(
                        CampaignBudgetReachDaily.BRAND_SURVEY_STOP_REASONS,
                        CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_LIFT_STOP_REASONS,
                        BrandSurveyStopReasonsConverter::fromString,
                        BrandSurveyStopReasonsConverter::toString))
                .build();
    }

    public void addCampaignBudgetReaches(int shard, List<CampaignBudgetReachDaily> campaignBudgets) {
        if (campaignBudgets.isEmpty()) {
            return;
        }

        var insertHelper = new InsertHelper<>(dslContextProvider.ppc(shard), CAMPAIGNS_BUDGET_REACH_DAILY);
        insertHelper.addAll(jooqMapper, campaignBudgets);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_SPENT, MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_SPENT))
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_ESTIMATED,
                            MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_ESTIMATED))
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_THRESHOLD,
                            MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_THRESHOLD))
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_FORECAST,
                            MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_FORECAST))
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_THRESHOLD,
                            MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_THRESHOLD))
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.TRAFFIC_LIGHT_COLOUR,
                            MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.TRAFFIC_LIGHT_COLOUR))
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_LIFT_STOP_REASONS,
                            MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_LIFT_STOP_REASONS))
                    .set(CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_SURVEY_ID,
                            MySQLDSL.values(CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_SURVEY_ID));
        }

        insertHelper.executeIfRecordsAdded();
    }

    public Map<Long, CampaignBudgetReach> getCampaignBudgetReachForCampaigns(int shard, List<Long> cids) {
        var campaignsBudgetReach2 = CAMPAIGNS_BUDGET_REACH_DAILY.as("c2");
        return dslContextProvider.ppc(shard)
                .select(asList(
                        CAMPAIGNS_BUDGET_REACH_DAILY.CID,
                        CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_SPENT,
                        CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_ESTIMATED,
                        CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_THRESHOLD,
                        CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_FORECAST,
                        CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_THRESHOLD,
                        CAMPAIGNS_BUDGET_REACH_DAILY.TRAFFIC_LIGHT_COLOUR,
                        CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_LIFT_STOP_REASONS))
                .from(CAMPAIGNS_BUDGET_REACH_DAILY)
                .leftJoin(campaignsBudgetReach2).on(
                        campaignsBudgetReach2.CID.eq(CAMPAIGNS_BUDGET_REACH_DAILY.CID)
                                .and(campaignsBudgetReach2.DATE.greaterThan(CAMPAIGNS_BUDGET_REACH_DAILY.DATE)))
                .where(campaignsBudgetReach2.DATE.isNull().and(CAMPAIGNS_BUDGET_REACH_DAILY.CID.in(cids)))
                .fetchMap(CAMPAIGNS_BUDGET_REACH_DAILY.CID,
                        record -> new CampaignBudgetReach()
                        .withCampaignId(record.getValue(CAMPAIGNS_BUDGET_REACH_DAILY.CID))
                        .withBudgetSpent(record.getValue(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_SPENT))
                        .withBudgetEstimated(record.getValue(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_ESTIMATED))
                        .withBudgetThreshold(record.getValue(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_THRESHOLD))
                        .withTargetForecast(record.getValue(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_FORECAST))
                        .withTargetThreshold(record.getValue(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_THRESHOLD))
                        .withTrafficLightColour(record.getValue(CAMPAIGNS_BUDGET_REACH_DAILY.TRAFFIC_LIGHT_COLOUR))
                        .withBrandSurveyStopReasons(fromString(record.get(CAMPAIGNS_BUDGET_REACH_DAILY.BRAND_LIFT_STOP_REASONS))));
    }

    public void cleanOldRecords(int shard, LocalDate thresholdDate) {
        dslContextProvider.ppc(shard).execute(SQL_DELETE_OLD_RECORDS_QUERY, thresholdDate);
    }

    public Map<Long, Integer> getBrandLiftWorkDuration(int shard, List<Long> cids) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS_BUDGET_REACH_DAILY.CID)
                .select(DSL.sum(
                            DSL.iif(CAMPAIGNS_BUDGET_REACH_DAILY.BUDGET_SPENT.gt(BigDecimal.ZERO)
                                    .and(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_FORECAST.gt(CAMPAIGNS_BUDGET_REACH_DAILY.TARGET_THRESHOLD)),
                                    1,0))
                        .as("days_sum")
                )
                .from(CAMPAIGNS_BUDGET_REACH_DAILY)
                .where(CAMPAIGNS_BUDGET_REACH_DAILY.CID.in(cids))
                .groupBy(CAMPAIGNS_BUDGET_REACH_DAILY.CID)
                .fetchMap(CAMPAIGNS_BUDGET_REACH_DAILY.CID,
                        record -> record.get("days_sum", Integer.class)

                );
    }
}
