package ru.yandex.direct.core.entity.campaign.repository

import org.jooq.Record1
import org.jooq.SelectConditionStep
import org.jooq.impl.DSL
import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.prefilters.CampaignPrefilter.Companion.COPYABLE_SOURCES
import ru.yandex.direct.core.copyentity.prefilters.CampaignPrefilter.Companion.COPYABLE_TYPES
import ru.yandex.direct.core.entity.campaign.container.CampaignInfoForCopy
import ru.yandex.direct.core.entity.campaign.container.CampaignsCopySelectionCriteria
import ru.yandex.direct.core.entity.campaign.container.CampaignsCopySelectionCriteria.SelectionType
import ru.yandex.direct.core.entity.campaign.model.CampaignSource
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS
import ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.Tables.PERF_CREATIVES
import ru.yandex.direct.dbschema.ppc.Tables.PHRASES
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsStatuspostmoderate
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesStatusmoderate
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

/**
 * Репозиторий с методами, специфичными для копирования кампаний
 */
@Component
class CampaignCopyRepository(
    private val dslContextProvider: DslContextProvider,
) {

    fun getCampaignsForCopy(shard: Int, selectionCriteria: CampaignsCopySelectionCriteria): Set<Long> {
        val select: SelectConditionStep<Record1<Long>> = dslContextProvider.ppc(shard)
            .select(CAMPAIGNS.CID)
            .from(CAMPAIGNS)
            .join(Tables.CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(Tables.CAMP_OPTIONS.CID))
            .where(CAMPAIGNS.TYPE.`in`(COPYABLE_TYPES.map { CampaignType.toSource(it) }))
            .and(CAMPAIGNS.SOURCE.`in`(COPYABLE_SOURCES.map { CampaignSource.toSource(it) }))
            .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
            .and(CAMPAIGNS.CLIENT_ID.eq(selectionCriteria.clientId.asLong()))

        if (selectionCriteria.campaignIds != null) {
            select.and(CAMPAIGNS.CID.`in`(selectionCriteria.campaignIds))
        } else {
            when (selectionCriteria.selectionType!!) {
                SelectionType.ACTIVE -> select
                    .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                    .and(CAMPAIGNS.STATUS_MODERATE.ne(CampaignsStatusmoderate.New))
                    .and(Tables.CAMP_OPTIONS.STATUS_POST_MODERATE.eq(CampOptionsStatuspostmoderate.Accepted))
                SelectionType.STOPPED -> select
                    .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                    .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.No))
                SelectionType.DRAFT -> select
                    .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                    .and(CAMPAIGNS.STATUS_MODERATE.eq(CampaignsStatusmoderate.New))
                SelectionType.ARCHIVED -> select
                    .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.Yes))
                SelectionType.ALL -> {}
            }
        }

        return select
            .fetchSet(CAMPAIGNS.CID)
    }

    fun getCampaignInfoForCopy(shard: Int, campaignIds: Collection<Long>): Map<Long, CampaignInfoForCopy> {
        val hasFeedDynamicAdGroupsField =
            DSL.field(
                DSL.exists(
                    DSL.selectOne()
                        .from(PHRASES)
                        .join(ADGROUPS_DYNAMIC).on(PHRASES.PID.eq(ADGROUPS_DYNAMIC.PID))
                        .where(PHRASES.CID.eq(CAMPAIGNS.CID))
                        .and(PHRASES.ADGROUP_TYPE.eq(PhrasesAdgroupType.dynamic))
                        .and(ADGROUPS_DYNAMIC.FEED_ID.isNotNull)
                        .limit(1)
                )
            ).`as`("hasFeedDynamicAdGroups")

        val hasRejectedCreatives =
            DSL.field(
                DSL.exists(
                    DSL.selectOne()
                        .from(BANNERS_PERFORMANCE)
                        .join(PERF_CREATIVES).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                        .where(BANNERS_PERFORMANCE.CID.eq(CAMPAIGNS.CID))
                        .and(PERF_CREATIVES.STATUS_MODERATE.eq(PerfCreativesStatusmoderate.AdminReject))
                        .limit(1)
                )
            ).`as`("hasRejectedCreatives")

        val hasCanvasCreatives =
            DSL.field(
                DSL.exists(
                    DSL.selectOne()
                        .from(BANNERS_PERFORMANCE)
                        .join(PERF_CREATIVES).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                        .where(BANNERS_PERFORMANCE.CID.eq(CAMPAIGNS.CID))
                        .and(
                            PERF_CREATIVES.CREATIVE_TYPE.`in`(
                                PerfCreativesCreativeType.canvas,
                                PerfCreativesCreativeType.html5_creative,
                            )
                        )
                        .limit(1)
                )
            ).`as`("hasCanvasCreatives")

        val hasCpcVideoBanners =
            DSL.field(
                DSL.exists(
                    DSL.selectOne()
                        .from(BANNERS)
                        .where(BANNERS.CID.eq(CAMPAIGNS.CID))
                        .and(BANNERS.BANNER_TYPE.eq(BannersBannerType.cpc_video))
                        .limit(1)
                )
            ).`as`("hasCpcVideoBanners")

        return dslContextProvider.ppc(shard)
            .select(
                CAMPAIGNS.CID,
                hasFeedDynamicAdGroupsField,
                hasRejectedCreatives,
                hasCanvasCreatives,
                hasCpcVideoBanners,
            )
            .from(CAMPAIGNS)
            .where(CAMPAIGNS.CID.`in`(campaignIds))
            .fetchMap(CAMPAIGNS.CID) { record ->
                CampaignInfoForCopy(
                    hasFeedDynamicAdGroups = record[hasFeedDynamicAdGroupsField],
                    hasRejectedCreatives = record[hasRejectedCreatives],
                    hasCanvasCreatives = record[hasCanvasCreatives],
                    hasCpcVideoBanners = record[hasCpcVideoBanners],
                )
            }
    }
}
