package ru.yandex.direct.core.entity.campaign.repository;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Table;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.BillingAggregateCampaign;
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign;
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory;
import ru.yandex.direct.core.entity.campaign.repository.type.CampaignRepositoryTypeSupportFacade;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.core.entity.vcard.repository.internal.OrgDetailsRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.repository.TypeModifyRepository;
import ru.yandex.direct.multitype.repository.filter.Filter;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;

@Repository
@ParametersAreNonnullByDefault
public class CampaignModifyRepository extends TypeModifyRepository<BaseCampaign, BaseCampaign,
        RestrictedCampaignsAddOperationContainer, RestrictedCampaignsUpdateOperationContainer> {
    private final DslContextProvider ppcDslContextProvider;

    private final ShardHelper shardHelper;
    private final ShardSupport shardSupport;
    private final OrgDetailsRepository orgDetailsRepository;

    @Autowired
    public CampaignModifyRepository(DslContextProvider ppcDslContextProvider,
                                    CampaignRepositoryTypeSupportFacade typeSupportFacade,
                                    ShardHelper shardHelper, ShardSupport shardSupport,
                                    OrgDetailsRepository orgDetailsRepository,
                                    CampaignTypedRepository typedRepository) {
        super(ppcDslContextProvider, typeSupportFacade, typedRepository);
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardHelper = shardHelper;
        this.shardSupport = shardSupport;
        this.orgDetailsRepository = orgDetailsRepository;
    }

    @Override
    protected void generateIds(RestrictedCampaignsAddOperationContainer addContainer, List<? extends BaseCampaign> models) {
        Iterator<Long> campaignIds = shardHelper.generateCampaignIds(addContainer.getClientId().asLong(),
                models.size()).iterator();
        models.forEach(model -> model.setId(campaignIds.next()));
    }

    @Override
    protected Table<?> getLockTable() {
        return CAMPAIGNS;
    }

    @Override
    protected Filter getIdFilter(Collection<Long> modelIds) {
        return CampaignFilterFactory.campaignIdsFilter(modelIds);
    }

    public List<Long> addCampaigns(DSLContext context,
                                   RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                   List<? extends BaseCampaign> campaigns) {
        return add(context, addCampaignParametersContainer, campaigns);
    }

    public void updateCampaigns(DSLContext dslContext, RestrictedCampaignsUpdateOperationContainer updateParameters,
                                Collection<? extends AppliedChanges<? extends BaseCampaign>> appliedChanges) {
        update(dslContext, updateParameters, appliedChanges);
    }

    public void updateCampaigns(RestrictedCampaignsUpdateOperationContainer updateParameters,
                                Collection<? extends AppliedChanges<? extends BaseCampaign>> appliedChanges) {
        update(updateParameters, appliedChanges);
    }

    /**
     * Обновление только таблиц campaigns и camp_opts.
     * Для обновления кампаний со всеми связанными сущностями использовать {@link this#updateCampaigns}
     */
    public void updateCampaignsTable(int shard,
                                     Collection<? extends AppliedChanges<? extends BaseCampaign>> appliedChanges) {
        updateCampaignsTable(ppcDslContextProvider.ppc(shard), appliedChanges);
    }

    public void updateCampaignsTable(DSLContext dsl,
                                     Collection<? extends AppliedChanges<? extends BaseCampaign>> appliedChanges) {
        updateTable(dsl, appliedChanges);
    }

    /**
     * Удаляем неиспользуемые org_details
     */
    public void cleanOrgDetails(int shard, Long clientUid) {
        DSLContext dslContext = ppcDslContextProvider.ppc(shard);

        List<Long> emptyClientOrgDetails = orgDetailsRepository.getUnusedClientOrgDetails(dslContext, clientUid);
        orgDetailsRepository.deleteClientOrgDetails(dslContext, emptyClientOrgDetails);
        shardSupport.deleteValues(ShardKey.ORG_DETAILS_ID, emptyClientOrgDetails);
    }

    public Long addWallet(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer, WalletTypedCampaign wallet) {
        DSLContext dslContext = ppcDslContextProvider.ppc(addCampaignParametersContainer.getShard());
        List<WalletTypedCampaign> campaigns = List.of(wallet);
        addCampaigns(dslContext, addCampaignParametersContainer, campaigns);

        return wallet.getId();
    }

    public List<Long> addBillingAggregate(RestrictedCampaignsAddOperationContainer addCampaignParametersContainer,
                                          BillingAggregateCampaign billingAggregate) {
        DSLContext dslContext = ppcDslContextProvider.ppc(addCampaignParametersContainer.getShard());
        List<BillingAggregateCampaign> campaigns = List.of(billingAggregate);
        return addCampaigns(dslContext, addCampaignParametersContainer, campaigns);
    }

}
