package ru.yandex.direct.core.entity.campaign.repository;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.DatePart;
import org.jooq.Field;
import org.jooq.InsertValuesStep2;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Record2;
import org.jooq.Record3;
import org.jooq.Record4;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.SelectHavingStep;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.StatusAutobudgetShow;
import ru.yandex.direct.core.entity.campaign.AvailableCampaignSources;
import ru.yandex.direct.core.entity.campaign.container.CampaignWithAutobudget;
import ru.yandex.direct.core.entity.campaign.container.CampaignWithBidsDomainsAndPhones;
import ru.yandex.direct.core.entity.campaign.container.CampaignsSelectionCriteria;
import ru.yandex.direct.core.entity.campaign.container.WalletsWithCampaigns;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.converter.PriceFlightConverter;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsDayBudgetNotificationStatus;
import ru.yandex.direct.core.entity.campaign.model.CampOptionsStrategy;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignCounts;
import ru.yandex.direct.core.entity.campaign.model.CampaignDayBudgetOptions;
import ru.yandex.direct.core.entity.campaign.model.CampaignEmailNotification;
import ru.yandex.direct.core.entity.campaign.model.CampaignForBlockedMoneyCheck;
import ru.yandex.direct.core.entity.campaign.model.CampaignForForecast;
import ru.yandex.direct.core.entity.campaign.model.CampaignForLauncher;
import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyFinishedByDate;
import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyOrder;
import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyUrlMonitoring;
import ru.yandex.direct.core.entity.campaign.model.CampaignMetatype;
import ru.yandex.direct.core.entity.campaign.model.CampaignName;
import ru.yandex.direct.core.entity.campaign.model.CampaignOpts;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStub;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTypeAndWalletId;
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget;
import ru.yandex.direct.core.entity.campaign.model.CampaignsDayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.model.ContentLanguage;
import ru.yandex.direct.core.entity.campaign.model.CpmYndxFrontpageShowTypeUtils;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetNotificationStatus;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightReasonIncorrect;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusCorrect;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.model.StatusAutobudgetForecast;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.model.WalletCampaign;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.minuskeywordspack.MinusKeywordsPackUtils;
import ru.yandex.direct.core.entity.time.model.TimeInterval;
import ru.yandex.direct.core.entity.vcard.repository.VcardMappings;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.dbschema.ppc.Indexes;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersPhoneflag;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsBroadMatchFlag;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsStatusmetricacontrol;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCpmPriceStatusApprove;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrency;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrencyconverted;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsSource;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusbssynced;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.UsersStatusblocked;
import ru.yandex.direct.dbschema.ppc.enums.WalletCampaignsAutopayMode;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbschema.ppc.tables.Phrases;
import ru.yandex.direct.dbschema.ppc.tables.records.CampOptionsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsMobileContentRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.CampsForServicingRecord;
import ru.yandex.direct.dbutil.ConditionsAccumulator;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.queryrec.model.Language;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.jooq.impl.DSL.case_;
import static org.jooq.impl.DSL.coalesce;
import static org.jooq.impl.DSL.count;
import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.ifnull;
import static org.jooq.impl.DSL.max;
import static org.jooq.impl.DSL.sum;
import static org.jooq.impl.DSL.when;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.common.util.RepositoryUtils.nullSafeReader;
import static ru.yandex.direct.common.util.RepositoryUtils.nullSafeWriter;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.allowedPageIdsFromDb;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.pageIdsToDb;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.statusBsSyncedToDb;
import static ru.yandex.direct.core.entity.client.repository.ClientOptionsRepository.IS_AUTO_OVERDRAFT_SWITCHED_ON_FIELD;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_CPM_YNDX_FRONTPAGE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPS_FOR_SERVICING;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_ADDITIONAL_DATA;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.MOBILE_APPS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.Tables.SUBCAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.Tables.VCARDS;
import static ru.yandex.direct.dbschema.ppc.Tables.WALLET_CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.WIDGET_PARTNER_CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName.autobudget_avg_cpi;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsType.internal_autobudget;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsType.internal_distrib;
import static ru.yandex.direct.dbschema.ppc.enums.CampaignsType.internal_free;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;
import static ru.yandex.direct.dbutil.SqlUtils.PRIMARY;
import static ru.yandex.direct.dbutil.SqlUtils.STRAIGHT_JOIN;
import static ru.yandex.direct.dbutil.SqlUtils.findInSet;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromPropertyToField;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Repository
@ParametersAreNonnullByDefault
public class CampaignRepository {
    private final org.slf4j.Logger logger = LoggerFactory.getLogger(CampaignRepository.class);
    /**
     * Общеупотребимый алиас для таблицы {@code campaigns} используемый для кампании - общего счета
     * При использовании извне репозитория - учитывайте, что алиас таблицы должен совпадать
     */
    public static final Campaigns WALLETS = CAMPAIGNS.as("wc");

    public static final Long DEFAULT_MONEY_WARNING_VALUE = 20L;

    // не отправляли писем с момента последнего зачисления средств
    public static final Long STATUS_MAIL_NO_MAIL_SEND = 0L;
    // отправляли письмо о том что денег на общем счете осталось примерно на один день
    public static final Long STATUS_MAIL_ONE_DAY_WARN_SEND = 1L;
    // отправляли письмо о том что денег на общем счете осталось примерно на три дня
    public static final Long STATUS_MAIL_THREE_DAYS_WARN_SEND = 4L;

    private static final Long DEFAULT_ORDER_ID = 0L;
    private static final long DEFAULT_DAY_BUDGET_CHANGE_COUNT = 0L;
    private static final Set<CampaignsType> INTERNAL_CAMPAIGN_TYPES =
            mapSet(CampaignTypeKinds.INTERNAL, CampaignType::toSource);
    private static final Set<CampaignsType> MOD_EXPORT_ONLY_TYPES =
            mapSet(CampaignTypeKinds.MOD_EXPORT_CAMPAIGNS_ONLY, CampaignType::toSource);
    private static final Set<CampaignsType> SKIP_IN_CLIENT_CAMPAIGNS_COUNT =
            mapSet(CampaignTypeKinds.SKIP_IN_CLIENT_CAMPAIGN_COUNT, CampaignType::toSource);
    private static final Set<CampaignsType> WEB_EDIT_BASE_TYPES =
            mapSet(CampaignTypeKinds.WEB_EDIT_BASE, CampaignType::toSource);
    private static final Set<CampaignsType> DAY_BUDGETS =
            mapSet(CampaignTypeKinds.DAY_BUDGET, CampaignType::toSource);
    private static final Set<CampaignsType> USE_IN_GET_CLIENT_MANAGER =
            Stream.of(CampaignTypeKinds.WEB_EDIT_BASE, CampaignTypeKinds.WITH_CURRENCY)
                    .flatMap(Set::stream)
                    .map(CampaignType::toSource)
                    .collect(toSet());

    public static final Long IS_CPD_PAUSED = 1L;

    private static final Field<BigDecimal> CAMPAIGN_AND_WALLET_SUM =
            CAMPAIGNS.SUM.plus(JooqMapperUtils.mysqlIf(WALLETS.CID, WALLETS.SUM, DSL.val(BigDecimal.ZERO)));
    private static final Collection<Field<?>> DISABLED_DOMAINS_TO_ID_MAP_FIELDS =
            asList(CAMPAIGNS.CID, CAMPAIGNS.DONT_SHOW);
    private static final Collection<Field<?>> GEO_TO_ID_MAP_FIELDS =
            asList(CAMPAIGNS.CID, CAMPAIGNS.GEO);
    private static final String CNT_ON_MODERATION_ALIAS = "cnt_on_moderation";
    private static final String CNT_MODERATED_ALIAS = "cnt_moderated";

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Campaign> campaignMapper;
    private final JooqMapperWithSupplier<DbStrategy> dbStrategyMapper;
    private final JooqMapperWithSupplier<BaseCampaign> campsForServicingMapper;
    private final Collection<Field<?>> campaignMapperFields;
    private final Collection<Field<?>> campaignWithStrategyMapperFields;
    private final Collection<Field<?>> campaignDayBudgetOptionsMapperFields;
    private final Collection<Field<?>> campaignSimpleMapperFields;
    private final Collection<Field<?>> campaignForForecastMapperFields;
    private final Collection<Field<?>> walletCampaignFieldsToRead;

    @Autowired
    public CampaignRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        campaignMapper = createCampaignMapper();
        dbStrategyMapper = createDbStrategyMapper(true);
        campsForServicingMapper = createCampsForServicingMapper();

        campaignMapperFields = campaignMapper.getFieldsToRead();
        campaignWithStrategyMapperFields =
                StreamEx.of(campaignMapperFields)
                        .append(dbStrategyMapper.getFieldsToRead())
                        .distinct()
                        .toList();
        campaignDayBudgetOptionsMapperFields =
                campaignMapper.getFieldsToRead(CampaignDayBudgetOptions.allModelProperties());
        campaignSimpleMapperFields = campaignMapper.getFieldsToRead(CampaignSimple.allModelProperties());
        campaignForForecastMapperFields = campaignMapper.getFieldsToRead(CampaignForForecast.allModelProperties());
        walletCampaignFieldsToRead = StreamEx.of(campaignMapper.getFieldsToRead(WalletCampaign.allModelProperties()))
                .append(dbStrategyMapper.getFieldsToRead())
                .distinct()
                .toList();
    }

    /**
     * Получает подлежащие кампании для всех переданных мастер-кампаний.
     *
     * @return Map (подлежащая кампания -> мастер-кампания)
     */
    public Map<Long, Long> getSubCampaignIdsWithMasterIds(int shard, Collection<Long> masterIds) {
        return dslContextProvider.ppc(shard)
                .select(SUBCAMPAIGNS.CID, SUBCAMPAIGNS.MASTER_CID)
                .from(SUBCAMPAIGNS)
                .where(SUBCAMPAIGNS.MASTER_CID.in(masterIds))
                .fetchMap(SUBCAMPAIGNS.CID, SUBCAMPAIGNS.MASTER_CID);
    }

    /**
     * Получение кампаний-кошельков по кампаниям, и всех кампаний, подключенных к ним – TODO переписать
     * в виде {@link Multimap}: кошелёк -> кампании под этим кошельком.
     * Если у кампании нет общего счёта (не привязана к кошельку и сама не кошелёк),
     * в результирующей структуре её не будет.
     */
    public WalletsWithCampaigns getWalletAllCampaigns(int shard, ClientId clientId, Collection<Campaign> campaigns) {
        Collection<WalletCampaign> wallets = getWalletsByCampaigns(shard, campaigns);
        Set<Long> foundWalletCampaignIds = listToSet(wallets, WalletCampaign::getId);

        Collection<WalletCampaign> campaignsFromDb = dslContextProvider.ppc(shard)
                .select(walletCampaignFieldsToRead)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                        .and(CAMPAIGNS.WALLET_CID.in(foundWalletCampaignIds)))
                .fetch()
                .map(this::campaignWithStrategyFromDb);

        return new WalletsWithCampaigns(wallets, campaignsFromDb);
    }

    /**
     * Получение отношения кампаний-кошельков и их кампаний (WalletsWithCampaigns) кампаний-кошельков
     * по id кампаний-кошельков.
     * Если кампании-кошелька не существует, то в результирующей структуре её не будет.
     */
    public WalletsWithCampaigns getWalletsWithCampaignsByWalletCampaignIds(int shard,
                                                                           Collection<Long> walletCampaignIds,
                                                                           boolean activeOnly) {
        Collection<WalletCampaign> wallets = getWalletsByWalletCampaignIds(shard, walletCampaignIds);
        Set<Long> foundWalletCampaignIds = listToSet(wallets, WalletCampaign::getId);

        Collection<WalletCampaign> campaignsUnderWallets = getCampaignsByWalletIds(
                shard, foundWalletCampaignIds, activeOnly);

        return new WalletsWithCampaigns(wallets, campaignsUnderWallets);
    }

    /**
     * Получение кампаний по id кошельков
     */
    public Collection<WalletCampaign> getCampaignsByWalletIds(
            int shard, Collection<Long> walletIds, boolean activeOnly) {

        Condition activeOnlyCondition = DSL.trueCondition();
        if (activeOnly) {
            activeOnlyCondition = activeOnlyCondition
                    .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                    .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                    .and(CAMPAIGNS.FINISH_TIME.eq(SqlUtils.mysqlZeroLocalDate())
                            .or(DSL.localDateDiff(DSL.currentLocalDate(), CAMPAIGNS.FINISH_TIME).lessOrEqual(0)));
        }
        return dslContextProvider.ppc(shard)
                .select(walletCampaignFieldsToRead)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.WALLET_CID.in(walletIds).and(activeOnlyCondition))
                .fetch()
                .map(this::campaignWithStrategyFromDb);
    }

    /**
     * Получение кампаний-кошельков по кампаниям
     */
    public Collection<WalletCampaign> getWalletsByCampaigns(int shard, Collection<Campaign> campaigns) {
        Set<Long> walletCampaignIds = filterAndMapToSet(campaigns, c -> c.getWalletId() > 0, Campaign::getWalletId);
        return getWalletsByWalletCampaignIds(shard, walletCampaignIds);
    }

    /**
     * Получение кампаний-кошельков по id кампаний-кошельков
     */
    public Collection<WalletCampaign> getWalletsByWalletCampaignIds(int shard, Collection<Long> walletCampaignIds) {
        return dslContextProvider.ppc(shard)
                .select(walletCampaignFieldsToRead)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.in(walletCampaignIds))
                .fetch()
                .map(this::campaignWithStrategyFromDb);
    }

    /**
     * Получение кампаний-кошельков по которым были оплаты или, возможно, нужно отправить письмо об
     * окончании средств
     * учитываем только кошельки с выключенным авто пополнением
     */
    public Collection<WalletCampaign> getWalletsForWarnSend(int shard) {
        return dslContextProvider.ppc(shard)
                .select(walletCampaignFieldsToRead)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(WALLET_CAMPAIGNS).on(WALLET_CAMPAIGNS.WALLET_CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.TYPE.eq(CampaignsType.wallet))
                .and(CAMPAIGNS.STATUS_MAIL.in(STATUS_MAIL_NO_MAIL_SEND, STATUS_MAIL_THREE_DAYS_WARN_SEND))
                .and(CAMPAIGNS.SUM.greaterThan(Currencies.EPSILON))
                .and(CAMP_OPTIONS.MONEY_WARNING_VALUE.eq(DEFAULT_MONEY_WARNING_VALUE))
                .and(WALLET_CAMPAIGNS.AUTOPAY_MODE.isNull()
                        .or(WALLET_CAMPAIGNS.AUTOPAY_MODE.eq(WalletCampaignsAutopayMode.none)))
                .fetch()
                .map(this::campaignWithStrategyFromDb);
    }

    /**
     * Возвращает кампании, готовые к тому, чтобы им было возможно отправить уведомление о том,
     * что они остановлены по дневному бюджету
     *
     * @param shard шард
     * @return кампании, готовые к отправке уведомления
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r8434756#L81-94">
     * Оригинал
     * </a>
     */
    public Collection<Campaign> getCampaignsReadyForDayBudgetNotifications(int shard) {
        var condition = DSL.and(
                readyForBudgetNotificationCondition(),
                makeSenseToSendNotification(),
                havePausedByDayBudgetEmailSendingPermission()
        );
        return getCampaignsSatisfyingCondition(shard, condition);
    }

    /**
     * Возвращает кампании, готовые к отправке уведомления об остановке по дневному бюджету,
     * привязанные к кошелькам из переданного набора
     *
     * @param shard   шард
     * @param wallets набор кошельков
     * @return кампании готовые к отправке уведомления кошельки которых принадлежат {@code walletIds}
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r8434756#L170-181">
     * Оригинал
     * </a>
     */
    public Collection<Campaign> getCampaignsUnderWalletsReadyForDayBudgetNotification(int shard,
                                                                                      Collection<Campaign> wallets) {
        var condition = DSL.and(
                makeSenseToSendNotification(),
                CAMPAIGNS.WALLET_CID.in(mapList(wallets, Campaign::getId)),
                CAMPAIGNS.UID.in(mapList(wallets, Campaign::getUserId))
        );
        return getCampaignsSatisfyingCondition(shard, condition);
    }

    /**
     * Вспомогательный метод.
     * Из кампаний, привязанных к какому-нибудь кошельку, выбирает те, которые удовлетворяют переданному условию
     *
     * @param shard     шард
     * @param condition условие
     * @return кампании удовлетворяющие условию
     */
    private Collection<Campaign> getCampaignsSatisfyingCondition(int shard, Condition condition) {
        Campaigns wc = CAMPAIGNS.as("wc");
        return dslContextProvider.ppc(shard)
                .select(campaignMapperFields)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .join(USERS).on(USERS.UID.eq(CAMPAIGNS.UID))
                .leftJoin(wc).on(wc.CID.eq(CAMPAIGNS.WALLET_CID))
                .where(condition)
                .fetch()
                .map(campaignMapper::fromDb);
    }

    /**
     * Возвращает ОС, готовые к тому, чтобы им было возможно отправить уведомление о том,
     * что они остановлены по дневному бюджету
     *
     * @param shard шард
     * @return ОС, готовые к отправке уведомления
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r8434756#L146-162">
     * Оригинал
     * </a>
     */
    public Collection<Campaign> getWalletsReadyForDayBudgetNotifications(int shard) {
        return dslContextProvider.ppc(shard)
                .select(campaignMapperFields)
                .from(WALLET_CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(WALLET_CAMPAIGNS.WALLET_CID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(WALLET_CAMPAIGNS.WALLET_CID))
                .where(CAMPAIGNS.TYPE.eq(CampaignsType.wallet))
                .and(readyForBudgetNotificationCondition())
                .and(DSL.or(havePausedByDayBudgetSmsSendingPermission(), havePausedByDayBudgetEmailSendingPermission()))
                .fetch()
                .map(campaignMapper::fromDb);
    }

    public boolean didClientHavePausedByDayBudgetNotification(int shard, ClientId clientId) {
        return !dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.TYPE.eq(CampaignsType.wallet))
                .and(DSL.or(havePausedByDayBudgetSmsSendingPermission(), havePausedByDayBudgetEmailSendingPermission()))
                .fetch(CAMPAIGNS.CID)
                .isEmpty();
    }


    /**
     * @return условие, по которому определяется, готова ли кампания к отправке уведомления
     * об остановке по дневному бюджету
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r7939428#L44-47">
     * Оригинал
     * </a>
     */
    private @NotNull Condition readyForBudgetNotificationCondition() {
        return CAMP_OPTIONS.DAY_BUDGET_STOP_TIME.ge(LocalDate.now(ZoneId.of("Europe/Moscow")).atStartOfDay())
                .and(CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS.eq(
                        ru.yandex.direct.dbschema.ppc.enums.CampOptionsDayBudgetNotificationStatus.Ready
                ))
                .and(CAMPAIGNS.DAY_BUDGET.gt(BigDecimal.ZERO));
    }


    private @NotNull Condition havePausedByDayBudgetEmailSendingPermission() {
        return SqlUtils.findInSet(
                CampaignMappings.emailNotificationsToDb(EnumSet.of(CampaignEmailNotification.PAUSED_BY_DAY_BUDGET)),
                CAMP_OPTIONS.EMAIL_NOTIFICATIONS
        ).gt(0L);
    }

    private @NotNull Condition havePausedByDayBudgetSmsSendingPermission() {
        return SqlUtils.findInSet(
                CampaignMappings.smsFlagsToDb(EnumSet.of(SmsFlag.PAUSED_BY_DAY_BUDGET_SMS)),
                CAMP_OPTIONS.SMS_FLAGS
        ).gt(0L);
    }

    /**
     * @return условие, по которому определяется, имеет ли смысл посылать уведомление
     * об остановке по дневному бюджету данной кампании
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DayBudgetAlerts.pm?rev=r8434756#L50-57">
     * Оригинал
     * </a>
     */
    private @NotNull Condition makeSenseToSendNotification() {
        return CAMPAIGNS.ORDER_ID.gt(0L)
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.STATUS_MODERATE.eq(CampaignsStatusmoderate.Yes))
                .and(CAMPAIGNS.TYPE.in(DAY_BUDGETS));
    }

    /**
     * Пометить кампании, которым было послано уведомление
     *
     * @param shard       шард
     * @param campaignIds кампании, для которых нужно обновить статус
     */
    public void markDayBudgetNotificationsAsSent(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS,
                        ru.yandex.direct.dbschema.ppc.enums.CampOptionsDayBudgetNotificationStatus.Sent)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Получение долгов по id кампаний-кошельков
     */
    public Map<Long, BigDecimal> getWalletsDebt(int shard, Collection<Long> walletIds) {
        var c = CAMPAIGNS.as("c");
        var debtCol = DSL.sum(DSL.iif(c.SUM.lt(c.SUM_SPENT), c.SUM_SPENT.minus(c.SUM), BigDecimal.ZERO)).as("debt");

        return dslContextProvider.ppc(shard)
                .select(c.WALLET_CID, debtCol)
                .from(c)
                .where(
                        c.WALLET_CID.in(walletIds)
                                .and(c.WALLET_CID.gt(0L))
                                .and(c.TYPE.in(mapList(CampaignTypeKinds.UNDER_WALLET, CampaignType::toSource)))
                                .and(c.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                )
                .groupBy(c.WALLET_CID)
                .fetchMap(c.WALLET_CID, debtCol);
    }

    private static CampaignWithType campaignWithTypeMapper(Record4<Long, Long, CampaignsType, CampaignsArchived> r) {
        return new CampaignWithType(
                r.get(CAMPAIGNS.CID),
                CampaignType.fromSource(r.get(CAMPAIGNS.TYPE)),
                r.get(CAMPAIGNS.ARCHIVED) == CampaignsArchived.Yes);
    }

    private static CampaignWithType campaignSimpleToCampaignWithType(CampaignSimple campaignSimple) {
        return new CampaignWithType(campaignSimple.getId(),
                campaignSimple.getType(),
                campaignSimple.getStatusArchived());
    }

    /**
     * Получить словарь соответствий типа кампании её номеру
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний
     */
    public Map<Long, CampaignType> getCampaignsTypeMap(int shard, Collection<Long> campaignIds) {
        return getCampaignsTypeMap(dslContextProvider.ppc(shard), campaignIds);
    }

    public Map<Long, CampaignType> getCampaignsTypeMap(DSLContext dslContext, Collection<Long> campaignIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withCampaignIds(campaignIds);
        return getCampaignsTypeMap(dslContext, selectionCriteria);
    }

    /**
     * Получить словарь соответствий типа кампании её номеру
     *
     * @param shard       шард
     * @param clientId    идентификатор клиента
     * @param campaignIds идентификаторы кампаний
     */
    public Map<Long, CampaignType> getCampaignsTypeMap(int shard, ClientId clientId, Collection<Long> campaignIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignIds(campaignIds);
        return getCampaignsTypeMap(shard, selectionCriteria);
    }

    /**
     * Получить словарь соответствий типа кампании её номеру
     *
     * @param shard          шард
     * @param clientId       идентификатор клиента
     * @param campaignIds    идентификаторы кампаний
     * @param allowableTypes типы "кампаний" из таблицы CAMPAIGNS
     */
    public Map<Long, CampaignType> getCampaignsTypeMap(int shard, ClientId clientId, Collection<Long> campaignIds,
                                                       Collection<CampaignType> allowableTypes) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignIds(campaignIds)
                .withCampaignTypes(allowableTypes);
        return getCampaignsTypeMap(shard, selectionCriteria);
    }

    private Map<Long, CampaignType> getCampaignsTypeMap(int shard, CampaignsSelectionCriteria selectionCriteria) {
        return getCampaignsTypeMap(dslContextProvider.ppc(shard), selectionCriteria);
    }

    private Map<Long, CampaignType> getCampaignsTypeMap(DSLContext dslContext,
                                                        CampaignsSelectionCriteria selectionCriteria) {
        return getCampaignsSimpleStream(dslContext, selectionCriteria)
                .toMap(CampaignSimple::getId, CampaignSimple::getType);
    }

    /**
     * Получить словарь соответствий информации о привязанном к кампании email и времени отправки сообщения
     * из camp_options её номеру.
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний
     */
    public Map<Long, Pair<String, TimeInterval>> getCampaignsEmailAndSmsTimeMap(int shard,
                                                                                Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.EMAIL, CAMP_OPTIONS.SMS_TIME)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID,
                        r -> Pair.of(
                                r.get(CAMP_OPTIONS.EMAIL),
                                CampaignMappings.smsTimeFromDb(r.get(CAMP_OPTIONS.SMS_TIME))
                        )
                );
    }

    /**
     * Получить словарь соответствий типа и источника кампании её номеру
     *
     * @param shard             шард
     * @param campaignIds       идентификаторы кампаний
     * @param filterStatusEmpty оставить только кампании с statusEmpty=No
     */
    public Map<Long, CampaignTypeSource> getCampaignsTypeSourceMap(int shard,
                                                                   Collection<Long> campaignIds,
                                                                   boolean filterStatusEmpty) {
        var selectStep = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.TYPE, CAMPAIGNS.SOURCE)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds));
        if (filterStatusEmpty) {
            selectStep = selectStep.and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));
        }

        return selectStep
                .fetchMap(CAMPAIGNS.CID,
                        r -> new CampaignTypeSource(
                                CampaignType.fromSource(r.get(CAMPAIGNS.TYPE)),
                                CampaignSource.fromSource(r.get(CAMPAIGNS.SOURCE))
                        )
                );
    }

    public Map<Long, Language> getCampaignsLang(int shard, ClientId clientId, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.CONTENT_LANG)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID, rec -> Language.getByName(rec.getValue(CAMP_OPTIONS.CONTENT_LANG)));
    }

    public Map<Long, Language> getCampaignsLang(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.CONTENT_LANG)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID, rec -> Language.getByName(rec.getValue(CAMP_OPTIONS.CONTENT_LANG)));
    }

    public void updateCampaignLang(int shard, Collection<Long> campaignIds, @Nullable ContentLanguage language) {
        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.CONTENT_LANG, ifNotNull(language, ContentLanguage::getTypedValue))
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .execute();
    }

    public Map<Long, Language> getCampaignsLangByAdGroupIds(int shard, ClientId clientId,
                                                            Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(PHRASES.PID, CAMP_OPTIONS.CONTENT_LANG)
                .from(PHRASES)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(PHRASES.CID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(PHRASES.PID.in(adGroupIds))
                .fetchMap(PHRASES.PID, rec -> Language.getByName(rec.getValue(CAMP_OPTIONS.CONTENT_LANG)));
    }

    /**
     * Получить email кампаний из camp_options по их id
     */
    public Map<Long, String> getEmailByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.EMAIL)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID, CAMP_OPTIONS.EMAIL);
    }

    /**
     * Получить place_id внутренней рекламы по идентификаторам групп
     *
     * @param shard      Шард
     * @param adGroupIds Идентификаторы групп
     */
    public Map<Long, Long> getCampaignInternalPlacesByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(PHRASES.PID, CAMPAIGNS_INTERNAL.PLACE_ID)
                .from(PHRASES)
                .join(CAMPAIGNS_INTERNAL).on(CAMPAIGNS_INTERNAL.CID.eq(PHRASES.CID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_INTERNAL.CID))
                .where(PHRASES.PID.in(adGroupIds))
                .fetchMap(PHRASES.PID, CAMPAIGNS_INTERNAL.PLACE_ID);
    }

    /**
     * Получить place_id внутренней рекламы по идентификаторам кампаний.
     * <p>
     * Метод не проверяет видимость кампаний клиентом. Об этом должен позаботится вызывающий код.
     *
     * @param campaignIds Идентификаторы кампаний
     */
    public Map<Long, Long> getCampaignInternalPlaces(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS_INTERNAL.CID, CAMPAIGNS_INTERNAL.PLACE_ID)
                .from(CAMPAIGNS_INTERNAL)
                .where(CAMPAIGNS_INTERNAL.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS_INTERNAL.CID, CAMPAIGNS_INTERNAL.PLACE_ID);
    }

    /**
     * Получить признак мобильности кампании внутренней рекламы по идентификаторам кампаний.
     * <p>
     * Метод не проверяет видимость кампаний клиентом. Об этом должен позаботится вызывающий код.
     *
     * @param campaignIds Идентификаторы кампаний
     */
    public Map<Long, Boolean> getCampaignInternalIsMobile(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS_INTERNAL.CID, CAMPAIGNS_INTERNAL.IS_MOBILE)
                .from(CAMPAIGNS_INTERNAL)
                .where(CAMPAIGNS_INTERNAL.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS_INTERNAL.CID, r -> r.get(CAMPAIGNS_INTERNAL.IS_MOBILE) != 0);
    }

    public Set<Long> getExistingCampaignIds(int shard, ClientId clientId, Collection<Long> campaignIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignIds(campaignIds);

        return getCampaignIds(shard, selectionCriteria);
    }

    public Set<Long> getExistingNotEmptyCampaignIds(int shard, ClientId clientId, Collection<Long> campaignIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignIds(campaignIds)
                .withStatusEmpty(false);
        return getCampaignIds(shard, selectionCriteria);
    }

    public Set<Long> getExistingNonSubCampaignIds(int shard, ClientId clientId, Collection<Long> campaignIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignIds(campaignIds)
                .withSubCampaigns(false);
        return getCampaignIds(shard, selectionCriteria);
    }

    /**
     * Получить множество id архивных кампаний,
     * среди тех, которые переданы в качестве параметра.
     *
     * @param shard       шард
     * @param campaignIds коллекция id кампаний, среди которых
     *                    необходимо выбрать архивные
     * @return множество архивных кампаний из тех,
     * которые переданы в качестве параметра
     */
    public Set<Long> getArchivedCampaigns(int shard, Collection<Long> campaignIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withCampaignIds(campaignIds)
                .withStatusArchived(true);
        return getCampaignIds(shard, selectionCriteria);
    }

    /**
     * Возвращает все id кампаний типа смарт-баннер для данного списка clientIds
     */
    public Set<Long> getNotEmptySmartCampaignsByClientIds(int shard, List<ClientId> clientIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withCampaignTypes(List.of(CampaignType.PERFORMANCE))
                .withStatusEmpty(false);
        return getCampaignIds(shard, selectionCriteria);
    }

    /**
     * Получит id кампаний по id клиентов
     *
     * @param shard     шард
     * @param clientIds id клиентов
     */
    public Set<Long> getCampaignIdsByClientIds(int shard, Collection<ClientId> clientIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withClientIds(clientIds);
        return getCampaignIds(shard, selectionCriteria);
    }

    /**
     * Получит id не-подлежащих кампаний по id клиентов
     *
     * @param shard     шард
     * @param clientIds id клиентов
     */
    public Set<Long> getNonSubCampaignIdsByClientIds(int shard, Collection<ClientId> clientIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withSubCampaigns(false);
        return getCampaignIds(shard, selectionCriteria);
    }

    public Set<Long> getNonArchivedNonDeletedCampaignIdsByClientIds(int shard, Collection<ClientId> clientIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withStatusArchived(false)
                .withStatusEmpty(false);
        return getCampaignIds(shard, selectionCriteria);
    }

    public Set<Long> getNonArchivedNonDeletedCampaignIdsByClientIdsAndTypes(int shard,
                                                                            Collection<ClientId> clientIds,
                                                                            Collection<CampaignType> campaignTypes) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withCampaignTypes(campaignTypes)
                .withStatusArchived(false)
                .withStatusEmpty(false);
        return getCampaignIds(shard, selectionCriteria);
    }

    /**
     * Получит id кампаний указанных типов по id клиентов
     *
     * @param shard         шард
     * @param clientIds     id клиентов
     * @param campaignTypes типы кампаний
     */
    public Set<Long> getCampaignIdsByClientIds(int shard, Collection<ClientId> clientIds,
                                               Collection<CampaignType> campaignTypes) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withCampaignTypes(campaignTypes);
        return getCampaignIds(shard, selectionCriteria);
    }

    /**
     * Получит id активных кампаний по id клиентов
     *
     * @param shard     шард
     * @param clientIds id клиентов
     */
    public Set<Long> getActiveCampaignIdsByClientIds(int shard, Collection<ClientId> clientIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withStatusActived(true);
        return getCampaignIds(shard, selectionCriteria);
    }

    private Set<Long> getCampaignIds(int shard, CampaignsSelectionCriteria selectionCriteria) {
        return getCampaignsWithFieldsStream(dslContextProvider.ppc(shard), selectionCriteria,
                List.of(CAMPAIGNS.CID), false)
                .map(Campaign::getId)
                .toSet();
    }

    /**
     * Получить список неостановленных кампаний под кошельками
     */
    public List<Long> getStartedCampaignIdsByWalletIds(int shard, List<Long> walletIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.WALLET_CID.in(walletIds))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .fetch(CAMPAIGNS.CID);
    }

    public Boolean getStatusShowByCampaignId(int shard, Long campaignId) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.STATUS_SHOW)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.eq(campaignId))
                .fetchOne(r -> CampaignMappings.statusShowFromDb(r.get(CAMPAIGNS.STATUS_SHOW)));
    }

    private StreamEx<CampaignWithTypeAndWalletId> getCampaignWithTypeAndWalletIdStream(int shard,
                                                                                       Collection<Long> campaignIds) {
        Set<Field<?>> fields = campaignMapper.getFieldsToRead(CampaignWithTypeAndWalletId.allModelProperties());
        return getCampaignsWithFieldsStream(shard, campaignIds, fields)
                .map(Function.identity());
    }

    public List<Campaign> getCampaigns(int shard,
                                       CampaignsSelectionCriteria selectionCriteria) {
        return getCampaigns(dslContextProvider.ppc(shard), selectionCriteria, false);
    }

    public List<Campaign> getCampaigns(DSLContext dslContext,
                                       CampaignsSelectionCriteria selectionCriteria,
                                       boolean forUpdate) {
        return getCampaignsWithFieldsStream(dslContext, selectionCriteria, campaignWithStrategyMapperFields, forUpdate)
                .toList();
    }

    public Map<Long, Long> getMasterIdBySubCampaignId(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(SUBCAMPAIGNS.CID, SUBCAMPAIGNS.MASTER_CID)
                .from(SUBCAMPAIGNS)
                .where(SUBCAMPAIGNS.CID.in(campaignIds))
                .fetchMap(SUBCAMPAIGNS.CID, SUBCAMPAIGNS.MASTER_CID);
    }

    public Map<Long, List<Long>> getSubCampaignIdsByMasterId(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(SUBCAMPAIGNS.MASTER_CID, SUBCAMPAIGNS.CID)
                .from(SUBCAMPAIGNS)
                .where(SUBCAMPAIGNS.MASTER_CID.in(campaignIds))
                .fetchGroups(SUBCAMPAIGNS.MASTER_CID, SUBCAMPAIGNS.CID);
    }

    /**
     * Получить wallet_cid-ы кампаний по id кампаний. Если кампания не имеет wallet_cid, то для нее в результате
     * будет 0.
     *
     * @param shard       шард.
     * @param campaignIds id кампаний.
     * @return {campaignId -> walletId}
     */
    public Map<Long, Long> getWalletIdsByCampaingIds(int shard, Collection<Long> campaignIds) {
        return getCampaignWithTypeAndWalletIdStream(shard, campaignIds)
                .toMap(CampaignWithTypeAndWalletId::getId, CampaignWithTypeAndWalletId::getWalletId);
    }

    /**
     * Получить информацию о типе кампании и её привязке к общему счету.
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний
     */
    public List<CampaignWithTypeAndWalletId> getCampaignsWithTypeAndWalletId(int shard, Collection<Long> campaignIds) {
        return getCampaignWithTypeAndWalletIdStream(shard, campaignIds)
                .toList();
    }

    /**
     * Получение clientId по cid
     *
     * @return {cid → clientId}
     */
    public Map<Long, Long> getClientIdsForCids(int shard, Collection<Long> cids) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withCampaignIds(cids);
        return getCampaignsSimpleStream(shard, selectionCriteria)
                .toMap(CampaignSimple::getId, CampaignSimple::getClientId);
    }

    /**
     * Поиск неархивных кампаний по id и имени для одного ClientID
     *
     * @param campaignTypes какие типы кампаний искать
     */
    public List<CampaignSimple> searchNotEmptyNotArchivedCampaigns(int shard, ClientId clientId,
                                                                   Collection<CampaignType> campaignTypes) {
        return searchNotEmptyNotArchivedCampaigns(shard, singletonList(clientId), campaignTypes)
                .getOrDefault(clientId.asLong(), emptyList());
    }

    /**
     * Поиск неархивных кампаний по id и имени для списка ClientID
     *
     * @param campaignTypes какие типы кампаний искать
     */
    public Map<Long, List<CampaignSimple>> searchNotEmptyNotArchivedCampaigns(int shard, Collection<ClientId> clientIds,
                                                                              Collection<CampaignType> campaignTypes) {
        if (clientIds.isEmpty()) {
            return emptyMap();
        }

        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withCampaignTypes(campaignTypes)
                .withStatusArchived(false)
                .withStatusEmpty(false);

        return getCampaignsSimpleStream(shard, selectionCriteria)
                .groupingBy(CampaignSimple::getClientId);
    }

    /**
     * Поиск созданных кампаний для одного ClientID
     *
     * @param campaignTypes какие типы кампаний искать
     */
    public List<Campaign> searchNotEmptyCampaigns(int shard, ClientId clientId,
                                                  Collection<CampaignType> campaignTypes) {
        return searchNotEmptyCampaigns(shard, singletonList(clientId), campaignTypes)
                .getOrDefault(clientId.asLong(), emptyList());
    }

    /**
     * Поиск созданных кампаний для списка ClientID
     *
     * @param campaignTypes какие типы кампаний искать
     */
    private Map<Long, List<Campaign>> searchNotEmptyCampaigns(int shard, Collection<ClientId> clientIds,
                                                              Collection<CampaignType> campaignTypes) {
        if (clientIds.isEmpty()) {
            return emptyMap();
        }

        DSLContext dslContext = dslContextProvider.ppc(shard);
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withCampaignTypes(campaignTypes)
                .withStatusEmpty(false);

        return getCampaignsWithFieldsStream(dslContext, selectionCriteria, campaignWithStrategyMapperFields, false)
                .groupingBy(Campaign::getClientId);
    }

    public Map<Long, List<CampaignSimple>> searchAllCampaigns(int shard, Collection<ClientId> clientIds,
                                                              Collection<CampaignType> campaignTypes) {
        if (clientIds.isEmpty()) {
            return emptyMap();
        }

        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientIds(clientIds)
                .withCampaignTypes(campaignTypes);

        return getCampaignsSimpleStream(shard, selectionCriteria)
                .groupingBy(CampaignSimple::getClientId);
    }

    public Map<Long, CampaignSimple> getCampaignsSimple(int shard, Collection<Long> campaignIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withCampaignIds(campaignIds);
        return getCampaignsSimpleStream(shard, selectionCriteria)
                .toMap(CampaignSimple::getId, Function.identity());
    }

    public List<CampaignSimple> getCampaignsSimpleByClientId(int shard, ClientId clientId) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withClientId(clientId);
        return getCampaignsSimpleStream(shard, selectionCriteria)
                .collect(toList());
    }

    public List<CampaignSimple> getCampaignsSimpleBySelectionCriteria(int shard,
                                                                      CampaignsSelectionCriteria selectionCriteria) {
        return getCampaignsSimpleStream(shard, selectionCriteria).collect(toList());
    }

    /**
     * Получить поток CampaignSimple по CampaignSelectionCriteria
     *
     * @param shard             шард для выборки данных
     * @param selectionCriteria условие выборки
     * @return StreamEx поток CampaignSimple
     */
    private StreamEx<CampaignSimple> getCampaignsSimpleStream(int shard, CampaignsSelectionCriteria selectionCriteria) {
        return getCampaignsSimpleStream(dslContextProvider.ppc(shard), selectionCriteria);
    }

    private StreamEx<CampaignSimple> getCampaignsSimpleStream(DSLContext dslContext,
                                                              CampaignsSelectionCriteria selectionCriteria) {
        return getCampaignsWithFieldsStream(dslContext, selectionCriteria, campaignSimpleMapperFields, false)
                .map(Function.identity());
    }

    public Map<Long, CampaignSimple> getCampaignsSimpleByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return getCampaignsSimpleByAdGroupIds(dslContextProvider.ppc(shard), adGroupIds);
    }

    public Map<Long, CampaignSimple> getCampaignsSimpleByAdGroupIds(DSLContext context, Collection<Long> adGroupIds) {
        Map<Long, Long> campaignIdsByAdGroupId = new HashMap<>();
        Iterables.partition(adGroupIds, SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .forEach(adGroupIdsChunk -> {
                    Map<Long, Long> campaignIdsByAdGroupIdChunk = context
                            .select(PHRASES.PID, PHRASES.CID)
                            .from(PHRASES)
                            .where(PHRASES.PID.in(adGroupIdsChunk))
                            .fetchMap(PHRASES.PID, PHRASES.CID);
                    campaignIdsByAdGroupId.putAll(campaignIdsByAdGroupIdChunk);
                });

        List<Field<?>> fieldsToReadFromCampaigns =
                new ArrayList<>(campaignMapper.getFieldsToRead(CampaignSimple.allModelProperties()));
        Map<Long, CampaignSimple> campaignsById = context
                .select(fieldsToReadFromCampaigns)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.in(new HashSet<>(campaignIdsByAdGroupId.values())))
                .fetchMap(CAMPAIGNS.CID, campaignMapper::fromDb);

        return EntryStream.of(campaignIdsByAdGroupId)
                .mapValues(campaignsById::get)
                .toMap();
    }

    /**
     * Метод для получения параметров дневного бюджета для указанных кампаний
     *
     * @param shard       шард
     * @param campaignIds список id-кампаний, для которых нужно получить параметры дневного бюджета
     * @return Map-а ключами, которого будут id-кампаний, а значениями параметры дневного бюджета
     */
    public Map<Long, CampaignDayBudgetOptions> getDayBudgetOptions(int shard, Collection<Long> campaignIds) {
        return getCampaignsWithFieldsStream(shard, campaignIds, campaignDayBudgetOptionsMapperFields)
                .toMap(Campaign::getId, Function.identity());
    }

    /**
     * Метод для получения данных по кампаниям под кошельком для NotifyOrder'а
     *
     * @param shard    шард
     * @param uid      uid пользователя владельца кошелька
     * @param walletId id кампании-кошелька, для которой нужно получить "дочерние" кампании
     * @return список с данными по кампаниям под кошельком
     */
    public List<CampaignForNotifyOrder> getCampaignsForNotifyOrder(int shard, Long uid, Long walletId) {
        List<Campaign> campaigns = getCampaignsUnderWalletWithFields(shard, uid, walletId,
                campaignMapper.getFieldsToRead(CampaignForNotifyOrder.allModelProperties()));

        return campaigns.stream()
                .map(c -> (CampaignForNotifyOrder) c)
                .collect(toList());
    }

    /**
     * Кампании, для которых наступила установленная в них дата окончания, но
     * на кампании ещё остались деньги, за исключением универсальных кампаний
     *
     * @param shard           шард
     * @param finishedDaysAgo дней назад кампания завершилась
     * @return список с данными для отсылки уведомлений клиентам и менеджерам
     * об остановившихся по дате окончания кампаниям.
     */
    @QueryWithoutIndex("Вызывается только из jobs")
    public List<CampaignForNotifyFinishedByDate> getActiveCampaignsFinishedByDate(int shard,
                                                                                  List<Integer> finishedDaysAgo) {
        if (finishedDaysAgo.isEmpty()) {
            throw new IllegalArgumentException("Empty 'Finished days ago' list is not allowed.");
        }
        // запрос тяжёлый (полностью просматривает campaigns)
        final Condition finishedDaysAgoCondition =
                DSL.localDateDiff(DSL.currentLocalDate(), CAMPAIGNS.FINISH_TIME).minus(1).in(finishedDaysAgo);
        return dslContextProvider.ppc(shard)
                .select(campaignMapper.getFieldsToRead(CampaignForNotifyFinishedByDate.allModelProperties()))
                .from(CAMPAIGNS)
                .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .leftJoin(WALLETS).on(CAMPAIGNS.WALLET_CID.eq(WALLETS.CID))
                .where(campTypeIn(CampaignTypeKinds.CAMP_FINISH))
                .and(sourceNotIn(AvailableCampaignSources.INSTANCE.ucSources()))
                .and(CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.Yes))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(CAMPAIGNS.FINISH_TIME.ne(SqlUtils.mysqlZeroLocalDate()))
                .and(finishedDaysAgo.isEmpty() ? DSL.trueCondition() : finishedDaysAgoCondition)
                .fetch(campaignMapper::fromDb);

    }

    /**
     * Кампании для нотификаций URL мониторинга доменов
     *
     * @param shard   шард
     * @param domains URL-ы доменов (пара протокол+домен), по которым сработал мониторинг
     * @return отображение ID пользователей на отображение URL-ов доменов на список кампаний для нотификации
     */
    public Map<Long, Map<String, List<CampaignForNotifyUrlMonitoring>>> getCampaignsForNotifyUrlMonitoring(
            int shard, Collection<Pair<String, String>> domains) {

        Set<String> reversedDomains = domains.stream().map(Pair::getValue).map(StringUtils::reverse).collect(toSet());

        Set<String> urls = domains.stream()
                .map(p -> p.getLeft() + "://" + p.getRight())
                .collect(Collectors.toUnmodifiableSet());

        Set<Field<?>> fieldsToRead =
                campaignMapper.getFieldsToRead(CampaignForNotifyUrlMonitoring.allModelProperties());
        fieldsToRead.add(BANNERS.DOMAIN);
        fieldsToRead.add(BANNERS.HREF);
        fieldsToRead.add(IS_AUTO_OVERDRAFT_SWITCHED_ON_FIELD);

        List<Record> records = dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .hint(STRAIGHT_JOIN)
                .from(BANNERS.forceIndex(Indexes.BANNERS_I_REVERSE_DOMAIN.getName()))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .leftJoin(CLIENTS_OPTIONS).on(CLIENTS_OPTIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID))
                .where(campTypeIn(CampaignTypeKinds.ALLOW_DOMAIN_MONITORING))
                .and(BANNERS.REVERSE_DOMAIN.in(reversedDomains))
                .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes))
                .and(CAMP_OPTIONS.STATUS_METRICA_CONTROL.eq(CampOptionsStatusmetricacontrol.Yes))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(CAMPAIGNS.FINISH_TIME.eq(SqlUtils.mysqlZeroLocalDate())
                        .or(DSL.localDateDiff(DSL.currentLocalDate(), CAMPAIGNS.FINISH_TIME).lessOrEqual(0)))
                // на кампании ещё остались деньги (на самом деле еще может понадобиться посчитать остаток на ОС)
                .and(CAMPAIGNS.WALLET_CID.ne(0L)
                        .or(CAMPAIGNS.SUM.minus(CAMPAIGNS.SUM_SPENT).greaterThan(Currencies.EPSILON)))
                .groupBy(fieldsToRead)
                .fetch();

        Set<Long> walletIdsToCheck = records.stream()
                // считаем баланс ОС положительным у клиентов с включенным авто овердрафтом
                // и не учитываем для таких клиентов потраченное на кампаниях под ОС
                .filter(r -> !r.get(IS_AUTO_OVERDRAFT_SWITCHED_ON_FIELD))
                .map(r -> r.get(CAMPAIGNS.WALLET_CID))
                .filter(walletId -> walletId > 0)
                .collect(Collectors.toSet());

        Set<Long> walletIdsWithPositiveBalance = walletIdsToCheck.isEmpty() ? emptySet()
                : getWalletsWithPositiveBalanceForNotifyUrlMonitoring(shard, walletIdsToCheck);

        return records.stream()
                // оставить кампании под кошельками только с положительным балансом
                .filter(c -> c.get(CAMPAIGNS.WALLET_CID) == 0L
                        || c.get(IS_AUTO_OVERDRAFT_SWITCHED_ON_FIELD)
                        || walletIdsWithPositiveBalance.contains(c.get(CAMPAIGNS.WALLET_CID)))
                // протокол в href баннера соответствует ожиданиям
                .filter(c -> urls.stream().anyMatch(url -> c.get(BANNERS.HREF).startsWith(url)))
                .collect(
                        groupingBy(r -> r.get(CAMPAIGNS.UID),
                                groupingBy(r -> r.get(BANNERS.DOMAIN),
                                        mapping(r -> (CampaignForNotifyUrlMonitoring) campaignMapper.fromDb(r),
                                                toList()))));
    }

    /**
     * Кошельки с положительным остатком для нотификаций URL мониторинга доменов
     *
     * @param shard     шард
     * @param walletIds ID кошельков
     * @return ID кошельков с положительным остатком
     */
    private Set<Long> getWalletsWithPositiveBalanceForNotifyUrlMonitoring(int shard, Set<Long> walletIds) {
        Field<BigDecimal> spentOnCampaigns =
                when(CAMPAIGNS.SUM_SPENT.isNotNull(), CAMPAIGNS.SUM_SPENT).otherwise(BigDecimal.ZERO);

        // Оригинал из perl-a
        // https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/BS/CheckUrlAvailability.pm?rev=6335378#L533
        //
        //  "SELECT wc.wallet_cid",
        //          "FROM campaigns c",
        //          "JOIN wallet_campaigns wc ON wc.wallet_cid = c.wallet_cid",
        //          WHERE => [
        //  _OR => [
        //  map { (_AND => {'c.uid__int' => $wallets_to_check{$_}, 'c.wallet_cid__int' => $_ }) } @$chunk
        //                              ],
        //                          ],
        //  "GROUP BY c.wallet_cid, wc.total_sum",
        //          "HAVING wc.total_sum - SUM(c.sum_spent) > ?",
        //          "ORDER BY NULL",
        return dslContextProvider.ppc(shard)
                .select(WALLET_CAMPAIGNS.WALLET_CID)
                .from(CAMPAIGNS)
                .leftJoin(WALLET_CAMPAIGNS).on(WALLET_CAMPAIGNS.WALLET_CID.eq(CAMPAIGNS.WALLET_CID))
                .where(CAMPAIGNS.WALLET_CID.in(walletIds))
                .groupBy(CAMPAIGNS.WALLET_CID, WALLET_CAMPAIGNS.TOTAL_SUM)
                .having(WALLET_CAMPAIGNS.TOTAL_SUM.minus(sum(spentOnCampaigns)).greaterThan(Currencies.EPSILON))
                .fetchSet(WALLET_CAMPAIGNS.WALLET_CID);
    }

    /**
     * Активные кампании РМП
     *
     * @param shard шард
     * @return список активных РМП кампаний
     */
    public List<CampaignForLauncher> getCampaignsForLauncher(int shard) {
        Set<ModelProperty<?, ?>> modelPropertiesWithoutStrategy = new HashSet<>(
                CampaignForLauncher.allModelProperties());
        modelPropertiesWithoutStrategy.remove(CampaignForLauncher.STRATEGY);

        final Condition finishedByDate =
                DSL.localDateDiff(DSL.currentLocalDate(), CAMPAIGNS.FINISH_TIME).gt(0);

        return dslContextProvider.ppc(shard)
                .select(Stream.concat(
                                campaignMapper.getFieldsToRead(modelPropertiesWithoutStrategy).stream(),
                                dbStrategyMapper.getFieldsToRead().stream())
                        .collect(toList()))
                .from(CAMPAIGNS)
                .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .leftJoin(WALLETS).on(CAMPAIGNS.WALLET_CID.eq(WALLETS.CID))
                .where(campTypeIn(singleton(CampaignType.MOBILE_CONTENT)))
                .and(CAMPAIGNS.STRATEGY_NAME.eq(autobudget_avg_cpi))
                .and(CAMPAIGNS.PLATFORM.in(
                        ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform.context,
                        ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform.both))
                .and(CAMPAIGNS.CURRENCY.in(CampaignsCurrency.RUB, CampaignsCurrency.USD, CampaignsCurrency.EUR))
                .and(CAMPAIGNS.ORDER_ID.greaterThan(0L))
                .and(CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.Yes))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(CAMPAIGNS.FINISH_TIME.eq(SqlUtils.mysqlZeroLocalDate()).orNot(finishedByDate))
                // на кампании ещё остались деньги
                .and(campaignBalance(Campaigns.CAMPAIGNS, WALLETS).greaterThan(Currencies.EPSILON))
                .fetch(record -> campaignMapper.fromDb(record).withStrategy(dbStrategyMapper.fromDb(record)));
    }

    private List<Campaign> getCampaignsUnderWalletWithFields(int shard, Long uid, Long walletId,
                                                             Collection<Field<?>> fieldsToRead) {
        Collection<Record> records = dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .where(CAMPAIGNS.UID.eq(uid))
                .and(CAMPAIGNS.WALLET_CID.eq(walletId))
                .fetch();

        return records.stream()
                .map(campaignMapper::fromDb)
                .collect(toList());
    }

    /**
     * Возвращает список кампаний по списку их ID.
     *
     * @param shard       Шард
     * @param campaignIds Список ID кампаний
     */
    public List<Campaign> getCampaigns(int shard, Collection<Long> campaignIds) {
        return getCampaignsWithFieldsStream(shard, campaignIds, campaignWithStrategyMapperFields)
                .toList();
    }

    public Map<Long, Campaign> getCampaignsMap(int shard, Collection<Long> campaignIds) {
        return listToMap(getCampaigns(shard, campaignIds), Campaign::getId);
    }

    public List<Campaign> getCampaignsWithStrategy(int shard, Collection<Long> campaignIds) {
        return getCampaignsWithFieldsStream(shard, campaignIds, campaignWithStrategyMapperFields)
                .toList();
    }

    /**
     * Выборка кампаний по идентификаторам для конкретного clientId
     */
    public List<CampaignSimple> getCampaignsForClient(int shard, ClientId clientId, Collection<Long> campaignIds) {
        var selectionCriteria = new CampaignsSelectionCriteria()
                .withCampaignIds(campaignIds)
                .withClientId(clientId);
        return getCampaignsSimpleStream(dslContextProvider.ppc(shard), selectionCriteria).toList();
    }

    /**
     * Выборка кампаний по id опроса взгляда
     */
    public Map<String, List<Campaign>> getCampaignsForBrandSurveys(int shard, ClientId clientId,
                                                                   List<String> brandSurveyIds) {
        return dslContextProvider.ppc(shard)
                .select(List.of(CAMPAIGNS.NAME, CAMPAIGNS.CID, CAMP_OPTIONS.BRAND_SURVEY_ID,
                        CAMPAIGNS.START_TIME, CAMPAIGNS.FINISH_TIME, CAMP_OPTIONS.CREATE_TIME, CAMPAIGNS.ORDER_ID,
                        CAMPAIGNS.STRATEGY_DATA, CAMPAIGNS.STRATEGY_NAME))
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMP_OPTIONS.BRAND_SURVEY_ID.in(brandSurveyIds))
                .fetchGroups(r -> r.get(CAMP_OPTIONS.BRAND_SURVEY_ID), r -> campaignWithStrategyFromDb(r));
    }

    public Map<String, List<Campaign>> getCampaignsForBrandSurveys(int shard, Collection<String> brandSurveyIds) {
        return dslContextProvider.ppc(shard)
                .select(List.of(CAMPAIGNS.NAME, CAMPAIGNS.CID, CAMP_OPTIONS.BRAND_SURVEY_ID,
                        CAMPAIGNS.START_TIME, CAMPAIGNS.FINISH_TIME, CAMP_OPTIONS.CREATE_TIME, CAMPAIGNS.ORDER_ID,
                        CAMPAIGNS.STRATEGY_DATA, CAMPAIGNS.STRATEGY_NAME))
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                .and(CAMP_OPTIONS.BRAND_SURVEY_ID.in(brandSurveyIds))
                .fetchGroups(r -> r.get(CAMP_OPTIONS.BRAND_SURVEY_ID), this::campaignWithStrategyFromDb);
    }

    private StreamEx<Campaign> getCampaignsWithFieldsStream(int shard, Collection<Long> campaignIds,
                                                            Collection<Field<?>> fields) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withCampaignIds(campaignIds);
        return getCampaignsWithFieldsStream(dslContextProvider.ppc(shard), selectionCriteria, fields, false);
    }

    private StreamEx<Campaign> getCampaignsWithFieldsStream(DSLContext dslContext,
                                                            CampaignsSelectionCriteria criteria,
                                                            Collection<Field<?>> fields,
                                                            boolean forUpdate) {
        ConditionsAccumulator accumulator = new ConditionsAccumulator();

        accumulator.add(criteria::getCampaignIds, Function.identity(), CAMPAIGNS.CID::in);
        accumulator.add(criteria::getOrderIds, Function.identity(), CAMPAIGNS.ORDER_ID::in);
        accumulator.add(criteria::getClientIds, ids -> listToSet(ids, ClientId::asLong), CAMPAIGNS.CLIENT_ID::in);

        // Условия ниже - не по ключевым колонкам, по ним нельзя делать выборку, но удобно делать фильтрацию
        checkState(accumulator.containsConditions(), "empty CampaignsSelectionCriteria");

        // Не добавляйте отрицательные условия на вхождение (notIn)
        // оптимизация с containsEmptyCondition даст некорректный результат
        accumulator.add(criteria::getCampaignTypes, t -> listToSet(t, CampaignType::toSource), CAMPAIGNS.TYPE::in);
        accumulator.add(criteria::getStatusEmpty, CampaignMappings::statusEmptyToDb, CAMPAIGNS.STATUS_EMPTY::eq);
        accumulator.add(criteria::getStatusArchived, CampaignMappings::archivedToDb, CAMPAIGNS.ARCHIVED::eq);
        accumulator.add(criteria::getStatusActive, CampaignMappings::statusActiveToDb, CAMPAIGNS.STATUS_ACTIVE::eq);
        accumulator.add(criteria::getSubCampaigns, Function.identity(), field(SUBCAMPAIGNS.MASTER_CID.isNotNull())::eq);

        if (accumulator.containsEmptyCondition()) {
            // не делаем запрос к базе, если хотя бы одна из колонок фильтруется пустым списком
            return StreamEx.empty();
        }
        //в accumulator-условии нет полей из CAMP_OPTIONS
        boolean joinCampOptions = Stream.of(CAMP_OPTIONS.fields()).anyMatch(fields::contains);
        boolean joinSubCampaigns = criteria.getSubCampaigns() != null;

        SelectJoinStep<Record> selectJoinStep = dslContext
                .select(fields)
                .from(CAMPAIGNS);
        if (joinCampOptions) {
            selectJoinStep = selectJoinStep.join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID));
        }
        if (joinSubCampaigns) {
            selectJoinStep = selectJoinStep.leftJoin(SUBCAMPAIGNS).on(CAMPAIGNS.CID.eq(SUBCAMPAIGNS.CID));
        }
        var selectStep = selectJoinStep
                .where(accumulator.getConditions());

        Collection<Record> records;

        if (forUpdate) {
            records = selectStep.forUpdate().fetch();
        } else {
            records = selectStep.fetch();
        }

        Function<Record, Campaign> recordMapper;
        if (dbStrategyMapper.canReadAtLeastOneProperty(fields)) {
            recordMapper = this::campaignWithStrategyFromDb;
        } else {
            recordMapper = campaignMapper::fromDb;
        }

        return StreamEx.of(records)
                .map(recordMapper);
    }

    private Campaign campaignWithStrategyFromDb(Record record) {
        Campaign campaign = campaignMapper.fromDb(record);
        DbStrategy dbStrategy = dbStrategyMapper.fromDb(record);
        return campaign.withStrategy(dbStrategy);
    }

    /**
     * Получение Map order_id->cid для списка orderIds
     */
    public Map<Long, Long> getCidsForOrderIds(int shard, List<Long> orderIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withOrderIds(orderIds);
        return getCampaignsSimpleStream(shard, selectionCriteria)
                .toMap(CampaignSimple::getOrderId, CampaignSimple::getId);
    }

    public Map<Long, CampaignSimple> getCampaignsForOrderIds(int shard, List<Long> orderIds) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withOrderIds(orderIds);
        return getCampaignsSimpleStream(shard, selectionCriteria)
                .toMap(CampaignSimple::getOrderId, c -> c);
    }

    public Collection<Long> getNonArchivedCampaignIdsWhichWereInBs(int shard, ClientId clientId) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignTypes(CampaignTypeKinds.BS_EXPORT)
                .withStatusArchived(false);

        return getCampaignsSimpleStream(shard, selectionCriteria)
                .filter(c -> !ID_NOT_SET.equals(c.getOrderId()))
                .map(CampaignSimple::getId)
                .toList();
    }

    /**
     * Получение Map cid->product_id
     */
    public Map<Long, Long> getProductIds(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.PRODUCT_ID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS.CID, CAMPAIGNS.PRODUCT_ID);
    }

    public Optional<CampaignsSource> getLastCampaignSource(int shard, Long clientId, CampaignsType campaignsType) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.SOURCE)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.TYPE.eq(campaignsType))
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId))
                .orderBy(CAMPAIGNS.CID.desc())
                .limit(1)
                .fetchOptional(CAMPAIGNS.SOURCE);
    }

    public void setStatusAutobudgetForecast(int shard, Collection<Long> campaignIds, StatusAutobudgetForecast status) {
        dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_AUTOBUDGET_FORECAST, StatusAutobudgetForecast.toSource(status))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    public void setStatusModerateIfNewOrNo(DSLContext context, Collection<Long> campaignIds,
                                           CampaignStatusModerate status) {
        if (campaignIds.isEmpty()) {
            return;
        }

        context
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_MODERATE, CampaignStatusModerate.toSource(status))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.STATUS_MODERATE.in(CampaignsStatusmoderate.No, CampaignsStatusmoderate.New))
                .execute();
    }

    /**
     * Проставить кампаниям статусы модерации statusModerate = "Yes", statusPostModerate = "Accepted".
     */
    public void markCampaignsAsModerated(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }
        DSLContext dslContext = dslContextProvider.ppc(shard);
        dslContext.update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_MODERATE, CampaignsStatusmoderate.Yes)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
        dslContext.update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.STATUS_POST_MODERATE, CampOptionsStatuspostmoderate.Accepted)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Запланировать перерасчёт прогноза для нескольких кампаний
     */
    public void setAutobudgetForecastDate(int shard, Collection<Long> campaignIds,
                                          @Nullable LocalDateTime forecastDate) {
        setAutobudgetForecastDate(dslContextProvider.ppc(shard).configuration(), campaignIds, forecastDate);
    }

    public void setAutobudgetForecastDate(Configuration conf, Collection<Long> campaignIds,
                                          @Nullable LocalDateTime forecastDate) {
        if (campaignIds.isEmpty()) {
            return;
        }

        conf.dsl()
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.AUTOBUDGET_FORECAST_DATE, forecastDate)
                .set(CAMPAIGNS.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    private Condition campTypeIn(Collection<CampaignType> types) {
        return CAMPAIGNS.TYPE.in(mapList(types, CampaignType::toSource));
    }

    private Condition sourceNotIn(Collection<CampaignSource> types) {
        return CAMPAIGNS.SOURCE.notIn(mapList(types, CampaignSource::toSource));
    }

    private Condition sourceIn(Collection<CampaignSource> types) {
        return CAMPAIGNS.SOURCE.in(mapList(types, CampaignSource::toSource));
    }

    public void updateCampaigns(int shard, Collection<AppliedChanges<Campaign>> appliedChanges) {
        updateCampaigns(dslContextProvider.ppc(shard), appliedChanges);
    }

    /**
     * Проставляет флаг StatusEmpty в Yes. Нужно для удаления кампаний
     *
     * @param shard       шард
     * @param campaignIds ids кампаний
     */
    public void setStatusEmptyToYes(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_EMPTY, CampaignsStatusempty.Yes)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Обновляет поле statusMail. Нужно для того, чтобы избежать повторных уведомлений
     *
     * @param shard       шард
     * @param campaignIds ids кампаний
     * @param statusMail  новый статус
     */
    public void setStatusMail(int shard, Collection<Long> campaignIds, Long statusMail, Long oldStatusMail) {
        if (campaignIds.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_MAIL, statusMail)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.STATUS_MAIL.eq(oldStatusMail))
                .execute();
    }

    /**
     * Метод для обновления минус-фраз по кампаниям:
     * обновляются не все поля, см. реализацию.
     * Обновляется в транзакции, используются 2 таблицы: CAMPAIGNS, CAMP_OPTIONS
     *
     * @param dslContext     контекст транзакции
     * @param appliedChanges применяемые изменения
     */
    public void updateCampaigns(DSLContext dslContext, Collection<AppliedChanges<Campaign>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return;
        }
        updateCampaignTable(dslContext, appliedChanges);
        updateCampaignOptionsTable(dslContext, appliedChanges);
    }

    private void updateCampaignTable(DSLContext dslContext, Collection<AppliedChanges<Campaign>> appliedChanges) {
        JooqUpdateBuilder<CampaignsRecord, Campaign> updateBuilder =
                new JooqUpdateBuilder<>(CAMPAIGNS.CID, appliedChanges);

        updateBuilder.processProperty(Campaign.SUM, CAMPAIGNS.SUM);
        updateBuilder.processProperty(Campaign.SOURCE, CAMPAIGNS.SOURCE, CampaignSource::toSource);
        updateBuilder.processProperty(Campaign.SUM_BALANCE, CAMPAIGNS.SUM_BALANCE);
        updateBuilder.processProperty(Campaign.NAME, CAMPAIGNS.NAME);
        updateBuilder.processProperty(Campaign.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE, localDateTime1 -> localDateTime1);
        updateBuilder.processProperty(Campaign.START_TIME, CAMPAIGNS.START_TIME);
        updateBuilder.processProperty(Campaign.FINISH_TIME, CAMPAIGNS.FINISH_TIME);
        updateBuilder.processProperty(Campaign.GEO, CAMPAIGNS.GEO, CampaignMappings::geoToDb);
        updateBuilder.processProperty(Campaign.STATUS_BS_SYNCED, CAMPAIGNS.STATUS_BS_SYNCED,
                CampaignMappings::statusBsSyncedToDb);
        updateBuilder.processProperty(Campaign.STATUS_SHOW, CAMPAIGNS.STATUS_SHOW, CampaignMappings::statusShowToDb);
        updateBuilder.processProperty(Campaign.AUTOBUDGET_FORECAST_DATE, CAMPAIGNS.AUTOBUDGET_FORECAST_DATE,
                localDateTime -> localDateTime);
        updateBuilder.processProperty(Campaign.DISABLED_DOMAINS, CAMPAIGNS.DONT_SHOW,
                CampaignMappings::disabledDomainsToDb);
        updateBuilder.processProperty(Campaign.DISABLED_VIDEO_PLACEMENTS, CAMPAIGNS.DISABLED_VIDEO_PLACEMENTS,
                CampaignMappings::disabledVideoPlacementsToJson);
        updateBuilder.processProperty(Campaign.OPTS, CAMPAIGNS.OPTS, CampaignMappings::optsToDb);
        updateBuilder.processProperty(Campaign.STATUS_ARCHIVED, CAMPAIGNS.ARCHIVED, CampaignMappings::archivedToDb);

        dslContext
                .update(CAMPAIGNS)
                .set(updateBuilder.getValues())
                .where(CAMPAIGNS.CID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    private void updateCampaignOptionsTable(DSLContext dslContext,
                                            Collection<AppliedChanges<Campaign>> appliedChanges) {
        JooqUpdateBuilder<CampOptionsRecord, Campaign> updateBuilder =
                new JooqUpdateBuilder<>(CAMP_OPTIONS.CID, appliedChanges);

        updateBuilder.processProperty(Campaign.MINUS_KEYWORDS, CAMP_OPTIONS.MINUS_WORDS, JsonUtils::toJsonNullable);
        updateBuilder.processProperty(Campaign.MINUS_KEYWORDS_ID, CAMP_OPTIONS.MW_ID);
        updateBuilder.processProperty(Campaign.BROAD_MATCH_FLAG, CAMP_OPTIONS.BROAD_MATCH_FLAG,
                CampaignMappings::broadMatchFlagToDb);
        updateBuilder.processProperty(Campaign.BROAD_MATCH_LIMIT, CAMP_OPTIONS.BROAD_MATCH_LIMIT);

        dslContext
                .update(CAMP_OPTIONS)
                .set(updateBuilder.getValues())
                .where(CAMP_OPTIONS.CID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    public Map<Long, List<String>> getMinusKeywordsByCampaignIds(int shard, List<Long> ids) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withCampaignIds(ids);
        return getCampaignsSimpleStream(shard, selectionCriteria)
                .filter(c -> c.getMinusKeywords() != null)
                .toMap(CampaignSimple::getId, CampaignSimple::getMinusKeywords);
    }

    public Map<Long, Set<String>> getDisabledDomainsByCampaignIds(DSLContext dslContext, List<Long> ids,
                                                                  boolean forUpdate) {

        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withCampaignIds(ids);
        return getCampaignsWithFieldsStream(dslContext, selectionCriteria, DISABLED_DOMAINS_TO_ID_MAP_FIELDS, forUpdate)
                .filter(c -> c.getDisabledDomains() != null)
                .toMap(Campaign::getId, Campaign::getDisabledDomains);
    }

    /**
     * Получить даты создания кампаний по их номерам заказов
     *
     * @param shard шард
     * @param ids   список OrderId
     * @return словарь (OrderId, дата начала)
     */
    public Map<Long, LocalDate> getCreateDatesByOrderIds(int shard, Collection<Long> ids) {

        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria().withOrderIds(ids);
        return getCampaignsWithFieldsStream(dslContextProvider.ppc(shard), selectionCriteria,
                Arrays.asList(CAMPAIGNS.CID, CAMPAIGNS.ORDER_ID, CAMP_OPTIONS.CREATE_TIME), false)
                .toMap(Campaign::getOrderId, this::getCampaignCreateDate);
    }

    private LocalDate getCampaignCreateDate(Campaign campaign) {
        return campaign.getCreateTime().toLocalDate();
    }

    public Map<Long, Set<Integer>> getGeoByCampaignIds(int shard, Collection<Long> ids) {
        return getCampaignsWithFieldsStream(shard, ids, GEO_TO_ID_MAP_FIELDS)
                .filter(c -> c.getGeo() != null)
                .toMap(Campaign::getId, Campaign::getGeo);
    }

    /**
     * Получить список кампаний, в которых используются заданные креативы
     * Кампании не принадлежащие клиенту и архивные кампании из запроса пропускаются
     *
     * @param clientId    клиент по которому фильтруем кампании
     * @param creativeIds список креативов
     * @return отображение creativeId -> List<CampaignName> campaignsName
     */
    public Map<Long, List<CampaignName>> getCampaignsByCreativeIds(int shard, ClientId clientId,
                                                                   List<Long> creativeIds) {
        Result<Record> result = dslContextProvider.ppc(shard)
                .selectDistinct(asList(BANNERS_PERFORMANCE.CREATIVE_ID, CAMPAIGNS.CID, CAMPAIGNS.NAME))
                .from(BANNERS_PERFORMANCE)
                .join(CAMPAIGNS)
                .on(CAMPAIGNS.CID.eq(BANNERS_PERFORMANCE.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(BANNERS_PERFORMANCE.CREATIVE_ID.in(creativeIds))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .fetch();

        Map<Long, List<Record>> creativeCampaignNames = StreamEx.of(result)
                .groupingBy(rec -> rec.getValue(BANNERS_PERFORMANCE.CREATIVE_ID));
        return EntryStream.of(creativeCampaignNames)
                .mapValues(list -> mapList(list, rec -> (CampaignName) campaignMapper.fromDb(rec)))
                .toMap();
    }

    /**
     * Получить список кампаний
     * Кампании не принадлежащие клиенту и архивные кампании из запроса пропускаются
     *
     * @param clientId    клиент по которому фильтруем кампании
     * @param campaignIds список id кампаний
     */
    public List<Campaign> getClientsCampaignsByIds(int shard, ClientId clientId, Collection<Long> campaignIds) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignIds(campaignIds)
                .withStatusArchived(false);
        return getCampaignsWithFieldsStream(dslContext, selectionCriteria, campaignWithStrategyMapperFields, false)
                .toList();
    }

    //TODO: DIRECT-106340 : нужно удалить эту логику, после раскатки тирной схемы на 100%.
    // Менеджер должен будет определяться только по главному менеджеру, а эти эвристики станут устаревшими

    /**
     * Извлекает менеджеров клиентов из кампаний. Берётся новейшая кампания, т.к. в ней больше шансов обнаружить
     * актуального менеджера.
     * Сделано по мотивам перлового кода:
     * https://a.yandex-team.ru/arc/trunk/arcadia//direct/perl/protected/SearchObjects.pm?blame=true&rev=5758093#L168
     */
    public Map<Long, Long> getClientManagerUidByClientId(int shard, Collection<Long> clientIds) {
        Field<Long> maxCid = field("maxCid", Long.class);
        SelectHavingStep<Record1<Long>> nested = dslContextProvider.ppc(shard)
                .select(max(CAMPAIGNS.CID).as(maxCid))
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.in(clientIds))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.TYPE.in(USE_IN_GET_CLIENT_MANAGER))
                .and(CAMPAIGNS.MANAGER_UID.isNotNull())
                .groupBy(CAMPAIGNS.CLIENT_ID);

        Map<Long, List<Long>> managersByClientId = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CLIENT_ID, CAMPAIGNS.MANAGER_UID)
                .from(CAMPAIGNS)
                .join(nested).on(maxCid.eq(CAMPAIGNS.CID))
                .fetchGroups(CAMPAIGNS.CLIENT_ID, CAMPAIGNS.MANAGER_UID);

        return EntryStream.of(managersByClientId)
                .mapValues(l -> l.get(0))
                .toMap();
    }

    //TODO: DIRECT-106340 : нужно удалить эту логику, после раскатки тирной схемы на 100%.
    // Менеджер должен будет определяться только по главному менеджеру, а эти эвристики станут устаревшими

    /**
     * Получает список всех менеджеров клиентов из кампаний. Учитываются все кампании клиента - архивные тоже
     * Сделано по мотивам перлового кода:
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Client.pm#L3881&rev=5758093
     */
    public Map<Long, Set<Long>> getClientManagersUidByClientId(int shard, Collection<Long> clientIds) {
        Stream<Record2<Long, Long>> stream = dslContextProvider.ppc(shard)
                .selectDistinct(CAMPAIGNS.CLIENT_ID, CAMPAIGNS.MANAGER_UID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.in(clientIds))
                .and(CAMPAIGNS.MANAGER_UID.isNotNull())
                .fetchStream();

        return StreamEx.of(stream)
                .mapToEntry(r -> r.get(CAMPAIGNS.CLIENT_ID), r -> r.get(CAMPAIGNS.MANAGER_UID))
                .grouping(Collectors.toSet());
    }

    /**
     * Получить сумму для кампаний
     * Если у клиента подключен общий счет, то к сумме кампаний прибавляется сумма кошелька:
     * {@link #CAMPAIGN_AND_WALLET_SUM}
     *
     * @param shard       шард
     * @param clientId    клиент по которому фильтруем кампании
     * @param campaignIds список id кампаний
     */
    public Map<Long, BigDecimal> getCampaignsSum(int shard, ClientId clientId, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGN_AND_WALLET_SUM)
                .from(CAMPAIGNS)
                .leftJoin(WALLETS).on(CAMPAIGNS.WALLET_CID.eq(WALLETS.CID))
                .where(CAMPAIGNS.CID.in(campaignIds)
                        .and((CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))))
                .fetchMap(CAMPAIGNS.CID, CAMPAIGN_AND_WALLET_SUM);
    }

    /**
     * Получить сумму для кампаний
     * Если у клиента подключен общий счет, то к сумме кампаний прибавляется сумма кошелька:
     * {@link #CAMPAIGN_AND_WALLET_SUM}
     *
     * @param shard       шард
     * @param campaignIds список id кампаний
     */
    public Map<Long, BigDecimal> getCampaignsSum(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGN_AND_WALLET_SUM)
                .from(CAMPAIGNS)
                .leftJoin(WALLETS).on(CAMPAIGNS.WALLET_CID.eq(WALLETS.CID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS.CID, CAMPAIGN_AND_WALLET_SUM);
    }

    /**
     * Метод для получения id кампаний, у которых время остановки дневного бюджета раньше заданного
     *
     * @param shard          шард
     * @param borderStopTime граничное время остановки дневного бюджета
     * @return id кампаний, у которых ppc.camp_options.day_budget_stop_time != 0000-00-00 00:00:00 и меньше
     * borderStopTime
     */
    @QueryWithoutIndex("Выборка всего подряд по времени, выполняется в jobs")
    public Set<Long> getCampaignIdsWhichDayBudgetStopTimeLessThan(int shard, LocalDateTime borderStopTime) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME.ne(SqlUtils.mysqlZeroLocalDateTime()))
                .and(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME.lessThan(borderStopTime))
                .fetchSet(CAMP_OPTIONS.CID);
    }

    /**
     * Метод для получения id кампаний, у которых был изменен дневной бюджет
     *
     * @param shard шард
     * @return id кампаний, у которых ppc.camp_options.day_budget_daily_change_count > 0
     */
    @QueryWithoutIndex("Выборка всего подряд по счетчику, выполняется в jobs")
    public Set<Long> getCampaignIdsWithDayBudgetDailyChanges(int shard) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT.greaterThan(DEFAULT_DAY_BUDGET_CHANGE_COUNT))
                .fetchSet(CAMP_OPTIONS.CID);
    }

    /**
     * Сброс времени остановки дневного бюджета и статуса нотификации кампаний,
     * у которых время остановки дневного бюджета раньше заданного
     *
     * @param shard             шард
     * @param campaignIds       список id-кампаний
     * @param dayBudgetStopTime граничное время остановки дневного бюджета
     */
    public void resetDayBudgetStopTimeAndNotificationStatus(int shard, Collection<Long> campaignIds,
                                                            LocalDateTime dayBudgetStopTime) {
        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME, SqlUtils.mysqlZeroLocalDateTime())
                .set(CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS,
                        ru.yandex.direct.dbschema.ppc.enums.CampOptionsDayBudgetNotificationStatus.Ready)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .and(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME.lessThan(dayBudgetStopTime))
                .execute();
    }

    /**
     * Сброс количество изменений дневного бюджета
     *
     * @param shard       шард
     * @param campaignIds список id-кампаний, для которых нужно сбросить количество изменений дневного бюджета
     */
    public void resetDayBudgetDailyChangeCount(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT, DEFAULT_DAY_BUDGET_CHANGE_COUNT)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Обновления параметров дневного бюджета кампании
     *
     * @param shard            шард
     * @param dayBudgetOptions параметры дневного бюджета, на которые нужно обновить
     */
    public void updateDayBudgetOptions(int shard, CampaignDayBudgetOptions dayBudgetOptions) {
        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME,
                        RepositoryUtils.zeroableDateTimeToDb(dayBudgetOptions.getDayBudgetStopTime()))
                .set(CAMP_OPTIONS.DAY_BUDGET_LAST_CHANGE,
                        RepositoryUtils.zeroableDateTimeToDb(LocalDateTime.now()))
                .set(CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT, dayBudgetOptions.getDayBudgetDailyChangeCount())
                .set(CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS, DayBudgetNotificationStatus.toSource(
                        dayBudgetOptions.getDayBudgetNotificationStatus()))
                .where(CAMP_OPTIONS.CID.eq(dayBudgetOptions.getId()))
                .execute();
    }

    /**
     * Получить из базы список кампаний с набором параметров, нужных для проверки на заблокированные средства
     *
     * @param shard     шард
     * @param userIds   список идентификаторов пользователей, которым принадлежат кампании
     * @param walletIds список идентификаторов кошельков, под которыми находятся кампании
     */
    public List<CampaignForBlockedMoneyCheck> getCampaignsUnderWalletsForBlockedMoneyCheck(int shard,
                                                                                           Collection<Long> userIds,
                                                                                           Collection<Long> walletIds) {
        if (userIds.isEmpty() || walletIds.isEmpty()) {
            return Collections.emptyList();
        }

        Set<Field<?>> fieldsToRead = campaignMapper.getFieldsToRead(CampaignForBlockedMoneyCheck.allModelProperties());
        fieldsToRead.remove(CAMPAIGNS.SUM);
        fieldsToRead.add(CAMPAIGNS.SUM
                .plus(JooqMapperUtils.mysqlIf(WALLETS.CID, WALLETS.SUM, DSL.val(BigDecimal.ZERO)))
                .as(CAMPAIGNS.SUM));
        fieldsToRead.remove(CAMPAIGNS.SUM_TO_PAY);
        fieldsToRead.add(CAMPAIGNS.SUM_TO_PAY
                .plus(JooqMapperUtils.mysqlIf(WALLETS.CID, WALLETS.SUM_TO_PAY, DSL.val(BigDecimal.ZERO)))
                .as(CAMPAIGNS.SUM_TO_PAY));

        return dslContextProvider.ppc(shard).
                select(fieldsToRead)
                .from(CAMPAIGNS
                        .leftJoin(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                        .leftJoin(WALLETS).on(CAMPAIGNS.WALLET_CID.eq(WALLETS.CID)))
                .where(CAMPAIGNS.UID.in(userIds)
                        .and(CAMPAIGNS.WALLET_CID.in(walletIds))
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .fetch(r -> campaignMapper.fromDb(r)
                        .withSum(r.get(CAMPAIGNS.SUM))
                        .withSumToPay(r.get(CAMPAIGNS.SUM_TO_PAY)));
    }

    /**
     * Получение id всех кампаний под указанным общим счетом
     *
     * @param shard     шард (подразумеваем, что все дочерние кампании находятся в одном шарде с кошельком)
     * @param walletCid id кампании кошелька
     * @return id кампаний у которых campaigns.wallet_cid = walletCid
     */
    public List<Long> getCampaignIdsUnderWallet(int shard, Long walletCid) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(WALLETS
                        .join(CAMPAIGNS).on(CAMPAIGNS.WALLET_CID.eq(WALLETS.CID)
                                .and(CAMPAIGNS.UID.eq(WALLETS.UID))))
                .where(WALLETS.CID.eq(walletCid)
                        .and(WALLETS.TYPE.eq(CampaignsType.wallet)))
                .fetch(CAMPAIGNS.CID);
    }

    /**
     * Сбросить статус синхронизации с БК в "No" для заданных кампаний.
     * <p>
     * WARNING: Сброс статуса означает, что кампания сразу берется в работу транспортом в БК.
     * Если вам захотелось использовать его на более чем ста кампаниях, скорее всего стоит воспользоваться ленивой
     * переотправкой в БК
     *
     * @param shard       шард
     * @param campaignIds список идентификаторов кампаний
     * @return количество измененных строк
     */
    public int resetBannerSystemSyncStatus(int shard, Collection<Long> campaignIds) {
        return resetBannerSystemSyncStatus(dslContextProvider.ppc(shard).configuration(), campaignIds);
    }

    /**
     * Сбросить статус синхронизации с БК в "No" для заданных кампаний.
     * <p>
     * WARNING: Сброс статуса означает, что кампания сразу берется в работу транспортом в БК.
     * Если вам захотелось использовать его на более чем ста кампаниях, скорее всего стоит воспользоваться ленивой
     * переотправкой в БК
     *
     * @param campaignIds список идентификаторов кампаний
     * @return количество измененных строк
     */
    public int resetBannerSystemSyncStatus(Configuration conf, Collection<Long> campaignIds) {
        return conf.dsl()
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Обновить время последнего пополнения кампании, now() вычисляется на стороне БД
     *
     * @param shard      шард
     * @param campaignId идентификатор кампании
     * @return количество измененных строк
     */
    public int setCampOptionsLastPayTimeNow(int shard, Long campaignId) {
        return dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.LAST_PAY_TIME, DSL.currentLocalDateTime())
                .where(CAMP_OPTIONS.CID.eq(campaignId))
                .execute();
    }

    /**
     * Устанавливает заданного менеджера сервисирующим менеджером во все кампании клиента, кроме агентских кампаний.
     * Эта бизнес-логика реализована в репозитории потому, что у клиента может быть под сотню тысяч кампаний
     * и фильтровать их на стороне приложения получается слишком затратно по ресурсам.
     * Можно ли будет задавать сервисирующего менеджера во все кампании без исключений решится в этом тикете:
     * https://st.yandex-team.ru/DIRECT-100928
     */
    public void setManagerForAllClientCampaigns(int shard, ClientId clientId, @Nullable Long managerUid) {
        dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.MANAGER_UID, managerUid)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.AGENCY_UID.isNull())
                .execute();
    }

    public Map<Long, DbStrategy> getStrategyByFilterIds(int shard, Collection<Long> filterIds) {
        if (isEmpty(filterIds)) {
            return emptyMap();
        }
        Set<Field<?>> fieldsToRead = new HashSet<>(dbStrategyMapper.getFieldsToRead());
        fieldsToRead.add(BIDS_PERFORMANCE.PERF_FILTER_ID);
        Map<Long, List<DbStrategy>> strategiesByFilterIds = dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(BIDS_PERFORMANCE)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(BIDS_PERFORMANCE.PERF_FILTER_ID.in(filterIds))
                .fetchGroups(BIDS_PERFORMANCE.PERF_FILTER_ID, dbStrategyMapper::fromDb);
        // по бизнес-логике стратегия выше в иерархии чем фильтр, поэтому всегда одна на фильтр
        return EntryStream.of(strategiesByFilterIds)
                .mapValues(l -> l.get(0))
                .toMap();
    }

    public void sendRejectedCampaignsToModerate(int shard, Collection<Long> campIds) {
        sendRejectedCampaignsToModerate(dslContextProvider.ppc(shard), campIds);
    }

    public void markReadyCampaignsAsSent(Configuration configuration, Collection<Long> campIds) {
        if (campIds.isEmpty()) {
            return;
        }

        configuration
                .dsl()
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_MODERATE, CampaignStatusModerate.toSource(CampaignStatusModerate.SENT))
                .set(CAMPAIGNS.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE)
                .where(CAMPAIGNS.CID.in(campIds))
                // Для внутренней рекламы и performance-кампаний statusModerate не трогаем; ожидается, что после создания кампания сразу
                // отправляется на модерацию и переводится в statusModerate=Yes навсегда
                .and(CAMPAIGNS.STATUS_MODERATE.eq(CampaignStatusModerate.toSource(CampaignStatusModerate.READY)))
                .andNot(CAMPAIGNS.TYPE.in(MOD_EXPORT_ONLY_TYPES))
                .execute();
    }

    /**
     * Переотправка отклоненных кампаний на модерацию.
     * Если у переданных кампаний статус модерации No, то проставляет статус Ready.
     *
     * @param dslContext DSLContext
     * @param campIds    список id кампаний, которые хотим отправить на модерацию
     */
    public void sendRejectedCampaignsToModerate(DSLContext dslContext, Collection<Long> campIds) {
        if (campIds.isEmpty()) {
            return;
        }
        dslContext
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_MODERATE, CampaignStatusModerate.toSource(CampaignStatusModerate.READY))
                .set(CAMPAIGNS.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE)
                .where(CAMPAIGNS.CID.in(campIds))
                .and(CAMPAIGNS.STATUS_MODERATE.eq(CampaignStatusModerate.toSource(CampaignStatusModerate.NO)))
                .execute();
    }

    /**
     * Получает CampaignWithType для указанных кампаний (с проверкой доступа клиента к кампаниям).
     *
     * @param shard       шард
     * @param clientId    id клиента
     * @param campaignIds id кампаний
     * @return отображение campaignId -> CampaignWithType
     */
    public Map<Long, CampaignWithType> getCampaignsWithTypeByCampaignIds(int shard, ClientId clientId,
                                                                         Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withClientId(clientId)
                .withCampaignIds(campaignIds);

        return getCampaignsSimpleStream(shard, selectionCriteria)
                .toMap(CampaignSimple::getId, CampaignRepository::campaignSimpleToCampaignWithType);
    }

    /**
     * @return отображение adGroupId -> CampaignWithType
     */
    public Map<Long, CampaignWithType> getCampaignsWithTypeByAdGroupIds(int shard, ClientId clientId,
                                                                        Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(PHRASES.PID, CAMPAIGNS.CID, CAMPAIGNS.TYPE, CAMPAIGNS.ARCHIVED)
                .from(CAMPAIGNS)
                .join(PHRASES).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(PHRASES.PID.in(adGroupIds)
                        .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetchMap(PHRASES.PID, CampaignRepository::campaignWithTypeMapper);
    }


    /**
     * @return отображение bannerId -> CampaignWithType
     */
    public Map<Long, CampaignWithType> getCampaignsWithTypeByBannerIds(int shard, ClientId clientId,
                                                                       Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, CAMPAIGNS.CID, CAMPAIGNS.TYPE, CAMPAIGNS.ARCHIVED)
                .from(CAMPAIGNS)
                .join(BANNERS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(BANNERS.BID.in(bannerIds)
                        .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetchMap(BANNERS.BID, CampaignRepository::campaignWithTypeMapper);
    }

    /**
     * Получает CampaignForForecast для указанных кампаний.
     *
     * @param shard       шард
     * @param clientId    идентификатор клиента
     * @param campaignIds id кампаний
     * @return отображение campaignId -> CampaignForForecast
     */
    public Map<Long, CampaignForForecast> getCampaignsForForecastByCampaignIds(int shard,
                                                                               @Nullable ClientId clientId,
                                                                               Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        DSLContext dslContext = dslContextProvider.ppc(shard);
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withCampaignIds(campaignIds);
        if (clientId != null) {
            selectionCriteria.withClientId(clientId);
        }
        return getCampaignsWithFieldsStream(dslContext, selectionCriteria, campaignForForecastMapperFields, false)
                .toMap(Campaign::getId, Function.identity());
    }

    /**
     * Устанавливает статус синхронизации с БК в указанный для заданной кампании.
     * Нужно передать текущий lastChange кампании для того, чтобы оно не было изменено апдейтом.
     */
    public void setCampaignBsSynced(int shard, long campaignId, StatusBsSynced statusBsSynced,
                                    LocalDateTime lastChange) {
        ModelChanges<Campaign> changes = new ModelChanges<>(campaignId, Campaign.class);
        // Обновляем STATUS_BS_SYNCED, а LAST_CHANGE принудительно присваиваем текущее значение
        // чтобы оно не обновилось автоматически из-за ON UPDATE CURRENT_TIMESTAMP
        changes.process(statusBsSynced, Campaign.STATUS_BS_SYNCED);
        changes.process(lastChange, Campaign.LAST_CHANGE);
        AppliedChanges<Campaign> appliedChanges = changes.applyTo(new Campaign().withId(campaignId));
        updateCampaigns(shard, singleton(appliedChanges));
    }

    /**
     * Обновить statusBsSynced у переданных кампаний
     *
     * @param shard       шард
     * @param campaignIds список id кампаний, которые хотим отправить на модерацию
     * @param status      новый статус
     * @return - количество изменённых строк
     */
    public int updateStatusBsSynced(int shard, Collection<Long> campaignIds, StatusBsSynced status) {
        if (campaignIds.isEmpty()) {
            return 0;
        }

        return updateStatusBsSynced(dslContextProvider.ppc(shard), campaignIds, status);
    }

    /**
     * Обновить statusBsSynced у переданных кампаний
     *
     * @param dslContext  контекст транзакции
     * @param campaignIds список id кампаний, которые хотим отправить на модерацию
     * @param status      новый статус
     * @return - количество изменённых строк
     */
    public int updateStatusBsSynced(DSLContext dslContext, Collection<Long> campaignIds, StatusBsSynced status) {
        if (campaignIds.isEmpty()) {
            return 0;
        }

        return dslContext
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, statusBsSyncedToDb(status))
                .set(CAMPAIGNS.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Обновить statusBsSynced и lastChange у переданных кампаний
     *
     * @param conf        контекст транзакции
     * @param campaignIds список id кампаний
     * @param status      новый статус синхронизации
     */
    public void updateStatusBsSyncedWithLastChange(Configuration conf, Collection<Long> campaignIds,
                                                   StatusBsSynced status) {
        if (campaignIds.isEmpty()) {
            return;
        }

        DSL.using(conf)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, statusBsSyncedToDb(status))
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now())
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Получить начальное значение {@code campaignId} для обработки ре-экспортом.
     * <p>
     * Логика получения (обработка от большего cid к меньшему)
     * аналогична {@link #getCampaignIdsForFullExport(int, long, int)}
     *
     * @param shard шард
     * @return увеличенное на 1 значение {@code campaignId}, или {@link SqlUtils#ID_NOT_SET} если подходящих записей нет
     */
    public Long getNewLastProcessedCampaignId(int shard) {
        return getNewLastProcessedCampaignId(dslContextProvider.ppc(shard));
    }

    Long getNewLastProcessedCampaignId(DSLContext dslContext) {
        return dslContext
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS.forceIndex(PRIMARY))
                .where(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .orderBy(CAMPAIGNS.CID.desc())
                .limit(1)
                .fetchOptional(CAMPAIGNS.CID)
                .map(v -> v + 1)
                .orElse(ID_NOT_SET);
    }

    public void setCampaignsFinished(int shard, Collection<Long> cids) {
        if (cids.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard).update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_EMPTY, CampaignsStatusempty.No)
                .where(CAMPAIGNS.CID.in(cids))
                .execute();
    }


    /**
     * Получить список следующих {@code campaignId} для добавления в очередь ре-экспорта,
     * начиная с {@code lastProcessedCampaignId} (при этом само значение в выборку не входит),
     * количеством не более {@code limit} штук.
     * <p>
     * Логика получения (обработка от большего cid к меньшему) аналогична {@link #getNewLastProcessedCampaignId(int)}
     *
     * @param shard                   шард
     * @param lastProcessedCampaignId граничное значение {@code campaignId} для выборки
     * @param limit                   ограничение на количество кампаний для выборки
     * @return список {@code campaignId}
     */
    public List<Long> getCampaignIdsForFullExport(int shard, long lastProcessedCampaignId, int limit) {
        return getCampaignIdsForFullExport(dslContextProvider.ppc(shard), lastProcessedCampaignId, limit);
    }

    List<Long> getCampaignIdsForFullExport(DSLContext dslContext, long lastProcessedCampaignId, int limit) {
        return dslContext
                .select(CAMPAIGNS.CID)
                .hint(STRAIGHT_JOIN)
                .from(CAMPAIGNS)
                .leftJoin(WALLETS).on(WALLETS.CID.eq(CAMPAIGNS.WALLET_CID))
                .where(CAMPAIGNS.CID.lt(lastProcessedCampaignId)
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                        .and(campTypeIn(CampaignTypeKinds.BS_EXPORT))
                        .and(getNoWalletOrWalletHasOrderIdCondition(CAMPAIGNS, WALLETS)))
                .orderBy(CAMPAIGNS.CID.desc())
                .limit(limit)
                .fetch(CAMPAIGNS.CID);

    }


    /**
     * Получить список id cpm_yndx_frontpage компаний с определенной датой начала и StatusBsSynced = Yes.
     *
     * @param shard     шард
     * @param startDate дата начала компании
     * @return список айди компаний
     */
    @QueryWithoutIndex("Вызывается только из jobs")
    public List<Long> getCampaignIdsForResendToBSToAvoidGenocide(int shard, LocalDate startDate) {
        return dslContextProvider.ppc(shard)
                .select(Campaigns.CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.START_TIME.eq(startDate)
                        .and(CAMPAIGNS.TYPE.eq(CampaignsType.cpm_yndx_frontpage))
                        .and(CAMPAIGNS.STATUS_BS_SYNCED.eq(CampaignsStatusbssynced.Yes)))
                .fetch(CAMPAIGNS.CID);
    }

    /**
     * Получить список следующих {@code campaignId} для пересчета статусов,
     * начиная с {@code lastProcessedCampaignId} (при этом само значение в выборку не входит),
     * количеством не более {@code limit} штук.
     * <p>
     * Логика получения (обработка от большего cid к меньшему) аналогична {@link #getNewLastProcessedCampaignId(int)}
     *
     * @param shard                   шард
     * @param lastProcessedCampaignId граничное значение {@code campaignId} для выборки
     * @param limit                   ограничение на количество кампаний для выборки
     * @return список {@code campaignId}
     */
    public List<Long> getCampaignIdsForStatusRecalculation(int shard, long lastProcessedCampaignId, int limit) {
        return getCampaignIdsForStatusRecalculation(dslContextProvider.ppc(shard), lastProcessedCampaignId, limit);
    }

    List<Long> getCampaignIdsForStatusRecalculation(DSLContext dslContext, long lastProcessedCampaignId, int limit) {
        return dslContext
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.lt(lastProcessedCampaignId)
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .orderBy(CAMPAIGNS.CID.desc())
                .limit(limit)
                .fetch(CAMPAIGNS.CID);
    }

    /**
     * Были ли у клиента {@param clientId} кампании с заданным типом {@param campaignTypes} и с деньгами
     */
    public boolean didClientHaveAnyCampaignWithMoney(int shard, ClientId clientId, Set<CampaignType> campaignTypes) {
        Condition condition = CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                .and(CAMPAIGNS.TYPE.in(mapSet(campaignTypes, CampaignType::toSource)))
                .and(CAMPAIGNS.SUM.gt(BigDecimal.ZERO)
                        .or(CAMPAIGNS.SUM_LAST.gt(BigDecimal.ZERO))
                        .or(CAMPAIGNS.SUM_SPENT.gt(BigDecimal.ZERO))
                );
        return dslContextProvider.ppc(shard)
                .fetchExists(CAMPAIGNS, condition);
    }

    /**
     * Возвращает map тип кампании -> список доменов
     *
     * @param shard         шард
     * @param clientId      id пользователя
     * @param campaignTypes какие типы кампаний искать
     */
    public Map<CampaignType, Set<String>> getCampaignTypeToDomains(int shard,
                                                                   ClientId clientId,
                                                                   Set<CampaignType> campaignTypes) {
        var result = dslContextProvider.ppc(shard)
                .selectDistinct(CAMPAIGNS.TYPE, BANNERS.DOMAIN, DOMAINS.DOMAIN)
                .from(CAMPAIGNS)
                .join(PHRASES).on(PHRASES.CID.eq(CAMPAIGNS.CID))
                .join(BANNERS).on(BANNERS.PID.eq(PHRASES.PID))
                .leftJoin(ADGROUPS_DYNAMIC).on(PHRASES.PID.eq(ADGROUPS_DYNAMIC.PID))
                .leftJoin(DOMAINS).on(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID.eq(DOMAINS.DOMAIN_ID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.TYPE.in(mapSet(campaignTypes, CampaignType::toSource)))
                .fetch();

        return StreamEx.of(result)
                .mapToEntry(Record3::component1)
                .invert()
                .mapKeys(CampaignType::fromSource)
                .flatMapValues(rec -> StreamEx.of(rec.component2(), rec.component3()))
                .nonNullValues()
                .grouping(toSet());
    }

    /**
     * Обновить дневной бюджет у переданной кампании
     *
     * @param shard      шард
     * @param campaignId id кампании
     * @param dayBudget  новый бюджет
     * @return - количество изменённых строк (0 или 1)
     */
    public int updateDailyBudget(int shard, Long campaignId, BigDecimal dayBudget) {
        return dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.DAY_BUDGET, dayBudget)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, statusBsSyncedToDb(StatusBsSynced.NO))
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now())
                .where(CAMPAIGNS.CID.eq(campaignId))
                .execute();
    }

    public int updateDailyBudgetAndResetShowMode(int shard, Long walletId, BigDecimal dayBudget) {
        return dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.DAY_BUDGET, dayBudget)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, statusBsSyncedToDb(StatusBsSynced.NO))
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now())
                .set(
                        CAMPAIGNS.DAY_BUDGET_SHOW_MODE,
                        ru.yandex.direct.dbschema.ppc.enums.CampaignsDayBudgetShowMode.default_
                )
                .where(CAMPAIGNS.CID.eq(walletId))
                .execute();
    }

    /**
     * Обновляет флаг status_autobudget_show для кампании
     *
     * @param shard       шард
     * @param campaignIds список id кампаний
     * @param status      новый статус
     */
    public void updateStatusAutoBudgetShow(int shard, Collection<Long> campaignIds, StatusAutobudgetShow status) {
        Configuration conf = dslContextProvider.ppc(shard).configuration();
        if (campaignIds.isEmpty()) {
            return;
        }
        DSL.using(conf)
                .update(Phrases.PHRASES)
                .set(Phrases.PHRASES.STATUS_AUTOBUDGET_SHOW, StatusAutobudgetShow.toSource(status))
                .set(Phrases.PHRASES.LAST_CHANGE, Phrases.PHRASES.LAST_CHANGE)
                .where(Phrases.PHRASES.CID.in(campaignIds))
                .execute();
    }

    /**
     * Ставит нескольким кампаниям mobile_app_id в таблице campaigns_mobile_content.
     * Что кампания существует и у неё тип mobile_content, не проверяется.
     *
     * @param campaignIdToMobileAppId какой кампании (cid) какой mobile_app_id поставить
     */
    public void setMobileAppIds(int shard, Map<Long, Long> campaignIdToMobileAppId) {
        setMobileAppIds(dslContextProvider.ppc(shard), campaignIdToMobileAppId);
    }

    /**
     * Ставит нескольким кампаниям mobile_app_id в таблице campaigns_mobile_content.
     * Что кампания существует и у неё тип mobile_content, не проверяется.
     *
     * @param campaignIdToMobileAppId какой кампании (cid) какой mobile_app_id поставить
     */
    public void setMobileAppIds(DSLContext dslContext, Map<Long, Long> campaignIdToMobileAppId) {

        InsertValuesStep2<CampaignsMobileContentRecord, Long, Long> step = dslContext
                .insertInto(CAMPAIGNS_MOBILE_CONTENT)
                .columns(CAMPAIGNS_MOBILE_CONTENT.CID, CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID);

        for (Map.Entry<Long, Long> entry : campaignIdToMobileAppId.entrySet()) {
            step = step.values(entry.getKey(), entry.getValue());
        }

        SqlUtils.onConflictUpdate(step, singleton(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID))
                .execute();
    }

    /**
     * Получить идентификаторы мобильных приложений, привязанных к РМП кампании
     *
     * @return отображение campaignId -> mobileAppId
     */
    public Map<Long, Long> getCampaignMobileAppIds(int shard, Collection<Long> campaignIds) {
        return getCampaignMobileAppIds(dslContextProvider.ppc(shard), campaignIds);
    }

    /**
     * Получить идентификаторы мобильных приложений, привязанных к РМП кампании
     *
     * @return отображение campaignId -> mobileAppId
     */
    public Map<Long, Long> getCampaignMobileAppIds(DSLContext dslContext, Collection<Long> campaignIds) {
        return dslContext.select(CAMPAIGNS_MOBILE_CONTENT.CID, CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID)
                .from(CAMPAIGNS_MOBILE_CONTENT)
                .where(CAMPAIGNS_MOBILE_CONTENT.CID.in(campaignIds))
                .and(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID.ne(0L))
                .fetchMap(CAMPAIGNS_MOBILE_CONTENT.CID, CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID);
    }

    /**
     * Получить идентификаторы кампаний клиента, привязанных к мобильному приложению, но не UAC
     */
    public List<Long> getNonArchivedMobileAppCampaignIds(int shard, ClientId clientId, Long mobileAppId) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS_MOBILE_CONTENT.CID, CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID)
                .from(CAMPAIGNS_MOBILE_CONTENT)
                .join(MOBILE_APPS).on(MOBILE_APPS.MOBILE_APP_ID.eq(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_MOBILE_CONTENT.CID))
                .where(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID.eq(mobileAppId))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(MOBILE_APPS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.SOURCE.ne(CampaignsSource.uac))
                .fetch(CAMPAIGNS_MOBILE_CONTENT.CID);
    }

    /**
     * Получить ссылку на приложение в магазине мобильных приложений, привязанных к РМП кампании
     *
     * @return отображение campaignId -> store_href
     */
    public Map<Long, String> getStoreUrlByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard).select(CAMPAIGNS_MOBILE_CONTENT.CID, MOBILE_APPS.STORE_HREF)
                .from(CAMPAIGNS_MOBILE_CONTENT)
                .join(MOBILE_APPS).on(MOBILE_APPS.MOBILE_APP_ID.eq(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID))
                .where(CAMPAIGNS_MOBILE_CONTENT.CID.in(campaignIds))
                .and(CAMPAIGNS_MOBILE_CONTENT.MOBILE_APP_ID.ne(0L))
                .fetchMap(CAMPAIGNS_MOBILE_CONTENT.CID, MOBILE_APPS.STORE_HREF);
    }

    /**
     * Определяет по списку id кампаний, есть ли у каждой из кампаний активные баннеры.
     */
    public Set<Long> getCampaignsWithActiveBanners(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptySet();
        }

        // Models/Banners.pm:1652
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .andExists(dslContextProvider.ppc(shard)
                        .selectOne()
                        .from(BANNERS)
                        .join(PHRASES)
                        .using(PHRASES.PID)
                        .where(BANNERS.CID.eq(CAMPAIGNS.CID))
                        // MTools.pm, IS_ACTIVE_CLAUSE
                        .and(
                                PHRASES.STATUS_POST_MODERATE.eq(PhrasesStatuspostmoderate.Yes)
                                        .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes))
                                        .and(BANNERS.BANNER_TYPE.ne(BannersBannerType.text)
                                                .or(BANNERS.PHONEFLAG.eq(BannersPhoneflag.Yes))
                                                .or(DSL.ifnull(BANNERS.HREF, "").ne("")))
                                        .or(BANNERS.STATUS_ACTIVE.eq(BannersStatusactive.Yes))
                        )
                        .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                )
                .fetchSet(CAMPAIGNS.CID);
    }

    /**
     * Проверить существование кампании.
     *
     * @param shard      шард
     * @param campaignId id кампании
     * @return {@code true} если в указанном шарде существует кампании со statusEmpty = No, иначе {@code false}
     */
    public boolean campaignExists(int shard, long campaignId) {
        CampaignsSelectionCriteria selectionCriteria = new CampaignsSelectionCriteria()
                .withCampaignIds(singleton(campaignId))
                .withStatusEmpty(false);

        return getCampaignsSimpleStream(shard, selectionCriteria)
                .findAny()
                .isPresent();
    }

    /*
     * По списку id-кампаний возвращает Map cid->allowed_page_ids, где allowed_page_ids это список id-площадок,
     * на которых допустимо показывать материалы данной кампании.
     * В случае невозможности десериализовать json кинет исключение, т.к.
     * данная ситуация говорит о неконсистентности данных, которую необходимо исправить
     * @param shard
     * @param campaignIds
     * @return Map<Long,List<Long>> cid -> camp_options.allowed_page_ids
     */
    public Map<Long, List<Long>> getCampaignsAllowedPageIds(int shard, Collection<Long> campaignIds) {
        Map<Long, String> jsonPageIdsByCid = dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.ALLOWED_PAGE_IDS)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID, CAMP_OPTIONS.ALLOWED_PAGE_IDS);

        return EntryStream.of(jsonPageIdsByCid).mapValues(CampaignMappings::allowedPageIdsFromDb).toMap();
    }

    /**
     * По списку id-кампаний возвращает Map cid->disallowed_page_ids, где disallowed_page_ids это список id-площадок,
     * на которых недопустимо показывать материалы данной кампании.
     */
    public Map<Long, List<Long>> getCampaignsDisallowedPageIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.DISALLOWED_PAGE_IDS)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID, r -> allowedPageIdsFromDb(r.get(CAMP_OPTIONS.DISALLOWED_PAGE_IDS)));
    }

    /**
     * Получить из camp_options словарь соответствий о id-площадок привязанных к кампании по её номеру списка
     *
     * @param shard       шард
     * @param campaignIds идентификаторы кампаний
     */
    public Map<Long, Pair<List<Long>, List<Long>>> getCampaignsAllowedAndDisallowedPageIdsMap(int shard,
                                                                                              Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.ALLOWED_PAGE_IDS, CAMP_OPTIONS.DISALLOWED_PAGE_IDS)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID,
                        r -> Pair.of(
                                allowedPageIdsFromDb(r.get(CAMP_OPTIONS.ALLOWED_PAGE_IDS)),
                                allowedPageIdsFromDb(r.get(CAMP_OPTIONS.DISALLOWED_PAGE_IDS))
                        )
                );
    }

    /**
     * По идентификаторам кампаний получить типы показа, которые им соответствуют
     * в колонке allowed_frontpage_types в таблице ppc.campaigns_cpm_yndx_frontpage
     *
     * @param shard                номер шарда
     * @param frontpageCampaignIds идентификаторы кампаний
     * @return Map id кампании -> список мест размещения этой кампании
     */
    public Map<Long, Set<FrontpageCampaignShowType>> getFrontpageTypesForCampaigns(int shard,
                                                                                   List<Long> frontpageCampaignIds) {
        Map<Long, String> showTypesByCampaignId = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID, CAMPAIGNS_CPM_YNDX_FRONTPAGE.ALLOWED_FRONTPAGE_TYPES)
                .from(CAMPAIGNS_CPM_YNDX_FRONTPAGE)
                .where(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID.in(frontpageCampaignIds))
                .fetchMap(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID, CAMPAIGNS_CPM_YNDX_FRONTPAGE.ALLOWED_FRONTPAGE_TYPES);

        return EntryStream.of(showTypesByCampaignId)
                .mapValues(CpmYndxFrontpageShowTypeUtils::toFrontpageShowType)
                .toMap();
    }

    /**
     * Обновляет camp_options.allowed_page_ids по cid-у. Падает если записи в camp_options для данного cid-а нет,
     * т.к. такая кампания - сломана
     * Если pageIds передано как null или пустой список, то сбрасывает allowed_page_ids в NULL в БД (дефолт)
     * После обновления сбрасываем StatusBsSynced на кампании
     */
    public void updateAllowedPageIdsAndFlushStatusBsSynced(int shard, Long campaignId, List<Long> pageIds) {
        updateAllowedPageIdsAndFlushStatusBsSynced(shard, campaignId, pageIds, true);

        // После разрешения https://github.com/jOOQ/jOOQ/issues/3266
        // можно будет переделать на один запрос по двум таблицам
    }

    public void updateAllowedPageIds(int shard, Long campaignId, List<Long> pageIds) {
        updateAllowedPageIdsAndFlushStatusBsSynced(shard, campaignId, pageIds, false);
    }

    private void updateAllowedPageIdsAndFlushStatusBsSynced(
            int shard, Long campaignId, List<Long> pageIds, boolean needFlush) {
        dslContextProvider.ppc(shard).update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.ALLOWED_PAGE_IDS, pageIdsToDb(pageIds))
                .where(CAMP_OPTIONS.CID.eq(campaignId))
                .execute();
        // В случае гонок с транспортом можем отправить лишний раз, что, ввиду очень малой вероятности
        // не страшно, поэтому транзакцию не делаем
        if (needFlush) {
            flushStatusBsSynced(shard, campaignId);
        }
    }

    /**
     * Обновляет camp_options.disallowed_page_ids по cid-у.
     * Падает если записи в camp_options для данного cid-а нет
     */
    public void updateDisallowedPageIds(int shard, Long campaignId, List<Long> pageIds) {
        updateDisallowedPageIdsAndFlushStatusBsSynced(shard, campaignId, pageIds, false);
    }

    private void updateDisallowedPageIdsAndFlushStatusBsSynced(
            int shard, Long campaignId, List<Long> pageIds, boolean needFlush) {
        dslContextProvider.ppc(shard).update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.DISALLOWED_PAGE_IDS, pageIdsToDb(pageIds))
                .where(CAMP_OPTIONS.CID.eq(campaignId))
                .execute();
        if (needFlush) {
            flushStatusBsSynced(shard, campaignId);
        }
    }

    private void flushStatusBsSynced(int shard, Long campaignId) {
        dslContextProvider.ppc(shard).update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .set(CAMPAIGNS.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE)
                .where(CAMPAIGNS.CID.eq(campaignId))
                .execute();
    }

    public BigDecimal getSumFromCampaignsUnderWallet(DSLContext context, long walletId) {
        Field<BigDecimal> childTotalSum = DSL.ifnull(sum(CAMPAIGNS.SUM), BigDecimal.ZERO);

        return context.select(childTotalSum)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.WALLET_CID.eq(walletId))
                .forUpdate()
                .fetchOne(childTotalSum);
    }

    private JooqMapperWithSupplier<Campaign> createCampaignMapper() {
        return JooqMapperWithSupplierBuilder.builder(Campaign::new)
                .map(property(Campaign.ID, CAMPAIGNS.CID))
                .map(convertibleProperty(Campaign.STATUS_ARCHIVED, CAMPAIGNS.ARCHIVED,
                        CampaignMappings::archivedFromDb,
                        CampaignMappings::archivedToDb))
                .map(convertibleProperty(Campaign.STATUS_ACTIVE, CAMPAIGNS.STATUS_ACTIVE,
                        CampaignMappings::statusActiveFromDb,
                        CampaignMappings::statusActiveToDb))
                .map(convertibleProperty(Campaign.AUTOBUDGET, CAMPAIGNS.AUTOBUDGET,
                        CampaignMappings::autobudgetFromDb,
                        CampaignMappings::autobudgetToDb))
                .map(convertibleProperty(Campaign.CURRENCY, CAMPAIGNS.CURRENCY,
                        CampaignMappings::currencyCodeFromDb,
                        CampaignMappings::currencyCodeToDb))
                .map(convertibleProperty(Campaign.CURRENCY_CONVERTED, CAMPAIGNS.CURRENCY_CONVERTED,
                        CampaignMappings::currencyConvertedFromDb,
                        CampaignMappings::currencyConvertedToDb))
                .map(property(Campaign.NAME, CAMPAIGNS.NAME))
                .map(convertibleProperty(Campaign.TYPE, CAMPAIGNS.TYPE,
                        CampaignType::fromSource,
                        CampaignType::toSource))
                .map(property(Campaign.CLIENT_ID, CAMPAIGNS.CLIENT_ID))
                .map(property(Campaign.USER_ID, CAMPAIGNS.UID))
                .map(property(Campaign.AGENCY_ID, CAMPAIGNS.AGENCY_ID))
                .map(property(Campaign.AGENCY_USER_ID, CAMPAIGNS.AGENCY_UID))
                .map(property(Campaign.MANAGER_USER_ID, CAMPAIGNS.MANAGER_UID))
                .map(property(Campaign.ORDER_ID, CAMPAIGNS.ORDER_ID))
                .map(convertibleProperty(Campaign.STATUS_BS_SYNCED, CAMPAIGNS.STATUS_BS_SYNCED,
                        CampaignMappings::statusBsSyncedFromDb,
                        CampaignMappings::statusBsSyncedToDb))
                .map(convertibleProperty(Campaign.STATUS_SHOW, CAMPAIGNS.STATUS_SHOW,
                        CampaignMappings::statusShowFromDb,
                        CampaignMappings::statusShowToDb))
                .map(convertibleProperty(Campaign.STATUS_MODERATE, CAMPAIGNS.STATUS_MODERATE,
                        CampaignStatusModerate::fromSource,
                        CampaignStatusModerate::toSource))
                .map(property(Campaign.WALLET_ID, CAMPAIGNS.WALLET_CID))
                .map(property(Campaign.SUM, CAMPAIGNS.SUM))
                .map(property(Campaign.SUM_BALANCE, CAMPAIGNS.SUM_BALANCE))
                .map(property(Campaign.SUM_SPENT, CAMPAIGNS.SUM_SPENT))
                .map(property(Campaign.SUM_LAST, CAMPAIGNS.SUM_LAST))
                .map(property(Campaign.SUM_TO_PAY, CAMPAIGNS.SUM_TO_PAY))
                .map(property(Campaign.LAST_CHANGE, CAMPAIGNS.LAST_CHANGE))
                .map(property(Campaign.AUTOBUDGET_FORECAST_DATE, CAMPAIGNS.AUTOBUDGET_FORECAST_DATE))
                .map(property(Campaign.START_TIME, CAMPAIGNS.START_TIME))
                .map(property(Campaign.FINISH_TIME, CAMPAIGNS.FINISH_TIME))
                .map(convertibleProperty(Campaign.MINUS_KEYWORDS, CAMP_OPTIONS.MINUS_WORDS,
                        MinusKeywordsPackUtils::minusKeywordsFromJson,
                        MinusKeywordsPackUtils::minusKeywordsToJson))
                .map(property(Campaign.MINUS_KEYWORDS_ID, CAMP_OPTIONS.MW_ID))
                .map(convertibleProperty(Campaign.DISABLED_SSP, CAMPAIGNS.DISABLED_SSP,
                        CampaignMappings::disabledSspFromJson,
                        CampaignMappings::disabledSspToJson))
                .map(convertibleProperty(Campaign.DISABLED_VIDEO_PLACEMENTS, CAMPAIGNS.DISABLED_VIDEO_PLACEMENTS,
                        CampaignMappings::disabledVideoPlacementsFromJson,
                        CampaignMappings::disabledVideoPlacementsToJson))
                /* //Когда кампании (в интерфейсе переедут на Java это понадобится
//                   + перевести getCampaignsAllowedPageIds на getCampaignsWithFieldsStream
//                .map(convertibleProperty(Campaign.ALLOWED_PAGE_IDS, CAMP_OPTIONS.ALLOWED_PAGE_IDS,
//                        CampaignMappings::allowedPageIdsFromDb,
//                        CampaignMappings::allowedPageIdsToDb))
                */
                .map(convertibleProperty(Campaign.ALLOWED_DOMAINS, CAMP_OPTIONS.ALLOWED_DOMAINS,
                        CampaignMappings::stringListFromDbJsonFormat,
                        CampaignMappings::stringListToDbJsonFormat))
                .map(convertibleProperty(Campaign.ALLOWED_SSP, CAMP_OPTIONS.ALLOWED_SSP,
                        CampaignMappings::stringListFromDbJsonFormat,
                        CampaignMappings::stringListToDbJsonFormat))
                .map(convertibleProperty(Campaign.DISABLED_DOMAINS, CAMPAIGNS.DONT_SHOW,
                        CampaignMappings::disabledDomainsFromDb,
                        CampaignMappings::disabledDomainsToDb))
                .map(convertibleProperty(Campaign.GEO, CAMPAIGNS.GEO,
                        CampaignMappings::geoFromDb,
                        CampaignMappings::geoToDb))
                .map(convertibleProperty(Campaign.STATUS_POST_MODERATE, CAMP_OPTIONS.STATUS_POST_MODERATE,
                        CampaignStatusPostmoderate::fromSource,
                        CampaignStatusPostmoderate::toSource))
                .map(convertibleProperty(Campaign.FAIR_AUCTION, CAMP_OPTIONS.FAIR_AUCTION,
                        CampaignMappings::fairAuctionFromDb,
                        CampaignMappings::fairAuctionToDb))
                .map(convertibleProperty(Campaign.OPTS, CAMPAIGNS.OPTS,
                        CampaignMappings::optsFromDb,
                        CampaignMappings::optsToDb))
                .map(convertibleProperty(Campaign.SMS_FLAGS, CAMP_OPTIONS.SMS_FLAGS,
                        CampaignMappings::smsFlagsFromDb,
                        CampaignMappings::smsFlagsToDb))
                .map(convertibleProperty(Campaign.EMAIL_NOTIFICATIONS, CAMP_OPTIONS.EMAIL_NOTIFICATIONS,
                        CampaignMappings::emailNotificationsFromDb,
                        CampaignMappings::emailNotificationsToDb))
                .map(property(Campaign.DAY_BUDGET_DAILY_CHANGE_COUNT, CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT))
                .map(property(Campaign.DAY_BUDGET_STOP_TIME, CAMP_OPTIONS.DAY_BUDGET_STOP_TIME))
                .map(property(Campaign.DAY_BUDGET_LAST_CHANGE, CAMP_OPTIONS.DAY_BUDGET_LAST_CHANGE))
                .map(property(Campaign.DAY_BUDGET_SHOW_MODE, CAMPAIGNS.DAY_BUDGET_SHOW_MODE))
                .map(property(Campaign.DAY_BUDGET, CAMPAIGNS.DAY_BUDGET))
                .map(property(Campaign.TIMEZONE_ID, CAMPAIGNS.TIMEZONE_ID))
                .map(convertibleProperty(Campaign.TIME_TARGET, CAMPAIGNS.TIME_TARGET,
                        nullSafeReader(TimeTarget::parseRawString),
                        nullSafeWriter(TimeTarget::toRawFormat)))
                .map(convertibleProperty(Campaign.DAY_BUDGET_NOTIFICATION_STATUS,
                        CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS,
                        DayBudgetNotificationStatus::fromSource,
                        DayBudgetNotificationStatus::toSource))
                .map(property(Campaign.EMAIL, CAMP_OPTIONS.EMAIL))
                .map(convertibleProperty(Campaign.DEVICE_TARGETING, CAMP_OPTIONS.DEVICE_TARGETING,
                        CampaignMappings::deviceTargetingFromDb,
                        CampaignMappings::deviceTargetingToDb))
                .map(integerProperty(Campaign.CONTEXT_PRICE_COEF, CAMPAIGNS.CONTEXT_PRICE_COEF))
                .map(convertibleProperty(Campaign.STATUS_EMPTY, CAMPAIGNS.STATUS_EMPTY,
                        CampaignMappings::statusEmptyFromDb,
                        CampaignMappings::statusEmptyToDb))
                .map(property(Campaign.BROAD_MATCH_LIMIT, CAMP_OPTIONS.BROAD_MATCH_LIMIT))
                .map(booleanProperty(Campaign.BROAD_MATCH_FLAG, CAMP_OPTIONS.BROAD_MATCH_FLAG,
                        CampOptionsBroadMatchFlag.class))
                .map(property(Campaign.CREATE_TIME, CAMP_OPTIONS.CREATE_TIME))
                .map(property(Campaign.MONEY_WARNING_VALUE, CAMP_OPTIONS.MONEY_WARNING_VALUE))
                .map(integerProperty(Campaign.STATUS_MAIL, CAMPAIGNS.STATUS_MAIL))
                .readProperty(Campaign.SMS_TIME, fromField(CAMP_OPTIONS.SMS_TIME)
                        .by(CampaignMappings::smsTimeFromDb))
                .readProperty(Campaign.SOURCE, fromField(CAMPAIGNS.SOURCE)
                        .by(CampaignSource::fromSource))
                .readProperty(Campaign.METATYPE, fromField(CAMPAIGNS.METATYPE)
                        .by(CampaignMetatype::fromSource))
                .build();
    }

    public static JooqMapperWithSupplier<DbStrategy> createDbStrategyMapper(boolean useRfFields) {
        var mapper = JooqMapperWithSupplierBuilder.builder(DbStrategy::new)

                // ppc.campaigns
                .map(convertibleProperty(DbStrategy.AUTOBUDGET, CAMPAIGNS.AUTOBUDGET,
                        CampaignsAutobudget::fromSource,
                        CampaignsAutobudget::toSource))
                .map(convertibleProperty(DbStrategy.STRATEGY_NAME, CAMPAIGNS.STRATEGY_NAME,
                        StrategyName::fromSource,
                        StrategyName::toSource))
                .map(convertibleProperty(DbStrategy.STRATEGY_DATA, CAMPAIGNS.STRATEGY_DATA,
                        CampaignMappings::strategyDataFromDb,
                        CampaignMappings::strategyDataToDb))
                .map(property(DbStrategy.DAY_BUDGET, CAMPAIGNS.DAY_BUDGET))
                .map(convertibleProperty(DbStrategy.DAY_BUDGET_SHOW_MODE, CAMPAIGNS.DAY_BUDGET_SHOW_MODE,
                        CampaignsDayBudgetShowMode::fromSource,
                        CampaignsDayBudgetShowMode::toSource))
                .map(convertibleProperty(DbStrategy.PLATFORM, CAMPAIGNS.PLATFORM,
                        CampaignsPlatform::fromSource,
                        CampaignsPlatform::toSource))

                // ppc.camp_options
                .map(convertibleProperty(DbStrategy.STRATEGY, CAMP_OPTIONS.STRATEGY,
                        CampOptionsStrategy::fromSource,
                        CampOptionsStrategy::toSource))
                .map(property(DbStrategy.DAY_BUDGET_DAILY_CHANGE_COUNT, CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT))
                .map(convertibleProperty(DbStrategy.DAY_BUDGET_NOTIFICATION_STATUS,
                        CAMP_OPTIONS.DAY_BUDGET_NOTIFICATION_STATUS,
                        CampOptionsDayBudgetNotificationStatus::fromSource,
                        CampOptionsDayBudgetNotificationStatus::toSource))
                .writeField(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME, fromPropertyToField(DbStrategy.DAY_BUDGET_STOP_TIME)
                        .by(RepositoryUtils::zeroableDateTimeToDb))
                .readProperty(DbStrategy.DAY_BUDGET_STOP_TIME, fromField(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME));

        // нет, если используем репозиторию 0
        if (useRfFields) {
            mapper
                    .map(integerProperty(DbStrategy.RF, CAMPAIGNS.RF))
                    .map(integerProperty(DbStrategy.RF_RESET, CAMPAIGNS.RF_RESET));
        }

        return mapper.build();
    }

    private static JooqMapperWithSupplier<BaseCampaign> createCampsForServicingMapper() {
        return JooqMapperWithSupplierBuilder.<BaseCampaign>builder(CampaignStub::new)
                .map(property(CommonCampaign.ID, CAMPS_FOR_SERVICING.CID))
                .build();
    }

    public void bindCampaignAndAbSegment(int shard, Long campaignId, Long retargetingConditionId) {
        dslContextProvider.ppc(shard).update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .set(CAMPAIGNS.AB_SEGMENT_RET_COND_ID, retargetingConditionId)
                .set(CAMPAIGNS.AB_SEGMENT_STAT_RET_COND_ID, retargetingConditionId)
                .where(CAMPAIGNS.CID.eq(campaignId))
                .execute();
    }

    public void unbindCampaignFromAbSegment(int shard, List<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard).update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .set(CAMPAIGNS.AB_SEGMENT_RET_COND_ID, (Long) null)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Типовой расчет остатка по кампании с учетом общего счета.
     * Работает некорректно для кампаний под не отключаемым ОС, подробнее в DIRECT-85857
     *
     * @param campaigns       таблица кампании
     * @param walletCampaigns таблица общего счета
     * @return поле, соответствующее {@code c.sum - c.sum_spent + (wc.sum - wc.sum_spent)}
     */
    public static Field<BigDecimal> campaignBalance(Campaigns campaigns, Campaigns walletCampaigns) {
        Field<BigDecimal> campaignsSumRest = campaigns.SUM.minus(campaigns.SUM_SPENT);
        Field<BigDecimal> walletCampaignsSumRest = DSL.iif(walletCampaigns.CID.isNull(), BigDecimal.ZERO,
                walletCampaigns.SUM.minus(walletCampaigns.SUM_SPENT));
        return campaignsSumRest.plus(walletCampaignsSumRest);
    }

    /**
     * Условие статуса активности кампании:
     * <ul>
     * <li>Для внутренних бесплатных кампаний вычисляется остаток юнитов {@code sum_units - sum_spent_units}
     * (остаток показов, кликов или дней)
     * <li>Для остальных внутренних кампаний статус активности не изменяется
     * <li>Для остальных кампаний считается остаток на самой кампании {@code c.sum - c.sum_spent} плюс
     * остаток на кошельке {@code wc.sum - wc.sum_spent} (если кошелёк есть)
     * </ul>
     *
     * @implNote Логика этого метода отличается от аналогичного условия в перле в части работы
     * с внутренними кампаниями
     * (см. <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Common.pm#L238">Common.pm</a>)
     */
    public static Condition campaignStatusActiveCondition(Campaigns campaigns, Campaigns walletCampaigns) {
        // Для обычных кампаний считаем сумму остатков на самой кампании и на её кошельке, если он есть
        Condition externalCampaignCondition = campaigns.TYPE.notIn(INTERNAL_CAMPAIGN_TYPES);
        Condition externalCondition = externalCampaignCondition.and(
                campaignBalance(campaigns, walletCampaigns).gt(Currencies.EPSILON));

        // Для внутренней бесплатной кампании считаем остаток юнитов
        Condition internalFreeCondition = campaigns.TYPE.eq(internal_free)
                .and(campaigns.SUM_UNITS.minus(campaigns.SUM_SPENT_UNITS).gt(0L));

        // Для дистрибуционных внутренних кампаний ничего не считаем
        Condition internalCondition = campaigns.TYPE.in(internal_distrib, internal_autobudget)
                .and(campaigns.STATUS_SHOW.eq(CampaignsStatusshow.Yes));

        return externalCondition.or(internalFreeCondition).or(internalCondition);
    }

    /**
     * Условие на то, что у кампании нет общего счета, или общий счет уже получил OrderID
     * <p>
     * копия перлового {@code $BS::Export::NO_WALLET_OR_WALLET_HAS_ORDERID}
     */
    public static Condition getNoWalletOrWalletHasOrderIdCondition(Campaigns campaigns, Campaigns walletCampaigns) {
        return campaigns.WALLET_CID.eq(ID_NOT_SET)
                .or(campaigns.WALLET_CID.gt(ID_NOT_SET)
                        .and(DSL.ifnull(walletCampaigns.ORDER_ID, ID_NOT_SET).gt(ID_NOT_SET)));
    }


    public Map<Long, String> getBrandSurveyIdsForCampaigns(int shard, List<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.BRAND_SURVEY_ID)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(CAMP_OPTIONS.CID, CAMP_OPTIONS.BRAND_SURVEY_ID);
    }

    public void deleteBrandSurveyId(int shard, List<Long> cids) {
        if (cids.isEmpty()) {
            return;
        }

        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.BRAND_SURVEY_ID, (String) null)
                .where(CAMP_OPTIONS.CID.in(cids))
                .execute();
    }

    /**
     * Массовая смена владельца кампаний.
     * Вызывается при смене главного представителя клиента и при удалении представителей
     *
     * @param shard      - шард
     * @param sourceUids - список uid-ов, которые нужно поменять
     * @param targetUid  - uid нового владельца визитки (проставляется chiefUid)
     */
    public void updateCampOwners(int shard, Collection<Long> sourceUids, Long targetUid) {
        dslContextProvider.ppc(shard).update(CAMPAIGNS)
                .set(CAMPAIGNS.UID, targetUid)
                .where(CAMPAIGNS.UID.in(sourceUids))
                .execute();
    }

    public void setCampaignsCpmPriceStatusCorrect(Configuration conf, Long campaignId,
                                                  PriceFlightStatusCorrect status,
                                                  @Nullable PriceFlightReasonIncorrect reasonIncorrect) {
        conf.dsl()
                .update(CAMPAIGNS_CPM_PRICE)
                .set(CAMPAIGNS_CPM_PRICE.STATUS_CORRECT, PriceFlightConverter.statusCorrectToDbFormat(status))
                .set(CAMPAIGNS_CPM_PRICE.REASON_INCORRECT,
                        PriceFlightConverter.reasonIncorrectToDbFormat(reasonIncorrect))
                .where(CAMPAIGNS_CPM_PRICE.CID.eq(campaignId))
                .execute();
    }

    public int setCampaignsCpmPriceAuctionPriority(int shard, Long campaignId, Long auctionPriority) {
        return dslContextProvider.ppc(shard)
                .update(CAMPAIGNS_CPM_PRICE)
                .set(CAMPAIGNS_CPM_PRICE.AUCTION_PRIORITY, auctionPriority)
                .where(CAMPAIGNS_CPM_PRICE.CID.eq(campaignId))
                .execute();
    }

    public CampaignCounts getCampaignCountsForCountLimitCheck(int shard, Long clientId) {
        return dslContextProvider.ppc(shard)
                .select(count().as("ALL"),
                        coalesce(sum(case_(CAMPAIGNS.ARCHIVED).when(CampaignsArchived.No, 1).else_(0)), 0).as("UNARC"))
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.TYPE.notIn(SKIP_IN_CLIENT_CAMPAIGNS_COUNT))
                .andNot(CAMPAIGNS.CURRENCY_CONVERTED.eq(CampaignsCurrencyconverted.Yes)
                        .and(ifnull(CAMPAIGNS.CURRENCY, CampaignsCurrency.YND_FIXED).eq(CampaignsCurrency.YND_FIXED)))
                .fetchOne(r -> new CampaignCounts(
                        (Integer) r.get("ALL"),
                        ((BigDecimal) r.get("UNARC")).intValue()));
    }

    public int addCampaignsForServicing(DSLContext context, Collection<CommonCampaign> campaigns) {
        List<CommonCampaign> campaignsForServicing = filterList(campaigns, CommonCampaign::getIsServiceRequested);

        InsertHelper<CampsForServicingRecord> insertHelper = new InsertHelper<>(context, CAMPS_FOR_SERVICING);
        insertHelper.addAll(campsForServicingMapper, campaignsForServicing);

        return insertHelper.executeIfRecordsAdded();
    }

    public Set<Long> getCampaignIdsForServicing(DSLContext dslContext, Collection<CommonCampaign> campaigns) {
        List<Long> campaignIds = mapList(campaigns, CommonCampaign::getId);
        return dslContext
                .select(campsForServicingMapper.getFieldsToRead())
                .from(CAMPS_FOR_SERVICING)
                .where(CAMPS_FOR_SERVICING.CID.in(campaignIds))
                .fetchSet(CAMPS_FOR_SERVICING.CID);
    }

    public boolean managerHasClientCampaignsIgnoringCurrentCampaigns(
            int shard,
            ClientId clientId,
            Long managerUid,
            Set<Long> currentCampaignIds,
            Set<CampaignType> campaignTypes) {
        //noinspection ConstantConditions
        checkState(managerUid != null, "managerUid must be not null");

        Condition condition = CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                .and(CAMPAIGNS.MANAGER_UID.eq(managerUid))
                .and(CAMPAIGNS.CID.notIn(currentCampaignIds))
                .and(CAMPAIGNS.TYPE.in(mapSet(campaignTypes, CampaignType::toSource)));

        return dslContextProvider.ppc(shard)
                .fetchExists(CAMPAIGNS, condition);
    }

    public boolean isFirstCampaignsUnderWallet(
            int shard, ClientId clientId,
            Collection<CampaignType> newCampaignTypes, Set<Long> currentAddingCampaignIds) {
        if (newCampaignTypes.stream().noneMatch(CampaignTypeKinds.UNDER_WALLET::contains)) {
            return false;
        }

        return isFirstCampaignsUnderWalletRegardlessOfCampaignTypes(shard, clientId, currentAddingCampaignIds);
    }

    public boolean isFirstCampaignsUnderWalletRegardlessOfCampaignTypes(
            int shard, ClientId clientId, Set<Long> currentAddingCampaignIds) {

        Condition condition = CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                .and(CAMPAIGNS.CID.notIn(currentAddingCampaignIds))
                .and(CAMPAIGNS.TYPE.in(mapSet(CampaignTypeKinds.UNDER_WALLET, CampaignType::toSource)))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));

        return !dslContextProvider.ppc(shard)
                .fetchExists(CAMPAIGNS, condition);
    }

    public boolean clientHasWebEditBaseAgencyCampaigns(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.AGENCY_ID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                        .and(CAMPAIGNS.TYPE.in(WEB_EDIT_BASE_TYPES))
                        .and(CAMPAIGNS.AGENCY_ID.gt(ID_NOT_SET)))
                .limit(1)
                .fetchOne(CAMPAIGNS.AGENCY_ID) != null;
    }

    public Map<Long, List<Long>> getCampaignIdsByFeedId(int shard, Collection<Long> feedIds) {
        return dslContextProvider.ppc(shard)
                .select(ADGROUPS_PERFORMANCE.FEED_ID, Phrases.PHRASES.CID)
                .from(ADGROUPS_PERFORMANCE)
                .join(Phrases.PHRASES).on(ADGROUPS_PERFORMANCE.PID.eq(Phrases.PHRASES.PID))
                .where(ADGROUPS_PERFORMANCE.FEED_ID.in(feedIds))
                .fetchGroups(ADGROUPS_PERFORMANCE.FEED_ID, Phrases.PHRASES.CID);
    }

    public List<Long> getCampaignIdsWithBrandSafety(int shard, Long clientId) {
        return new ArrayList<>(dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.BRANDSAFETY_RET_COND_ID.isNotNull())
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId))
                .fetchSet(CAMPAIGNS.CID));
    }

    public Set<Long> getTouchCampaignIds(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID).from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(findInSet(CampaignOpts.IS_TOUCH.getTypedValue(), CAMPAIGNS.OPTS).gt(0L))
                .fetchSet(CAMPAIGNS.CID);
    }

    /**
     * Вернуть кампании из {@code campaignIds}, у которых прописаны агентства из {@code agencyIds}.
     */
    public Set<Long> getCampaignIdsWithAgencyIds(int shard, Collection<Long> campaignIds, Collection<Long> agencyIds) {
        return getCampaignIdsWithAgencyIds(dslContextProvider.ppc(shard), campaignIds, agencyIds);
    }

    /**
     * Вернуть кампании из {@code campaignIds}, у которых прописаны агентства из {@code agencyIds}.
     */
    public Set<Long> getCampaignIdsWithAgencyIds(DSLContext context, Collection<Long> campaignIds,
                                                 Collection<Long> agencyIds) {
        if (campaignIds.isEmpty() || agencyIds.isEmpty()) {
            return Collections.emptySet();
        }

        return context
                .select(CAMPAIGNS.CID).from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.AGENCY_ID.in(agencyIds))
                .fetchSet(CAMPAIGNS.CID);
    }

    public List<Campaign> selectCampaignsForUpdate(DSLContext dslContext, Set<Long> campaignIds,
                                                   Set<CampaignType> campaignTypes) {
        return dslContext
                .select(campaignMapperFields)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.TYPE.in(mapList(campaignTypes, CampaignType::toSource)))
                .forUpdate()
                .fetch()
                .map(campaignMapper::fromDb);
    }

    public void setCampOptionsStopTime(int shard, Collection<Long> campaignIds, LocalDateTime stopTime) {
        dslContextProvider.ppc(shard)
                .update(CAMP_OPTIONS)
                .set(CAMP_OPTIONS.STOP_TIME, stopTime)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .execute();
    }

    public void stopCampaigns(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(CAMPAIGNS
                        .join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID)))
                .set(CAMPAIGNS.STATUS_SHOW, CampaignsStatusshow.No)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now())
                .set(CAMP_OPTIONS.STOP_TIME, LocalDateTime.now())
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .execute();
    }

    public void stopCampaignsForBlockedUsers(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(CAMPAIGNS
                        .join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID))
                        .join(USERS).on(CAMPAIGNS.UID.eq(USERS.UID)))
                .set(CAMPAIGNS.STATUS_SHOW, CampaignsStatusshow.No)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now())
                .set(CAMP_OPTIONS.STOP_TIME, LocalDateTime.now())
                .where(CAMPAIGNS.CID.in(campaignIds).and(USERS.STATUS_BLOCKED.eq(UsersStatusblocked.Yes)))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .execute();
    }

    public Map<Long, List<Long>> getCampaignIdsForStopping(int shard, Collection<Long> clientIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.CLIENT_ID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(CAMPAIGNS.CLIENT_ID.in(clientIds))
                .and(CAMPAIGNS.TYPE.ne(CampaignsType.wallet))
                .fetchGroups(CAMPAIGNS.CLIENT_ID, CAMPAIGNS.CID);
    }

    public Set<Long> getCampaignsWithTurboAppsEnables(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID).from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds)
                        .and(findInSet(CampaignOpts.HAS_TURBO_APP.getTypedValue(), CAMPAIGNS.OPTS).gt(0L)))
                .fetchSet(CAMPAIGNS.CID);
    }

    /**
     * Находит кампании с указанной опцией, принадлежащие клиентам.
     *
     * @param shard     шард клиентов
     * @param clientIds список клиентов
     * @param option    опция кампании
     * @return Map с ключом ClientId и знаением списком cid
     */
    public Map<Long, List<Long>> getCampaignsWithOptionByClientIds(int shard, List<ClientId> clientIds,
                                                                   CampaignOpts option) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CLIENT_ID, CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .and(findInSet(option.getTypedValue(), CAMPAIGNS.OPTS).gt(0L))
                .fetchGroups(CAMPAIGNS.CLIENT_ID, CAMPAIGNS.CID);
    }

    public Set<Long> getCampaignsWithOption(int shard, ClientId clientId, CampaignOpts option) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID).from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                        .and(findInSet(option.getTypedValue(), CAMPAIGNS.OPTS).gt(0L)))
                .fetchSet(CAMPAIGNS.CID);
    }

    public Map<Long, String> getHrefCampAdditionalData(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_ADDITIONAL_DATA.CID, CAMP_ADDITIONAL_DATA.HREF).from(CAMP_ADDITIONAL_DATA)
                .where(CAMP_ADDITIONAL_DATA.CID.in(campaignIds))
                .fetchMap(CAMP_ADDITIONAL_DATA.CID, CAMP_ADDITIONAL_DATA.HREF);
    }

    public Map<Long, LocalDateTime> getLastShowTimeByCampaignId(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.LAST_SHOW_TIME)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS.CID, CAMPAIGNS.LAST_SHOW_TIME);
    }

    public Map<Long, Long> getCampaignIdsByPricePackageIds(int shard, Collection<Long> pricePackageIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS_CPM_PRICE.PACKAGE_ID)
                .from(CAMPAIGNS)
                .join(CAMPAIGNS_CPM_PRICE).on(CAMPAIGNS.CID.eq(CAMPAIGNS_CPM_PRICE.CID))
                .where(CAMPAIGNS_CPM_PRICE.PACKAGE_ID.in(pricePackageIds))
                .fetchMap(CAMPAIGNS.CID, CAMPAIGNS_CPM_PRICE.PACKAGE_ID);
    }

    public List<Long> getBookedCampaignIds(int shard, Collection<Long> pricePackageIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .hint(STRAIGHT_JOIN)
                .from(CAMPAIGNS_CPM_PRICE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_CPM_PRICE.CID))
                .where(CAMPAIGNS_CPM_PRICE.PACKAGE_ID.in(pricePackageIds))
                .and(CAMPAIGNS.FINISH_TIME.ge(LocalDate.now()))
                .and(CAMPAIGNS_CPM_PRICE.STATUS_APPROVE.eq(CampaignsCpmPriceStatusApprove.Yes))
                .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                .fetch(CAMPAIGNS.CID);
    }

    public List<Long> getCampaignIdsForResendBSByPricePackageIds(int shard, Collection<Long> pricePackageIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .join(CAMPAIGNS_CPM_PRICE).on(CAMPAIGNS.CID.eq(CAMPAIGNS_CPM_PRICE.CID))
                .where(CAMPAIGNS_CPM_PRICE.PACKAGE_ID.in(pricePackageIds)
                        .and(CAMPAIGNS.FINISH_TIME.greaterThan(LocalDate.now()))
                        .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes))
                        .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No)))
                .fetch(CAMPAIGNS.CID);
    }

    /**
     * Определяет по списку id кампаний, является ли каждая из них универсальной кампанией
     */
    public Set<Long> getUniversalCampaigns(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptySet();
        }

        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(sourceIn(AvailableCampaignSources.INSTANCE.ucSources()))
                .fetchSet(CAMPAIGNS.CID);
    }

    public Map<Long, CampaignType> getUniversalCampaignType(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.TYPE)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(sourceIn(AvailableCampaignSources.INSTANCE.ucSources()))
                .fetchMap(CAMPAIGNS.CID, r -> CampaignType.fromSource(r.get(CAMPAIGNS.TYPE)));
    }

    /**
     * Получить id последних созданных у клиента кампаний нужных типов
     *
     * @param shard    шард
     * @param clientId id клиента
     * @param types    типы кампаний
     * @param limit    лимит числа кампаний
     * @return коллекция id кампаний, отсортированная по убыванию
     */
    public Collection<Long> getLatestCampaignIds(int shard, ClientId clientId,
                                                 Collection<CampaignType> types, int limit) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                        .and(CAMPAIGNS.TYPE.in(mapList(types, CampaignType::toSource)))
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .orderBy(CAMPAIGNS.CID.desc())
                .limit(limit)
                .fetch(CAMPAIGNS.CID);
    }

    /**
     * Обновить campaigns.source для некоторых кампаний
     *
     * @param shard       - шард в котором нужны изменения
     * @param campaignIds - список идентификаторов кампаний
     * @param source      - новое значение поля source
     */
    public int updateCampaignSource(int shard, Collection<Long> campaignIds, CampaignsSource source) {
        if (isEmpty(campaignIds)) {
            return 0;
        }
        return dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .set(CAMPAIGNS.SOURCE, source)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    /**
     * Для пары (клиент, агентство) отдать id последней созданной неархивной кампании с переданным campaigns.source
     *
     * @param shard          — шард клиента
     * @param clientId       — id клиента
     * @param agencyClientId — id агентства
     * @param source         — искомое значение поля source
     * @return Optional c cid
     */
    public Optional<Long> getLastNotArchivedCampaignOfSource(int shard,
                                                             ClientId clientId, @Nullable ClientId agencyClientId,
                                                             CampaignsSource source) {
        var agencyCondition = (agencyClientId == null) ?
                CAMPAIGNS.AGENCY_ID.eq(0L) : CAMPAIGNS.AGENCY_ID.eq(agencyClientId.asLong());
        var context = dslContextProvider.ppc(shard);
        return context.select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())
                        .and(agencyCondition)
                        .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                        .and(CAMPAIGNS.SOURCE.eq(source)))
                .orderBy(CAMPAIGNS.CID.desc())
                .limit(1)
                .fetchOptional(CAMPAIGNS.CID);
    }

    /**
     * Установить дефолтный (NULL) campaigns.timeTarget
     *
     * @param shard       - шард в котором нужны изменения
     * @param campaignIds - список идентификаторов кампаний
     */
    public int setDefaultCampaignTimeTarget(int shard, Collection<Long> campaignIds) {
        if (isEmpty(campaignIds)) {
            return 0;
        }
        return dslContextProvider.ppc(shard)
                .update(CAMPAIGNS)
                .setNull(CAMPAIGNS.TIME_TARGET)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    public List<Long> getCampaignIdsThatCouldBeFronpageAndActive(int shard, LocalDate date) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .join(CAMPAIGNS_CPM_PRICE).on(CAMPAIGNS.CID.eq(CAMPAIGNS_CPM_PRICE.CID))
                .where(CAMPAIGNS.FINISH_TIME.greaterOrEqual(date)
                        .and(CAMPAIGNS.START_TIME.lessOrEqual(date))
                        .and(CAMPAIGNS_CPM_PRICE.IS_CPD_PAUSED.eq(0L)))
                .unionAll(DSL.select(CAMPAIGNS.CID)
                        .from(CAMPAIGNS)
                        .join(CAMPAIGNS_CPM_YNDX_FRONTPAGE).on(CAMPAIGNS.CID.eq(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID))
                        .where(CAMPAIGNS.FINISH_TIME.greaterOrEqual(date)
                                .and(CAMPAIGNS.START_TIME.lessOrEqual(date))
                                .and(CAMPAIGNS_CPM_YNDX_FRONTPAGE.IS_CPD_PAUSED.eq(0L)))).fetch(CAMPAIGNS.CID);

    }

    public int updateHasSiteMonitoring(int shard, Collection<Long> campaignIds, boolean hasSiteMonitoring) {
        if (isEmpty(campaignIds)) {
            return 0;
        }
        return dslContextProvider.ppc(shard)
                .update(CAMPAIGNS.join(CAMP_OPTIONS).on(CAMPAIGNS.CID.eq(CAMP_OPTIONS.CID)))
                .set(CAMP_OPTIONS.STATUS_METRICA_CONTROL, hasSiteMonitoring ?
                        CampOptionsStatusmetricacontrol.Yes : CampOptionsStatusmetricacontrol.No)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, statusBsSyncedToDb(StatusBsSynced.NO))
                .set(CAMPAIGNS.LAST_CHANGE, LocalDateTime.now())
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    public List<CampaignWithAutobudget> getCampaignsAutobudgetData(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.TIME_TARGET, CAMPAIGNS.CLIENT_ID, CAMPAIGNS.AUTOBUDGET)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetch()
                .map(this::getCampaignWithAutobudget);
    }

    private CampaignWithAutobudget getCampaignWithAutobudget(org.jooq.Record4<Long, String, Long,
            ru.yandex.direct.dbschema.ppc.enums.CampaignsAutobudget> record) {
        final var cid = record.get(CAMPAIGNS.CID);
        final var timeTarget = this.getTimeTarget(cid, record.get(CAMPAIGNS.TIME_TARGET));
        final var clientId = ClientId.fromLong(record.get(CAMPAIGNS.CLIENT_ID));

        return new CampaignWithAutobudget(cid, timeTarget, record.get(CAMPAIGNS.AUTOBUDGET), clientId);
    }

    @Nullable
    private TimeTarget getTimeTarget(Long cid, @Nullable String timeTargetStr) {
        if (timeTargetStr == null) {
            return null;
        }
        try {
            return TimeTarget.parseRawString(timeTargetStr);
        } catch (RuntimeException e) {
            logger.error("Campaign {} has wrong format of timetable '{}'", cid,
                    timeTargetStr);
            throw e;
        }
    }

    /**
     * Получает widgetPartnerIds для переданных campaignIds.
     *
     * @return Map (cid -> widget_partner_id)
     */
    public Map<Long, Long> getWidgetPartnerIdsByCids(int shard, Collection<Long> cids) {
        return dslContextProvider.ppc(shard)
                .select(WIDGET_PARTNER_CAMPAIGNS.CID, WIDGET_PARTNER_CAMPAIGNS.WIDGET_PARTNER_ID)
                .from(WIDGET_PARTNER_CAMPAIGNS)
                .where(WIDGET_PARTNER_CAMPAIGNS.CID.in(cids))
                .fetchMap(WIDGET_PARTNER_CAMPAIGNS.CID, WIDGET_PARTNER_CAMPAIGNS.WIDGET_PARTNER_ID);
    }

    /**
     * Возвращает кампании с проставленным флагом is_cpd_paused
     *
     * @param shard - шард из которого необходимо получить кампании
     * @return коллекция кампаний
     */

    public Collection<Campaign> getCampaignsWithIsCpdPaused(int shard) {
        var context = dslContextProvider.ppc(shard);
        SelectConditionStep<Record> frontPageCampaigns = context.select(campaignMapperFields)
                .from(CAMPAIGNS_CPM_YNDX_FRONTPAGE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID))
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID))
                .where(CAMPAIGNS_CPM_YNDX_FRONTPAGE.IS_CPD_PAUSED.eq(IS_CPD_PAUSED));
        SelectConditionStep<Record> cpmPriceCampaigns = context.select(campaignMapperFields)
                .from(CAMPAIGNS_CPM_PRICE)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_CPM_PRICE.CID))
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS_CPM_PRICE.CID))
                .where(CAMPAIGNS_CPM_PRICE.IS_CPD_PAUSED.eq(IS_CPD_PAUSED));
        return frontPageCampaigns
                .union(cpmPriceCampaigns)
                .fetch()
                .map(campaignMapper::fromDb);
    }

    /**
     * Возвращает Id охватных кампаний, которые активны (были показы за вчера/сегодня) и у которых нет эксперимента
     *
     * @param shard - шард из которого необходимо получить кампании
     * @return коллекция Id кампаний
     */

    public List<Long> getCpmCampaignIdsWithoutExperiment(int shard) {
        var context = dslContextProvider.ppc(shard);
        List<Long> fetch = context.select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(DSL.localDateTimeAdd(CAMPAIGNS.LAST_SHOW_TIME, 2, DatePart.DAY).greaterThan(DSL.currentLocalDateTime())
                        .and(CAMPAIGNS.TYPE.in(CampaignsType.cpm_banner, CampaignsType.cpm_price,
                                CampaignsType.cpm_yndx_frontpage))
                        .and(CAMPAIGNS.AB_SEGMENT_RET_COND_ID.isNull()))
                .fetch(CAMPAIGNS.CID);
        return fetch;
    }

    /**
     * Проставляет значение флага is_cpd_paused в указанных кампаниях
     *
     * @param shard       - шард в котором необходимо применить изменения
     * @param campaignIds - список id кампаний у которых необходимо выставить значение is_cpd_paused
     * @param isCpdPaused - значение флага is_cpd_paused, которое необходимо выставить
     */
    public void updateCampaignsIsCpdPaused(int shard, Collection<Long> campaignIds, boolean isCpdPaused) {
        var context = dslContextProvider.ppc(shard);
        context.update(CAMPAIGNS_CPM_YNDX_FRONTPAGE)
                .set(CAMPAIGNS_CPM_YNDX_FRONTPAGE.IS_CPD_PAUSED, isCpdPaused ? 1L : 0L)
                .where(CAMPAIGNS_CPM_YNDX_FRONTPAGE.CID.in(campaignIds))
                .execute();
        context.update(CAMPAIGNS_CPM_PRICE)
                .set(CAMPAIGNS_CPM_PRICE.IS_CPD_PAUSED, isCpdPaused ? 1L : 0L)
                .where(CAMPAIGNS_CPM_PRICE.CID.in(campaignIds))
                .execute();
        context.update(CAMPAIGNS)
                .set(CAMPAIGNS.STATUS_BS_SYNCED, CampaignsStatusbssynced.No)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .execute();
    }

    public Map<Long, List<MeaningfulGoal>> getMeaningfulGoalsByCampaignId(
            int shard,
            Collection<Long> campaignIds
    ) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.MEANINGFUL_GOALS)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(
                        r -> r.get(CAMP_OPTIONS.CID),
                        r -> CampaignConverter.meaningfulGoalsFromDb(r.get(CAMP_OPTIONS.MEANINGFUL_GOALS))
                );
    }

    public Long getLastCidForType(int shard, Long clientId, CampaignType campaignType) {
        var context = dslContextProvider.ppc(shard);
        return context.select(max(CAMPAIGNS.CID))
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId))
                .and(Campaigns.CAMPAIGNS.TYPE.eq(CampaignType.toSource(campaignType)))
                .fetchOne(0, Long.class);
    }

    public Map<Long, Long> getStrategyIdsByCampaignIds(int shard, ClientId clientId,
                                                       Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.STRATEGY_ID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS.CID, CAMPAIGNS.STRATEGY_ID);
    }

    public Map<Long, Long> getStrategyIdsByCampaignIds(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.STRATEGY_ID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS.CID, CAMPAIGNS.STRATEGY_ID);
    }

    /**
     * if not exist moderated banner and some banners on moderation now
     * small hack for change campaign's status
     * https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/CampaignTools.pm?rev=r9190755#L520
     */
    public boolean needChangeStatusModeratePayCondition(int shard, Long cid) {
        Result<Record2<BigDecimal, BigDecimal>> result = dslContextProvider.ppc(shard)
                .select(sum(when(BANNERS.STATUS_MODERATE.in(
                                        BannersStatusmoderate.Ready, BannersStatusmoderate.Sent,
                                        BannersStatusmoderate.Sending)
                                .or(PHRASES.STATUS_MODERATE.in(
                                        PhrasesStatusmoderate.Ready, PhrasesStatusmoderate.Sent,
                                        PhrasesStatusmoderate.Sending))
                                .or(BANNERS.VCARD_ID.isNotNull().and(BANNERS.PHONEFLAG.in(
                                        BannersPhoneflag.Ready, BannersPhoneflag.Sent,
                                        BannersPhoneflag.Sending))
                                ), 1).otherwise(0)).as(CNT_ON_MODERATION_ALIAS),
                        sum(when(BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Yes, BannersStatusmoderate.No)
                                .or(PHRASES.STATUS_MODERATE.in(PhrasesStatusmoderate.Yes, PhrasesStatusmoderate.No))
                                .or(BANNERS.PHONEFLAG.in(BannersPhoneflag.Yes, BannersPhoneflag.No)
                                ), 1).otherwise(0)).as(CNT_MODERATED_ALIAS)
                )
                .from(PHRASES)
                .join(BANNERS)
                .on(BANNERS.PID.eq(PHRASES.PID))
                .where(PHRASES.CID.eq(cid))
                .fetch();

        BigDecimal onModeration = (BigDecimal) result.get(0).get(CNT_ON_MODERATION_ALIAS);
        BigDecimal moderated = (BigDecimal) result.get(0).get(CNT_MODERATED_ALIAS);
        return (onModeration != null && onModeration.intValue() > 0)
                && (moderated == null || moderated.intValue() == 0);
    }

    public Set<Long> getStoppedCampaignsForArchiving(int shard, Collection<Long> campaignIds,
                                                     int minutesAfterLastShow) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .join(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .and(CAMPAIGNS.STATUS_SHOW.equal(CampaignsStatusshow.No))
                .and(CAMPAIGNS.LAST_SHOW_TIME.equal(SqlUtils.mysqlZeroLocalDateTime())
                        .or(DSL.localDateTimeAdd(CAMPAIGNS.LAST_SHOW_TIME, minutesAfterLastShow, DatePart.MINUTE)
                                .lessThan(DSL.currentLocalDateTime())))
                .and(CAMPAIGNS.ORDER_ID.equal(DEFAULT_ORDER_ID)
                        .or(CAMP_OPTIONS.STOP_TIME.equal(SqlUtils.mysqlZeroLocalDateTime()))
                        .or(DSL.localDateTimeAdd(CAMP_OPTIONS.STOP_TIME, minutesAfterLastShow, DatePart.MINUTE)
                                .lessThan(DSL.currentLocalDateTime())))
                .fetchSet(CAMPAIGNS.CID);
    }

    public Set<Long> getCampaignIdsWithBanners(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .leftSemiJoin(BANNERS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetchSet(CAMPAIGNS.CID);
    }

    public List<Long> getSortedNonArchivedCampaignIdsByClientId(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No)
                        .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .orderBy(CAMPAIGNS.CID)
                .fetch(CAMPAIGNS.CID);
    }

    // https://a.yandex-team.ru/arcadia/direct/perl/protected/Common.pm?rev=r9287185#L3109
    public List<CampaignWithBidsDomainsAndPhones> getCampaignsWithBidsDomainsAndPhones(
            int shard, Collection<Long> campaignIds) {
        Map<Long, Result<Record4<Long, Long, String, String>>> campaignIdToDataList = dslContextProvider.ppc(shard)
                .select(PHRASES.CID, BANNERS.BID, BANNERS.DOMAIN, VCARDS.PHONE)
                .from(PHRASES)
                .join(BANNERS).on(BANNERS.PID.eq(PHRASES.PID))
                .leftJoin(VCARDS).on(VCARDS.VCARD_ID.eq(BANNERS.VCARD_ID))
                .where(PHRASES.CID.in(campaignIds))
                .and(BANNERS.STATUS_MODERATE.ne(BannersStatusmoderate.New))
                .fetchGroups(PHRASES.CID);
        return EntryStream.of(campaignIdToDataList)
                .mapKeyValue((k, v) -> new CampaignWithBidsDomainsAndPhones(k,
                        v.getValues(BANNERS.BID),
                        v.getValues(BANNERS.DOMAIN).stream()
                                .filter(Objects::nonNull)
                                .collect(toList()),
                        v.getValues(VCARDS.PHONE).stream()
                                .filter(Objects::nonNull)
                                .map(VcardMappings::phoneFromDb)
                                .map(p -> p.getCountryCode() + p.getCityCode() + p.getPhoneNumber())
                                .collect(toList())
                )).toList();
    }
}
