package ru.yandex.direct.core.entity.campaign.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.SelectQuery;
import org.jooq.Table;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType;
import ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory;
import ru.yandex.direct.core.entity.campaign.repository.type.CampaignRepositoryTypeSupportFacade;
import ru.yandex.direct.core.entity.campaign.service.type.add.container.RestrictedCampaignsAddOperationContainer;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.multitype.repository.TypedRepository;
import ru.yandex.direct.multitype.repository.filter.Filter;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIsNotArchivedTypesFilter;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignTypesFilter;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.greaterThan;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Repository
@ParametersAreNonnullByDefault
public class CampaignTypedRepository extends TypedRepository<BaseCampaign, CampaignType,
        RestrictedCampaignsAddOperationContainer, RestrictedCampaignsUpdateOperationContainer> {

    @Autowired
    public CampaignTypedRepository(CampaignRepositoryTypeSupportFacade typeSupportFacade,
                                   DslContextProvider dslContextProvider) {
        super(dslContextProvider, typeSupportFacade);
    }

    @Override
    protected Filter getIdFilter(Collection<Long> modelIds) {
        return CampaignFilterFactory.campaignIdsFilter(modelIds);
    }

    @Override
    protected Table<?> getBaseTable() {
        return CAMPAIGNS;
    }

    public Set<Long> getClientCampaignIds(int shard, ClientId clientId, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptySet();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.CID.in(campaignIds))
                .fetchSet(CAMPAIGNS.CID);
    }

    public Map<Long, CampaignsType> getClientCampaignIdsWithType(int shard, ClientId clientId,
                                                                 Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.TYPE)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.CID.in(campaignIds))
                .fetch()
                .intoMap(CAMPAIGNS.CID, CAMPAIGNS.TYPE);
    }

    public Map<Long, CampaignsType> getCampaignIdsWithType(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID, CAMPAIGNS.TYPE)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetch()
                .intoMap(CAMPAIGNS.CID, CAMPAIGNS.TYPE);
    }

    /**
     * Из переданной мапы campaignTypeMap достаёт id кампаний заданного типа и загружает их.
     */
    public <T extends CampaignWithCampaignType> Map<Long, T> getTypedCampaignsMap(
            int shard,
            Map<Long, CampaignType> campaignTypeMap,
            CampaignType campaignType) {
        return getTypedCampaignsMap(dslContextProvider.ppc(shard), campaignTypeMap, campaignType);
    }

    /**
     * Из переданной мапы campaignTypeMap достаёт id кампаний заданного типа и загружает их.
     */
    public <T extends CampaignWithCampaignType> Map<Long, T> getTypedCampaignsMap(
            DSLContext ctx,
            Map<Long, CampaignType> campaignTypeMap,
            CampaignType campaignType) {
        List<Long> campaignIdsOfType = EntryStream.of(campaignTypeMap)
                .filterValues(campaignType::equals)
                .keys().toList();
        return getTypedCampaignsMap(ctx, campaignIdsOfType);
    }

    /**
     * В DIRECT-107937 можно переделать.
     */
    public <T extends CampaignWithCampaignType> Map<Long, T> getTypedCampaignsMap(
            DSLContext ctx,
            Collection<Long> campaignIds) {
        var campaigns = getTypedCampaigns(ctx, campaignIds);
        @SuppressWarnings("unchecked")
        Map<Long, T> result = (Map<Long, T>) listToMap(campaigns, BaseCampaign::getId);
        return result;
    }

    public Map<Long, ? extends BaseCampaign> getTypedCampaignsMap(int shard, Collection<Long> campaignIds) {
        var campaigns = getTypedCampaigns(shard, campaignIds);
        return listToMap(campaigns, BaseCampaign::getId);
    }

    public Map<Long, ? extends BaseCampaign> getClientsTypedCampaignsByType(int shard, ClientId clientId,
                                                                            Collection<CampaignType> campaignTypes) {
        var campaigns = getTypedCampaigns(shard, clientId, campaignTypes);
        return listToMap(campaigns, BaseCampaign::getId);
    }

    public List<? extends BaseCampaign> getTypedCampaigns(int shard, ClientId clientId,
                                                          Collection<CampaignType> campaignTypes) {
        return getTypedCampaigns(dslContextProvider.ppc(shard), clientId, campaignTypes);
    }

    public List<? extends BaseCampaign> getTypedCampaigns(DSLContext dslContext, ClientId clientId,
                                                          Collection<CampaignType> campaignTypes) {
        return getTyped(dslContext, null, campaignTypesFilter(clientId, campaignTypes), null);
    }

    public List<BaseCampaign> getTypedCampaigns(DSLContext dslContext, Filter filter,
                                                @Nullable LimitOffset limitOffset) {
        return getTyped(dslContext, filter, limitOffset);
    }

    public List<BaseCampaign> getTypedCampaigns(DSLContext dslContext, Filter filter) {
        return getTyped(dslContext, filter, null);
    }

    public List<? extends BaseCampaign> getTypedCampaigns(int shard, Collection<Long> campaignIds) {
        return getTyped(shard, campaignIds);
    }

    public List<? extends BaseCampaign> getTypedCampaigns(DSLContext dslContext, Collection<Long> campaignIds) {
        return getTyped(dslContext, campaignIds);
    }

    /**
     * Оптимизатор mysql выбирает использовать BNL для джойна и внешняя сортировка сваливается в filesort
     * https://bugs.mysql.com/bug.php?id=69721
     * Подробнее тут https://st.yandex-team.ru/DIRECT-170741
     * <p>
     * Вместо этого стоит использовать {@link #getSafelyCampaignsForClassesAndIdGreaterThan}.
     */
    @Deprecated
    public List<BaseCampaign> getCampaignsForTypesAndIdGreaterThan(int shard, Long campaignId,
                                                                   Collection<CampaignsType> types,
                                                                   int limit) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        SelectQuery<Record> query = getBaseQuery(dslContext);
        query.addConditions(CAMPAIGNS.TYPE.in(types).and(CAMPAIGNS.CID.greaterThan(campaignId)));
        query.addOrderBy(CAMPAIGNS.CID);
        return getTyped(dslContext, query, null, new LimitOffset(limit, 0));
    }

    public List<BaseCampaign> getSafelyCampaignsForClassesAndIdGreaterThan(int shard,
                                                                           Long campaignId,
                                                                           Collection<Class<? extends BaseCampaign>> classes,
                                                                           int limit) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return getSafely(
                dslContext,
                greaterThan(CAMPAIGNS.CID, campaignId),
                new LimitOffset(limit, 0),
                classes
        );
    }

    public <T extends BaseCampaign> List<T> getSafelyNotArchivedCampaignsWithType(
            int shard,
            ClientId clientId,
            Collection<CampaignType> campaignTypes,
            Class<T> clazz) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        Filter filter = campaignIsNotArchivedTypesFilter(clientId, campaignTypes);
        return getSafely(dslContext, filter, clazz);
    }

    public List<BaseCampaign> getSafelyOrderedByCampaignId(
            int shard,
            Filter filter,
            LimitOffset limitOffset,
            Set<Class<? extends BaseCampaign>> classes) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        SelectQuery<Record> query = getBaseQuery(dslContext);
        query.addOrderBy(CAMPAIGNS.CID);
        return getSafely(dslContext, filter, limitOffset, classes);
    }
}
