package ru.yandex.direct.core.entity.campaign.repository;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.UpdateSetMoreStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.CampaignMulticurrencySums;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsMulticurrencySumsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_MULTICURRENCY_SUMS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@ParametersAreNonnullByDefault
@Repository
public class CampaignsMulticurrencySumsRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<CampaignMulticurrencySums> jooqMapper;

    @Autowired
    public CampaignsMulticurrencySumsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = createMapper();
    }

    /**
     * Добавить записи в таблицу campaigns_multicurrency_sums
     *
     * @param campaignMulticurrencySums модель записи для добавления в таблицу
     * @param shard                     номер шарда в который добавляем данные
     * @return количество записей, добавленных в таблицу
     */
    public int insertCampaignsMulticurrencySums(int shard,
                                                CampaignMulticurrencySums campaignMulticurrencySums) {
        InsertHelper<CampaignsMulticurrencySumsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), CAMPAIGNS_MULTICURRENCY_SUMS);
        insertHelper.add(jooqMapper, campaignMulticurrencySums);
        return insertHelper.execute();
    }

    /**
     * Обновить запись в таблице campaigns_multicurrency_sums по cid и balance_tid
     *
     * @param shard                            номер шарда в котором обновляем данные
     * @param campaignMulticurrencySumsChanges новое состояние модели
     * @return true, если запись была успешно обновлена, иначе false
     */

    public boolean updateCampaignsMulticurrencySumsByCidAndBalanceTid(
            int shard,
            CampaignMulticurrencySums campaignMulticurrencySumsChanges,
            Long cmsBalanceTid
    ) {
        UpdateSetMoreStep<CampaignsMulticurrencySumsRecord> updateStep = dslContextProvider.ppc(shard)
                .update(CAMPAIGNS_MULTICURRENCY_SUMS)
                .set(CAMPAIGNS_MULTICURRENCY_SUMS.BALANCE_TID, cmsBalanceTid);

        if (campaignMulticurrencySumsChanges.getSum() != null) {
            updateStep.set(CAMPAIGNS_MULTICURRENCY_SUMS.SUM, campaignMulticurrencySumsChanges.getSum());
        }

        if (campaignMulticurrencySumsChanges.getChipsCost() != null) {
            updateStep.set(CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST, campaignMulticurrencySumsChanges.getChipsCost());
        }

        if (campaignMulticurrencySumsChanges.getChipsSpent() != null) {
            updateStep.set(CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_SPENT, campaignMulticurrencySumsChanges.getChipsSpent());
        }

        if (campaignMulticurrencySumsChanges.getAvgDiscount() != null) {
            updateStep
                    .set(CAMPAIGNS_MULTICURRENCY_SUMS.AVG_DISCOUNT, campaignMulticurrencySumsChanges.getAvgDiscount());
        }

        return updateStep
                .where(CAMPAIGNS_MULTICURRENCY_SUMS.CID.eq(campaignMulticurrencySumsChanges.getId())
                        .and(CAMPAIGNS_MULTICURRENCY_SUMS.BALANCE_TID.eq(cmsBalanceTid)))
                .execute() > 0;
    }

    /**
     * Возвращает мапу со стоимостью потраченных фишек до конвертации в валюту.
     * <p>
     * Ключ - campaignId.
     * <p>
     * В мапе могут присутствовать не все запрошенные кампании.
     */
    public Map<Long, BigDecimal> getCampaignsMulticurrencyChipsCosts(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS_MULTICURRENCY_SUMS.CID, CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST)
                .from(CAMPAIGNS_MULTICURRENCY_SUMS)
                .where(CAMPAIGNS_MULTICURRENCY_SUMS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS_MULTICURRENCY_SUMS.CID, CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST);
    }

    private JooqMapperWithSupplier<CampaignMulticurrencySums> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(CampaignMulticurrencySums::new)
                .map(property(CampaignMulticurrencySums.ID, CAMPAIGNS_MULTICURRENCY_SUMS.CID))
                .map(property(CampaignMulticurrencySums.SUM, CAMPAIGNS_MULTICURRENCY_SUMS.SUM))
                .map(property(CampaignMulticurrencySums.CHIPS_COST, CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST))
                .map(property(CampaignMulticurrencySums.CHIPS_SPENT, CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_SPENT))
                .map(property(CampaignMulticurrencySums.AVG_DISCOUNT, CAMPAIGNS_MULTICURRENCY_SUMS.AVG_DISCOUNT))
                .map(property(CampaignMulticurrencySums.BALANCE_TID, CAMPAIGNS_MULTICURRENCY_SUMS.BALANCE_TID))
                .build();
    }

}
