package ru.yandex.direct.core.entity.campaign.repository;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.CampaignsPromotion;
import ru.yandex.direct.dbschema.ppc.tables.records.CampaignsPromotionsRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static org.jooq.impl.DSL.falseCondition;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS_PROMOTIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.flatMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Repository
public class CampaignsPromotionsRepository {

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private static final JooqMapperWithSupplier<CampaignsPromotion> CAMPAIGNS_PROMOTION_MAPPER =
            createCampaignsPromotionMapper();

    @Autowired
    public CampaignsPromotionsRepository(ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
    }

    public Map<Long, List<CampaignsPromotion>> getCampaignsPromotionsByCid(
            DSLContext dslContext, Collection<Long> campaignIds) {
        return dslContext
                .select(CAMPAIGNS_PROMOTION_MAPPER.getFieldsToRead())
                .from(CAMPAIGNS_PROMOTIONS)
                .where(CAMPAIGNS_PROMOTIONS.CID.in(campaignIds))
                .fetchGroups(CAMPAIGNS_PROMOTIONS.CID, CAMPAIGNS_PROMOTION_MAPPER::fromDb);
    }

    public Map<Long, List<CampaignsPromotion>> getCampaignsPromotionsByCid(
            int shard, Collection<Long> campaignIds) {
        return getCampaignsPromotionsByCid(dslContextProvider.ppc(shard), campaignIds);
    }

    public void updateCampaignsPromotions(DSLContext dslContext, Map<Long, List<CampaignsPromotion>> promotionsByCid) {
        Condition deleteCondition = createDeleteCondition(promotionsByCid);
        dslContext.deleteFrom(CAMPAIGNS_PROMOTIONS).where(deleteCondition).execute();

        Set<CampaignsPromotion> campaignsPromotionsForUpdate = createSetForUpdate(promotionsByCid);
        InsertHelper<CampaignsPromotionsRecord> insertHelper = new InsertHelper<>(dslContext, CAMPAIGNS_PROMOTIONS)
                .addAll(CAMPAIGNS_PROMOTION_MAPPER, campaignsPromotionsForUpdate);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CAMPAIGNS_PROMOTIONS.START, MySQLDSL.values(CAMPAIGNS_PROMOTIONS.START))
                    .set(CAMPAIGNS_PROMOTIONS.FINISH, MySQLDSL.values(CAMPAIGNS_PROMOTIONS.FINISH))
                    .set(CAMPAIGNS_PROMOTIONS.PERCENT, MySQLDSL.values(CAMPAIGNS_PROMOTIONS.PERCENT));
        }
        insertHelper.executeIfRecordsAdded();
    }

    public void updateCampaignsPromotions(int shard, Map<Long, List<CampaignsPromotion>> promotionsByCid) {
        updateCampaignsPromotions(dslContextProvider.ppc(shard), promotionsByCid);
    }

    private Condition createDeleteCondition(Map<Long, List<CampaignsPromotion>> promotionsByCid) {
        Condition deleteCondition = falseCondition();
        for (Long cid : promotionsByCid.keySet()) {
            Condition condition = CAMPAIGNS_PROMOTIONS.CID.eq(cid).and(
                    CAMPAIGNS_PROMOTIONS.PROMOTION_ID
                            .notIn(mapList(promotionsByCid.get(cid), CampaignsPromotion::getPromotionId)));
            deleteCondition = deleteCondition.or(condition);
        }
        return deleteCondition;
    }

    private Set<CampaignsPromotion> createSetForUpdate(Map<Long, List<CampaignsPromotion>> promotionsByCid) {
        Map<Boolean, List<CampaignsPromotion>> noPromotionIdMap =
                StreamEx.of(flatMapToSet(promotionsByCid.values(), Function.identity()))
                        .partitioningBy(cp -> cp.getPromotionId() == 0);
        Iterator<Long> newPromotionIds
                = shardHelper.generateCampaignsPromotionsIds(noPromotionIdMap.get(true).size()).iterator();
        return mapSet(flatMapToSet(noPromotionIdMap.values(),
                Function.identity()), cp -> cp.withPromotionId(cp.getPromotionId() != 0
                ? cp.getPromotionId()
                : newPromotionIds.next()));
    }

    private static JooqMapperWithSupplier<CampaignsPromotion> createCampaignsPromotionMapper() {
        return JooqMapperWithSupplierBuilder.builder(CampaignsPromotion::new)
                .map(property(CampaignsPromotion.PROMOTION_ID,
                        CAMPAIGNS_PROMOTIONS.PROMOTION_ID))
                .map(property(CampaignsPromotion.CID,
                        CAMPAIGNS_PROMOTIONS.CID))
                .map(property(CampaignsPromotion.START,
                        CAMPAIGNS_PROMOTIONS.START))
                .map(property(CampaignsPromotion.FINISH,
                        CAMPAIGNS_PROMOTIONS.FINISH))
                .map(property(CampaignsPromotion.PERCENT,
                        CAMPAIGNS_PROMOTIONS.PERCENT))
                .build();
    }
}
