package ru.yandex.direct.core.entity.campaign.repository;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DatePart;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.OptimizingCampaignRequest;
import ru.yandex.direct.core.entity.campaign.model.OptimizingCampaignRequestNotificationData;
import ru.yandex.direct.core.entity.campaign.model.OptimizingReqType;
import ru.yandex.direct.core.entity.campaign.model.OptimizingRequestStatus;
import ru.yandex.direct.dbschema.ppc.enums.OptimizingCampaignRequestsStatus;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.OPTIMIZING_CAMPAIGN_REQUESTS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Работа с заявками на "Первую помощь"
 * Таблица: ppc.optimizing_campaign_requests
 */
@Repository
@ParametersAreNonnullByDefault
public class OptimizingCampaignRequestRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<OptimizingCampaignRequest> mapper;

    @Autowired
    public OptimizingCampaignRequestRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = createMapper();
    }


    /**
     * Получить данные для отправки уведомления
     *
     * @param shard                         шард
     * @param optimizeNotificationIntervals список временных интервалов для которых отправляем уведомление
     * @return список данных для отправки уведомления
     */
    public List<OptimizingCampaignRequestNotificationData> getNotificationsData(int shard,
                                                                                Collection<Duration> optimizeNotificationIntervals) {
        Field<Integer> daysToGo = SqlUtils.localDateTimeDiff(DatePart.DAY,
                DSL.ifnull(OPTIMIZING_CAMPAIGN_REQUESTS.READY_TIME, OPTIMIZING_CAMPAIGN_REQUESTS.CREATE_TIME),
                LocalDateTime.now()
        );
        Set<Field<?>> fieldsToRead = getNotificationsFieldsToRead();
        fieldsToRead.add(daysToGo);

        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(OPTIMIZING_CAMPAIGN_REQUESTS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(OPTIMIZING_CAMPAIGN_REQUESTS.CID))
                .join(USERS).on(USERS.UID.eq(CAMPAIGNS.UID))
                .where(OPTIMIZING_CAMPAIGN_REQUESTS.STATUS.eq(OptimizingCampaignRequestsStatus.Ready)
                        .and(daysToGo.in(mapList(optimizeNotificationIntervals, Duration::toDays))))
                .fetch(r -> mapper.fromDb(r)
                        .withDaysToGo(r.get(daysToGo)));
    }

    private Set<Field<?>> getNotificationsFieldsToRead() {
        var modelPropertiesToRead =
                new HashSet<>(OptimizingCampaignRequestNotificationData.allModelProperties());
        modelPropertiesToRead.remove(OptimizingCampaignRequestNotificationData.DAYS_TO_GO);
        return mapper.getFieldsToRead(modelPropertiesToRead);
    }

    /**
     * Сохраняет список заявок в таблице ppc.optimizing_campaign_requests
     *
     * @param shard    шард
     * @param requests список заявок
     */
    public void addRequests(int shard, Collection<OptimizingCampaignRequest> requests) {
        if (requests.isEmpty()) {
            return;
        }
        new InsertHelper<>(dslContextProvider.ppc(shard), OPTIMIZING_CAMPAIGN_REQUESTS)
                .addAll(mapper, requests)
                .execute();
    }

    public void closeRequestsFirstAid(int shard, Collection<Long> campaignIds,
                                      OptimizingCampaignRequestsStatus status) {
        OptimizingCampaignRequestsStatus[] allowedStatuses =
                {OptimizingCampaignRequestsStatus.New, OptimizingCampaignRequestsStatus.InProcess};
        dslContextProvider.ppc(shard)
                .update(OPTIMIZING_CAMPAIGN_REQUESTS)
                .set(OPTIMIZING_CAMPAIGN_REQUESTS.ACCEPT_TIME, DSL.currentLocalDateTime())
                .set(OPTIMIZING_CAMPAIGN_REQUESTS.STATUS, status)
                .where(OPTIMIZING_CAMPAIGN_REQUESTS.STATUS.in(allowedStatuses))
                .and(OPTIMIZING_CAMPAIGN_REQUESTS.CID.in(campaignIds))
                .execute();
    }

    private JooqMapperWithSupplier<OptimizingCampaignRequest> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(OptimizingCampaignRequest::new)
                .map(property(OptimizingCampaignRequest.REQUEST_ID, OPTIMIZING_CAMPAIGN_REQUESTS.REQUEST_ID))
                .map(property(OptimizingCampaignRequest.CAMPAIGN_ID, OPTIMIZING_CAMPAIGN_REQUESTS.CID))
                .map(integerProperty(OptimizingCampaignRequest.BANNERS_COUNT,
                        OPTIMIZING_CAMPAIGN_REQUESTS.ADDED_BANNERS_COUNT))
                .map(booleanProperty(OptimizingCampaignRequest.IS_AUTOMATIC, OPTIMIZING_CAMPAIGN_REQUESTS.IS_AUTOMATIC))
                .map(booleanProperty(OptimizingCampaignRequest.IS_SUPPORT, OPTIMIZING_CAMPAIGN_REQUESTS.IS_SUPPORT))
                .map(convertibleProperty(OptimizingCampaignRequest.REQ_TYPE, OPTIMIZING_CAMPAIGN_REQUESTS.REQ_TYPE,
                        OptimizingReqType::fromSource,
                        OptimizingReqType::toSource))
                .map(convertibleProperty(OptimizingCampaignRequest.STATUS, OPTIMIZING_CAMPAIGN_REQUESTS.STATUS,
                        OptimizingRequestStatus::fromSource,
                        OptimizingRequestStatus::toSource))
                .map(property(OptimizingCampaignRequest.CREATE_TIME, OPTIMIZING_CAMPAIGN_REQUESTS.CREATE_TIME))
                .map(property(OptimizingCampaignRequest.READY_TIME, OPTIMIZING_CAMPAIGN_REQUESTS.READY_TIME))
                .map(property(OptimizingCampaignRequest.UID, USERS.UID))
                .map(property(OptimizingCampaignRequest.FIO, USERS.FIO))
                .map(property(OptimizingCampaignRequest.EMAIL, USERS.EMAIL))
                .build();
    }
}
