package ru.yandex.direct.core.entity.campaign.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.Comparator;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Index;
import org.jooq.Record5;
import org.jooq.SelectHavingStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.Wallet;
import ru.yandex.direct.core.entity.payment.model.AutopayParams;
import ru.yandex.direct.core.entity.payment.model.AutopaySettingsPaymethodType;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppc.Indexes;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsCurrency;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.ClientsWorkCurrency;
import ru.yandex.direct.dbschema.ppc.enums.WalletCampaignsAutopayMode;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static org.jooq.impl.DSL.ifnull;
import static org.jooq.impl.DSL.when;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.currencyCodeToDb;
import static ru.yandex.direct.dbschema.ppc.Tables.AUTOPAY_SETTINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.WALLET_CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Repository
public class WalletRepository {
    private static final Campaigns UNDER_WALLET_CAMPAIGNS = CAMPAIGNS.as("sub_camp");

    private final DslContextProvider dslContextProvider;

    @Autowired
    public WalletRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public List<Wallet> getAllWalletExistingCampByClientId(int shard, List<ClientId> clientIds) {
        Condition conditionFilter = CAMPAIGNS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong));
        return createAllWalletCampaignsLoader(shard, conditionFilter, Indexes.CAMPAIGNS_CLIENTID)
                .fetch(this::toWallet);
    }

    /**
     * Выдает Map - идентификатор кошелька -> включен/нет по всем кошелькам клиента clientId,
     * на которых подключен автоплатеж с payer_uid = uid
     *
     * @param shard
     * @param uid
     * @param clientId
     */
    public Map<Long, Boolean> findWalletsWithSameAutoPayPayerUid(int shard, Long uid, ClientId clientId) {
        Map<Long, WalletCampaignsAutopayMode> autopayModeByCid = dslContextProvider.ppc(shard)
                .select(Tables.CAMPAIGNS.CID, WALLET_CAMPAIGNS.AUTOPAY_MODE)
                .from(Tables.CAMPAIGNS)
                .join(WALLET_CAMPAIGNS).on(WALLET_CAMPAIGNS.WALLET_CID.eq(CAMPAIGNS.CID))
                .join(AUTOPAY_SETTINGS).on(AUTOPAY_SETTINGS.WALLET_CID.eq(WALLET_CAMPAIGNS.WALLET_CID))
                .where(Tables.CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(AUTOPAY_SETTINGS.PAYER_UID.eq(uid))
                .fetch()
                .intoMap(Tables.CAMPAIGNS.CID, WALLET_CAMPAIGNS.AUTOPAY_MODE);

        return EntryStream.of(autopayModeByCid)
                .mapValues(mode -> !mode.equals(WalletCampaignsAutopayMode.none))
                .toMap();
    }

    /**
     * Удаляет автоплатежи на указанных кошельках
     *
     * @param shard
     * @param walletCids
     */
    public void deleteInactiveAutoPays(int shard, Collection<Long> walletCids) {
        DSLContext ppc = dslContextProvider.ppc(shard);
        if (walletCids.isEmpty()) {
            return;
        }
        //Удаляем записи из autopau_settings
        ppc.deleteFrom(AUTOPAY_SETTINGS).where(AUTOPAY_SETTINGS.WALLET_CID.in(walletCids)).execute();

        //Затем однозначно проставляем в wallet_campaigns autopay_mode="none"
        ppc.update(WALLET_CAMPAIGNS)
                .set(WALLET_CAMPAIGNS.AUTOPAY_MODE, WalletCampaignsAutopayMode.none)
                .where(WALLET_CAMPAIGNS.WALLET_CID.in(walletCids)).execute();
    }

    /**
     * Получает wallet campaign id клиентского кошелька. Если кошельков оказалось несколько, то возвращает любой.
     * Получает кошелек только для терминальных клиентов !
     */
    @Nullable
    public Long getActualClientWalletId(int shard, ClientId clientid, CurrencyCode clientCurrency) {
        return getClientWalletId(shard, clientid, clientCurrency, true);
    }

    /**
     * Получает wallet campaign id клиентского кошелька. Если кошельков оказалось несколько, то возвращает любой.
     */
    @Nullable
    public Long getClientWalletId(int shard, ClientId clientid, CurrencyCode clientCurrency, Boolean withoutAgency) {

        var condition = CAMPAIGNS.CLIENT_ID.eq(clientid.asLong())
                .and(CAMPAIGNS.CURRENCY.eq(currencyCodeToDb(clientCurrency)))
                .and(CAMPAIGNS.TYPE.eq(CampaignsType.wallet))
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No));

        if (withoutAgency) {
            condition = condition
                    .and(CAMPAIGNS.AGENCY_ID.eq(ID_NOT_SET));
        }

        List<Long> walletIds = dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(condition)
                .fetch(CAMPAIGNS.CID);

        if (walletIds.isEmpty()) {
            return null;
        } else {
            return walletIds.get(0);
        }
    }

    @ParametersAreNonnullByDefault
    public void turnOnAutopay(int shard, Long walletId, Long uid, AutopayParams autopayParams) {

        var paymethodType = AutopaySettingsPaymethodType.toSource(autopayParams.getPaymentType());

        dslContextProvider.ppc(shard).transaction(config -> {
            config.dsl()
                    .insertInto(AUTOPAY_SETTINGS)
                    .set(AUTOPAY_SETTINGS.WALLET_CID, walletId)
                    .set(AUTOPAY_SETTINGS.PAYER_UID, uid)
                    .set(AUTOPAY_SETTINGS.PAYMETHOD_TYPE, paymethodType)
                    .set(AUTOPAY_SETTINGS.PAYMETHOD_ID, autopayParams.getCardId())
                    .set(AUTOPAY_SETTINGS.REMAINING_SUM, autopayParams.getRemainingSum())
                    .set(AUTOPAY_SETTINGS.PAYMENT_SUM, autopayParams.getPaymentSum())
                    .set(AUTOPAY_SETTINGS.PERSON_ID, autopayParams.getPersonId())
                    .onDuplicateKeyUpdate()
                    .set(AUTOPAY_SETTINGS.PAYMETHOD_TYPE, paymethodType)
                    .set(AUTOPAY_SETTINGS.PAYMETHOD_ID, autopayParams.getCardId())
                    .set(AUTOPAY_SETTINGS.REMAINING_SUM, autopayParams.getRemainingSum())
                    .set(AUTOPAY_SETTINGS.PAYMENT_SUM, autopayParams.getPaymentSum())
                    .set(AUTOPAY_SETTINGS.PERSON_ID, autopayParams.getPersonId())
                    .execute();

            config.dsl()
                    .update(WALLET_CAMPAIGNS)
                    .set(WALLET_CAMPAIGNS.AUTOPAY_MODE, WalletCampaignsAutopayMode.min_balance)
                    .where(WALLET_CAMPAIGNS.WALLET_CID.eq(walletId))
                    .execute();
        });
    }

    /**
     * Получить кошелек для создания новой кампании c валютой клиента
     */
    @Nullable
    public Wallet getWalletForCampaigns(
            int shard,
            Long chiefUid,
            @Nullable List<Long> agencyRepresentativeUids) {
        Condition condition = CAMPAIGNS.UID.eq(chiefUid)
                .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No))
                .and(CAMPAIGNS.CURRENCY.cast(String.class).eq(ifnull(CLIENTS.WORK_CURRENCY,
                        ClientsWorkCurrency.YND_FIXED).cast(String.class)));
        if (agencyRepresentativeUids != null) {
            condition = condition.and(CAMPAIGNS.AGENCY_UID.in(agencyRepresentativeUids))
                    .and(CAMPAIGNS.MANAGER_UID.isNull());
        } else {
            condition = condition.and(CAMPAIGNS.AGENCY_UID.isNull());
        }

        return createAllWalletCampaignsLoader(shard, condition, Indexes.CAMPAIGNS_I_UID)
                .fetchOne(this::toWallet);
    }

    public LocalDateTime getAutopayLastChange(int shard, Long walletId) {
        return dslContextProvider.ppc(shard)
                .select(AUTOPAY_SETTINGS.LAST_CHANGE)
                .from(AUTOPAY_SETTINGS)
                .where(AUTOPAY_SETTINGS.WALLET_CID.eq(walletId))
                .fetchOne(AUTOPAY_SETTINGS.LAST_CHANGE);
    }

    private Wallet toWallet(Record5<Long, Long, Long, Boolean, CampaignsCurrency> record5) {
        return new Wallet(
                record5.value1(),
                record5.value2(),
                ClientId.fromLong(record5.value3()),
                record5.value4(),
                Currencies.getCurrency(record5.value5().getLiteral())
        );
    }

    private SelectHavingStep<Record5<Long, Long, Long, Boolean, CampaignsCurrency>>
    createAllWalletCampaignsLoader(int shard, Condition condition, Index campaignForceIndex) {
        return dslContextProvider.ppc(shard)
                .select(
                        CAMPAIGNS.AGENCY_ID,
                        CAMPAIGNS.CID,
                        CAMPAIGNS.CLIENT_ID,
                        fldIsEnabled(),
                        fldCurrency()
                )
                .from(CAMPAIGNS.forceIndex(campaignForceIndex.getName()))
                .leftOuterJoin(UNDER_WALLET_CAMPAIGNS.forceIndex(Indexes.CAMPAIGNS_WALLET_CID.getName())).on(
                        UNDER_WALLET_CAMPAIGNS.UID.eq(CAMPAIGNS.UID)
                                .and(UNDER_WALLET_CAMPAIGNS.WALLET_CID.eq(CAMPAIGNS.CID))
                                .and(UNDER_WALLET_CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .join(CLIENTS).on(CAMPAIGNS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(CAMPAIGNS.TYPE.eq(CampaignsType.wallet).and(condition))
                .groupBy(CAMPAIGNS.AGENCY_ID, CAMPAIGNS.CID, CAMPAIGNS.CURRENCY, CAMPAIGNS.CLIENT_ID, CAMPAIGNS.UID);
    }

    private Field<Boolean> fldIsEnabled() {
        return when(DSL.count(UNDER_WALLET_CAMPAIGNS.CID).compare(Comparator.GREATER, 0), true)
                .otherwise(false);
    }

    private Field<CampaignsCurrency> fldCurrency() {
        return CAMPAIGNS.CURRENCY.nvl(CampaignsCurrency.YND_FIXED);
    }


    /**
     * Возвращает true, если автоплатеж включен
     *
     * @param shard     - шард
     * @param walletCid - id wallet кампании
     * @return - включен ли автоплатеж
     */
    public Boolean isAutopayEnabled(int shard, Long walletCid) {
        return dslContextProvider.ppc(shard).fetchExists(WALLET_CAMPAIGNS,
                WALLET_CAMPAIGNS.WALLET_CID.eq(walletCid)
                        .and(WALLET_CAMPAIGNS.AUTOPAY_MODE.notEqual(WalletCampaignsAutopayMode.none)));
    }
}
