package ru.yandex.direct.core.entity.campaign.repository;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.InsertValuesStep2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.WhenMoneyOnCampaignWas;
import ru.yandex.direct.dbschema.ppc.tables.records.WhenMoneyOnCampWasRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;

import static ru.yandex.direct.dbschema.ppc.tables.WhenMoneyOnCampWas.WHEN_MONEY_ON_CAMP_WAS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;


@Repository
@ParametersAreNonnullByDefault
public class WhenMoneyOnCampWasRepository {

    /* В условиях интервал считается открытым, если дата его окончания не меньше, чем
     END_OF_TIME минус END_OF_TIME_GAP
     Причины исторические — https://st.yandex-team.ru/DIRECT-69540#1503571337000 */
    private static final LocalDateTime END_OF_TIME = LocalDateTime.parse("2038-01-19T00:00:00");
    private static final Duration END_OF_TIME_GAP = Duration.ofDays(2);
    private static final LocalDateTime END_OF_TIME_FOR_CONDITIONS = END_OF_TIME.minus(END_OF_TIME_GAP);

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<WhenMoneyOnCampaignWas> mapper;

    @Autowired
    WhenMoneyOnCampWasRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        mapper = JooqMapperWithSupplierBuilder.builder(WhenMoneyOnCampaignWas::new)
                .map(property(WhenMoneyOnCampaignWas.CAMPAIGN_ID, WHEN_MONEY_ON_CAMP_WAS.CID))
                .map(property(WhenMoneyOnCampaignWas.INTERVAL_START, WHEN_MONEY_ON_CAMP_WAS.INTERVAL_START))
                .map(property(WhenMoneyOnCampaignWas.INTERVAL_END, WHEN_MONEY_ON_CAMP_WAS.INTERVAL_END))
                .build();
    }

    /**
     * Получить когда на кампании были деньги для указанного списка кампаний.
     *
     * @param shard       - шард
     * @param campaignIds - список идентификаторов кампаний
     * @return список с данными когда на кампании были деньги
     */
    public List<WhenMoneyOnCampaignWas> getWhenMoneyOnCampaignsWas(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(WHEN_MONEY_ON_CAMP_WAS)
                .where(WHEN_MONEY_ON_CAMP_WAS.CID.in(campaignIds))
                .fetch(mapper::fromDb);
    }

    public Set<Long> getCidsWithOpenedInterval(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(WHEN_MONEY_ON_CAMP_WAS.CID)
                .from(WHEN_MONEY_ON_CAMP_WAS)
                .where(WHEN_MONEY_ON_CAMP_WAS.CID.in(campaignIds)
                        .and(WHEN_MONEY_ON_CAMP_WAS.INTERVAL_END.ge(END_OF_TIME_FOR_CONDITIONS)))
                .fetchSet(WHEN_MONEY_ON_CAMP_WAS.CID);
    }

    public void openInterval(int shard, Long campaignId) {
        openInterval(shard, Collections.singleton(campaignId));
    }

    public void openInterval(int shard, Collection<Long> campaignIds) {
        InsertValuesStep2<WhenMoneyOnCampWasRecord, Long, LocalDateTime> valuesStep =
                dslContextProvider.ppc(shard)
                        .insertInto(WHEN_MONEY_ON_CAMP_WAS)
                        .columns(WHEN_MONEY_ON_CAMP_WAS.CID,
                                WHEN_MONEY_ON_CAMP_WAS.INTERVAL_END);
        for (Long campaignId : campaignIds) {
            valuesStep = valuesStep.values(campaignId, END_OF_TIME);
        }
        valuesStep.onDuplicateKeyIgnore()
                .execute();
    }

    public void closeInterval(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(WHEN_MONEY_ON_CAMP_WAS)
                .set(WHEN_MONEY_ON_CAMP_WAS.INTERVAL_END, LocalDateTime.now())
                .where(WHEN_MONEY_ON_CAMP_WAS.CID.in(campaignIds)
                        .and(WHEN_MONEY_ON_CAMP_WAS.INTERVAL_END.ge(END_OF_TIME_FOR_CONDITIONS)))
                .execute();
    }

    /* метод используется для тестирования */
    public void addInterval(int shard, Long cid, LocalDateTime intervalStart, LocalDateTime intervalEnd) {
        dslContextProvider.ppc(shard)
                .insertInto(WHEN_MONEY_ON_CAMP_WAS)
                .columns(WHEN_MONEY_ON_CAMP_WAS.CID,
                        WHEN_MONEY_ON_CAMP_WAS.INTERVAL_START,
                        WHEN_MONEY_ON_CAMP_WAS.INTERVAL_END)
                .values(cid, intervalStart, intervalEnd)
                .onDuplicateKeyIgnore()
                .execute();
    }
}
