package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithAdditionalData;
import ru.yandex.direct.core.entity.campaign.service.type.update.container.RestrictedCampaignsUpdateOperationContainer;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;
import ru.yandex.direct.multitype.typesupport.TypeSupportUtils;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_ADDITIONAL_DATA;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;

/**
 * Специальный type support для таблицы {@code camp_additional_data}.
 * Нужен для того, чтобы вставлять запись в таблицу если хотя бы одно поле непустое и удалять, если все поля пустые.
 * <p>
 * Логика работы с полями находится в наследниках {@link AbstractCampaignWithAdditionalDataRepositorySupport}
 * <p>
 * Эта таблица могла вести себя как {@code camp_options} (вставлять запись при создании кампании и удалять при удалении).
 * Но тогда это была бы еще одна таблица, в которую надо вставлять при создании кампании (в том числе из перла).
 * Так-же, пришлось бы создать запись для всех существующих кампаний.
 */
@Component
@ParametersAreNonnullByDefault
@SuppressWarnings("unchecked")
public class CampaignAdditionalDataTableSupport extends
        AbstractCampaignRepositoryTypeSupport<CampaignWithAdditionalData> {

    private final List<AbstractCampaignWithAdditionalDataRepositorySupport> additionalDataSupports;

    @Autowired
    public CampaignAdditionalDataTableSupport(
            List<AbstractCampaignWithAdditionalDataRepositorySupport> additionalDataSupports,
            DslContextProvider dslContextProvider
    ) {
        super(dslContextProvider);
        this.additionalDataSupports = additionalDataSupports;
    }

    @Override
    public Class<CampaignWithAdditionalData> getTypeClass() {
        return CampaignWithAdditionalData.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return new HashSet<>();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(CAMP_ADDITIONAL_DATA,
                JoinType.LEFT_OUTER_JOIN,
                CAMP_ADDITIONAL_DATA.CID.eq(CAMPAIGNS.CID)));
    }

    @Override
    public void fillFromRecord(CampaignWithAdditionalData campaign, Record record) {
    }

    @Override
    public void updateAdditionTables(DSLContext context, RestrictedCampaignsUpdateOperationContainer updateParameters,
                                     Collection<AppliedChanges<CampaignWithAdditionalData>> appliedChanges) {
        List<CampaignWithAdditionalData> modelsToInsert = filterAndMapList(appliedChanges,
                this::doInsertIntoCampAdditionalData, AppliedChanges::getModel);
        insertIntoCampAdditionalData(context, modelsToInsert);

        List<CampaignWithAdditionalData> modelsToDelete = filterAndMapList(appliedChanges,
                this::doDeleteFromCampAdditionalData, AppliedChanges::getModel);
        List<Long> idsToDelete = filterAndMapList(modelsToDelete, Objects::nonNull, CampaignWithAdditionalData::getId);
        deleteCampAdditionalData(context, idsToDelete);
    }

    private boolean doInsertIntoCampAdditionalData(AppliedChanges<CampaignWithAdditionalData> appliedChanges) {
        var typeSupports
                = TypeSupportUtils.getSupportsByClass(additionalDataSupports, appliedChanges.getModel().getClass());

        boolean allPropsWereAbsentBefore = typeSupports.stream()
                .allMatch(t -> t.wasNotPresentBefore(appliedChanges));
        boolean anyPropAdded = typeSupports.stream()
                .anyMatch(t -> t.isCampAdditionalDataAdded(appliedChanges));

        return allPropsWereAbsentBefore && anyPropAdded;
    }

    private void insertIntoCampAdditionalData(DSLContext context, List<CampaignWithAdditionalData> modelsToInsert) {
        if (modelsToInsert.isEmpty()) {
            return;
        }

        InsertHelperAggregator insertHelperAggregator = new InsertHelperAggregator(context);

        modelsToInsert.forEach(model -> {
            var supports
                    = TypeSupportUtils.getSupportsByClass(additionalDataSupports, model.getClass());
            supports.forEach(t -> t.pushToInsert(insertHelperAggregator, model));
            insertHelperAggregator.newRecord();
        });

        insertHelperAggregator.executeIfRecordsAdded();
    }

    private boolean doDeleteFromCampAdditionalData(
            AppliedChanges<CampaignWithAdditionalData> appliedChanges) {
        var typeSupports =
                TypeSupportUtils.getSupportsByClass(additionalDataSupports, appliedChanges.getModel().getClass());

        boolean isAnyPropDeletedNow = typeSupports.stream()
                .anyMatch(t -> t.isCampAdditionalDataChangedAndDeleted(appliedChanges));
        boolean areAllPropsAbsentAfter = typeSupports.stream()
                .allMatch(t -> t.isMissingAfter(appliedChanges));

        return isAnyPropDeletedNow && areAllPropsAbsentAfter;
    }

    private void deleteCampAdditionalData(DSLContext context, List<Long> idsToDelete) {
        if (!idsToDelete.isEmpty()) {
            context.deleteFrom(CAMP_ADDITIONAL_DATA).where(CAMP_ADDITIONAL_DATA.CID.in(idsToDelete)).execute();
        }
    }
}
