package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.converter.ExperimentsConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandLift;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.dbschema.ppc.tables.Campaigns;
import ru.yandex.direct.dbschema.ppc.tables.RetargetingConditions;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBrandLiftTypeSupport extends AbstractCampaignRepositoryTypeSupport<CampaignWithBrandLift> {

    private static final RetargetingConditions RETARGETING_CONDITIONS_SECTIONS =
            RETARGETING_CONDITIONS.as("retageting_condition_sections");
    private static final RetargetingConditions RETARGETING_CONDITIONS_AB_GOALS =
            RETARGETING_CONDITIONS.as("retageting_condition_abgoals");

    private static final JooqMapper<CampaignWithBrandLift> MAPPER = createMapper();

    @Autowired
    public CampaignWithBrandLiftTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CampaignWithBrandLift> getTypeClass() {
        return CampaignWithBrandLift.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(
                new JoinQuery(
                    RETARGETING_CONDITIONS_SECTIONS,
                    JoinType.LEFT_OUTER_JOIN,
                    CAMPAIGNS.AB_SEGMENT_STAT_RET_COND_ID.eq(RETARGETING_CONDITIONS_SECTIONS.RET_COND_ID)
                ),
                new JoinQuery(
                    RETARGETING_CONDITIONS_AB_GOALS,
                    JoinType.LEFT_OUTER_JOIN,
                    CAMPAIGNS.AB_SEGMENT_RET_COND_ID.eq(RETARGETING_CONDITIONS_AB_GOALS.RET_COND_ID)
                )
        );
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithBrandLift>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMP_OPTIONS.CID).processUpdateAll(MAPPER, appliedChanges);
        updateHelperAggregator.getOrCreate(CAMPAIGNS.CID).processUpdateAll(MAPPER, appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CampaignWithBrandLift campaign) {
        insertHelperAggregator.getOrCreate(CAMP_OPTIONS).add(MAPPER, campaign);
        insertHelperAggregator.getOrCreate(CAMPAIGNS).add(MAPPER, campaign);
    }

    @Override
    public void fillFromRecord(CampaignWithBrandLift campaign, Record record) {
        MAPPER.fromDb(record, campaign);
    }

    private static JooqMapper<CampaignWithBrandLift> createMapper() {
        return JooqMapperBuilder.<CampaignWithBrandLift>builder()
                .map(property(CampaignWithBrandLift.BRAND_SURVEY_ID, CAMP_OPTIONS.BRAND_SURVEY_ID))
                .map(property(CampaignWithBrandLift.AB_SEGMENT_RETARGETING_CONDITION_ID,
                        CAMPAIGNS.AB_SEGMENT_RET_COND_ID))
                .map(property(CampaignWithBrandLift.AB_SEGMENT_STATISTIC_RETARGETING_CONDITION_ID,
                        CAMPAIGNS.AB_SEGMENT_STAT_RET_COND_ID))
                .map(convertibleProperty(CampaignWithBrandLift.IS_CPM_GLOBAL_AB_SEGMENT,
                        CAMP_OPTIONS.IS_CPM_GLOBAL_AB_SEGMENT,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLong))
                .map(convertibleProperty(CampaignWithBrandLift.IS_SEARCH_LIFT_ENABLED,
                        CAMP_OPTIONS.IS_SEARCH_LIFT_ENABLED,
                        RepositoryUtils::booleanFromLong, RepositoryUtils::nullSafeBooleanToLong))
                .readProperty(
                        CampaignWithBrandLift.SECTION_IDS,
                        fromField(RETARGETING_CONDITIONS_SECTIONS.CONDITION_JSON)
                                .by(ExperimentsConverter::sectionIdFromDb))
                .readProperty(
                        CampaignWithBrandLift.AB_SEGMENT_GOAL_IDS,
                        fromField(RETARGETING_CONDITIONS_AB_GOALS.CONDITION_JSON)
                                .by(ExperimentsConverter::abSegmentGoalsFromDb))
                .readProperty(CommonCampaign.IS_BRAND_LIFT_HIDDEN, fromField(Campaigns.CAMPAIGNS.OPTS).by(
                        CampaignConverter::isBrandLiftHiddenFromDb))
                .build();
    }
}
