package ru.yandex.direct.core.entity.campaign.repository.type;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandSafety;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.Rule;
import ru.yandex.direct.core.entity.retargeting.model.RuleType;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelperAggregator;
import ru.yandex.direct.jooqmapperhelper.UpdateHelperAggregator;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Component
@ParametersAreNonnullByDefault
public class CampaignWithBrandSafetyTypeSupport extends
        AbstractCampaignRepositoryTypeSupport<CampaignWithBrandSafety> {

    private static final JooqMapper<CampaignWithBrandSafety> READ_MAPPER = createReadMapper();
    private static final JooqMapper<CampaignWithBrandSafety> SAVE_MAPPER = createSaveMapper();

    @Autowired
    public CampaignWithBrandSafetyTypeSupport(DslContextProvider dslContextProvider) {
        super(dslContextProvider);
    }

    @Override
    public Class<CampaignWithBrandSafety> getTypeClass() {
        return CampaignWithBrandSafety.class;
    }

    @Override
    public Collection<Field<?>> getFields() {
        return READ_MAPPER.getFieldsToRead();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(RETARGETING_CONDITIONS,
                JoinType.LEFT_OUTER_JOIN,
                CAMPAIGNS.BRANDSAFETY_RET_COND_ID.eq(RETARGETING_CONDITIONS.RET_COND_ID)));
    }

    @Override
    public void processUpdate(UpdateHelperAggregator updateHelperAggregator,
                              Collection<AppliedChanges<CampaignWithBrandSafety>> appliedChanges) {
        updateHelperAggregator.getOrCreate(CAMPAIGNS.CID).processUpdateAll(SAVE_MAPPER, appliedChanges);
    }

    @Override
    public void pushToInsert(InsertHelperAggregator insertHelperAggregator, CampaignWithBrandSafety campaign) {
        insertHelperAggregator.getOrCreate(CAMPAIGNS).add(SAVE_MAPPER, campaign);
    }

    @Override
    public void fillFromRecord(CampaignWithBrandSafety campaign, Record record) {
        READ_MAPPER.fromDb(record, campaign);
    }

    private static JooqMapper<CampaignWithBrandSafety> createReadMapper() {
        return JooqMapperBuilder.<CampaignWithBrandSafety>builder()
                .map(property(CampaignWithBrandSafety.ID, CAMPAIGNS.CID))
                .map(property(CampaignWithBrandSafety.BRAND_SAFETY_RET_COND_ID, CAMPAIGNS.BRANDSAFETY_RET_COND_ID))
                .map(convertibleProperty(
                        CampaignWithBrandSafety.BRAND_SAFETY_CATEGORIES,
                        RETARGETING_CONDITIONS.CONDITION_JSON,
                        CampaignWithBrandSafetyTypeSupport::retargetingConditionJsonFromDb,
                        CampaignWithBrandSafetyTypeSupport::retargetingConditionJsonToDb))
                .build();
    }

    private static JooqMapper<CampaignWithBrandSafety> createSaveMapper() {
        return JooqMapperBuilder.<CampaignWithBrandSafety>builder()
                .map(property(CampaignWithBrandSafety.ID, CAMPAIGNS.CID))
                .map(property(CampaignWithBrandSafety.BRAND_SAFETY_RET_COND_ID, CAMPAIGNS.BRANDSAFETY_RET_COND_ID))
                .build();
    }

    private static List<Long> retargetingConditionJsonFromDb(@Nullable String record) {
        return record == null ? emptyList() : RetargetingConditionMappings
                .rulesFromJson(record)
                .get(0)
                .getGoals()
                .stream()
                .map(Goal::getId)
                .collect(Collectors.toList());
    }

    private static String retargetingConditionJsonToDb(List<Long> categories) {
        List<Goal> goals = categories
                .stream()
                .map(category -> (Goal) new Goal()
                        .withId(category)
                        .withType(GoalType.BRANDSAFETY)
                        .withTime(0))
                .collect(Collectors.toList());
        Rule rule = new Rule()
                .withType(RuleType.NOT)
                .withGoals(goals);
        return RetargetingConditionMappings.rulesToJson(singletonList(rule));
    }
}
